/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.types;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public enum BackupPhase {
    /**
     * In this phase, the backup is invoking 'stop_backup' operation in order to complete
     * the backup and unlock the relevant disk.
     */
    FINALIZING("finalizing"),
    /**
     * The initial phase of the backup. It is set on entity creation.
     */
    INITIALIZING("initializing"),
    /**
     * The phase means that the relevant disks' backup URLs are ready to be used and downloaded
     * using image transfer.
     */
    READY("ready"),
    /**
     * The phase is set before invoking 'start_backup' operation in vdsm/libvirt
     * (which means that 'stop_backup' should be invoked to complete the flow).
     */
    STARTING("starting"),
    ;
    
    private static final Logger log = LoggerFactory.getLogger(BackupPhase.class);
    
    private String image;
    
    BackupPhase(String image) {
        this.image = image;
    }
    
    public String value() {
        return image;
    }
    
    public static BackupPhase fromValue(String value) {
        try {
            return valueOf(value.toUpperCase());
        }
        catch (IllegalArgumentException exception) {
            log.error(
            "The string '" + value + "' isn't a valid value for the 'BackupPhase' enumerated type. " +
            "Valid values are 'finalizing', 'initializing', 'ready' and 'starting'.",
            exception
            );
            return null;
        }
    }
    
}

