//
// Copyright (c) 2017 Joey <majunjiev@gmail.com>.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
package ovirtsdk

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/http/httputil"
	"net/url"
	"strings"
)

//
// Provides a way for clients to list available files.
// This service is specifically targeted to ISO storage domains, which contain ISO images and virtual floppy disks
// (VFDs) that an administrator uploads.
// The addition of a CD-ROM device to a virtual machine requires an ISO image from the files of an ISO storage domain.
//
type FilesService struct {
	BaseService
}

func NewFilesService(connection *Connection, path string) *FilesService {
	var result FilesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of ISO images and virtual floppy disks available in the storage domain. The order of
// the returned list is not guaranteed.
// If the `refresh` parameter is `false`, the returned list may not reflect recent changes to the storage domain;
// for example, it may not contain a new ISO file that was recently added. This is because the
// server caches the list of files to improve performance. To get the very latest results, set the `refresh`
// parameter to `true`.
// The default value of the `refresh` parameter is `true`, but it can be changed using the configuration value
// `ForceRefreshDomainFilesByDefault`:
// [source]
// ----
// # engine-config -s ForceRefreshDomainFilesByDefault=false
// ----
// IMPORTANT: Setting the value of the `refresh` parameter to `true` has an impact on the performance of the
// server. Use it only if necessary.
//
type FilesServiceListRequest struct {
	FilesService  *FilesService
	header        map[string]string
	query         map[string]string
	caseSensitive *bool
	follow        *string
	max           *int64
	refresh       *bool
	search        *string
}

func (p *FilesServiceListRequest) Header(key, value string) *FilesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FilesServiceListRequest) Query(key, value string) *FilesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FilesServiceListRequest) CaseSensitive(caseSensitive bool) *FilesServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *FilesServiceListRequest) Follow(follow string) *FilesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *FilesServiceListRequest) Max(max int64) *FilesServiceListRequest {
	p.max = &max
	return p
}

func (p *FilesServiceListRequest) Refresh(refresh bool) *FilesServiceListRequest {
	p.refresh = &refresh
	return p
}

func (p *FilesServiceListRequest) Search(search string) *FilesServiceListRequest {
	p.search = &search
	return p
}

func (p *FilesServiceListRequest) Send() (*FilesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FilesService.connection.URL(), p.FilesService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.refresh != nil {
		values["refresh"] = []string{fmt.Sprintf("%v", *p.refresh)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLFileReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &FilesServiceListResponse{file: result}, nil
}

func (p *FilesServiceListRequest) MustSend() *FilesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of ISO images and virtual floppy disks available in the storage domain. The order of
// the returned list is not guaranteed.
// If the `refresh` parameter is `false`, the returned list may not reflect recent changes to the storage domain;
// for example, it may not contain a new ISO file that was recently added. This is because the
// server caches the list of files to improve performance. To get the very latest results, set the `refresh`
// parameter to `true`.
// The default value of the `refresh` parameter is `true`, but it can be changed using the configuration value
// `ForceRefreshDomainFilesByDefault`:
// [source]
// ----
// # engine-config -s ForceRefreshDomainFilesByDefault=false
// ----
// IMPORTANT: Setting the value of the `refresh` parameter to `true` has an impact on the performance of the
// server. Use it only if necessary.
//
type FilesServiceListResponse struct {
	file *FileSlice
}

func (p *FilesServiceListResponse) File() (*FileSlice, bool) {
	if p.file != nil {
		return p.file, true
	}
	return nil, false
}

func (p *FilesServiceListResponse) MustFile() *FileSlice {
	if p.file == nil {
		panic("file in response does not exist")
	}
	return p.file
}

//
// Returns the list of ISO images and virtual floppy disks available in the storage domain. The order of
// the returned list is not guaranteed.
// If the `refresh` parameter is `false`, the returned list may not reflect recent changes to the storage domain;
// for example, it may not contain a new ISO file that was recently added. This is because the
// server caches the list of files to improve performance. To get the very latest results, set the `refresh`
// parameter to `true`.
// The default value of the `refresh` parameter is `true`, but it can be changed using the configuration value
// `ForceRefreshDomainFilesByDefault`:
// [source]
// ----
// # engine-config -s ForceRefreshDomainFilesByDefault=false
// ----
// IMPORTANT: Setting the value of the `refresh` parameter to `true` has an impact on the performance of the
// server. Use it only if necessary.
//
func (p *FilesService) List() *FilesServiceListRequest {
	return &FilesServiceListRequest{FilesService: p}
}

//
//
func (op *FilesService) FileService(id string) *FileService {
	return NewFileService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FilesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.FileService(path), nil
	}
	return op.FileService(path[:index]).Service(path[index+1:])
}

func (op *FilesService) String() string {
	return fmt.Sprintf("FilesService:%s", op.path)
}

//
// This service manages the set of disks attached to a template. Each attached disk is represented by a
// <<types/disk_attachment,DiskAttachment>>.
//
type TemplateDiskAttachmentsService struct {
	BaseService
}

func NewTemplateDiskAttachmentsService(connection *Connection, path string) *TemplateDiskAttachmentsService {
	var result TemplateDiskAttachmentsService
	result.connection = connection
	result.path = path
	return &result
}

//
// List the disks that are attached to the template.
// The order of the returned list of attachments isn't guaranteed.
//
type TemplateDiskAttachmentsServiceListRequest struct {
	TemplateDiskAttachmentsService *TemplateDiskAttachmentsService
	header                         map[string]string
	query                          map[string]string
	follow                         *string
}

func (p *TemplateDiskAttachmentsServiceListRequest) Header(key, value string) *TemplateDiskAttachmentsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TemplateDiskAttachmentsServiceListRequest) Query(key, value string) *TemplateDiskAttachmentsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TemplateDiskAttachmentsServiceListRequest) Follow(follow string) *TemplateDiskAttachmentsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *TemplateDiskAttachmentsServiceListRequest) Send() (*TemplateDiskAttachmentsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TemplateDiskAttachmentsService.connection.URL(), p.TemplateDiskAttachmentsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TemplateDiskAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TemplateDiskAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TemplateDiskAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TemplateDiskAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TemplateDiskAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskAttachmentReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &TemplateDiskAttachmentsServiceListResponse{attachments: result}, nil
}

func (p *TemplateDiskAttachmentsServiceListRequest) MustSend() *TemplateDiskAttachmentsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List the disks that are attached to the template.
// The order of the returned list of attachments isn't guaranteed.
//
type TemplateDiskAttachmentsServiceListResponse struct {
	attachments *DiskAttachmentSlice
}

func (p *TemplateDiskAttachmentsServiceListResponse) Attachments() (*DiskAttachmentSlice, bool) {
	if p.attachments != nil {
		return p.attachments, true
	}
	return nil, false
}

func (p *TemplateDiskAttachmentsServiceListResponse) MustAttachments() *DiskAttachmentSlice {
	if p.attachments == nil {
		panic("attachments in response does not exist")
	}
	return p.attachments
}

//
// List the disks that are attached to the template.
// The order of the returned list of attachments isn't guaranteed.
//
func (p *TemplateDiskAttachmentsService) List() *TemplateDiskAttachmentsServiceListRequest {
	return &TemplateDiskAttachmentsServiceListRequest{TemplateDiskAttachmentsService: p}
}

//
// Reference to the service that manages a specific attachment.
//
func (op *TemplateDiskAttachmentsService) AttachmentService(id string) *TemplateDiskAttachmentService {
	return NewTemplateDiskAttachmentService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *TemplateDiskAttachmentsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.AttachmentService(path), nil
	}
	return op.AttachmentService(path[:index]).Service(path[index+1:])
}

func (op *TemplateDiskAttachmentsService) String() string {
	return fmt.Sprintf("TemplateDiskAttachmentsService:%s", op.path)
}

//
//
type TemplateNicService struct {
	BaseService
}

func NewTemplateNicService(connection *Connection, path string) *TemplateNicService {
	var result TemplateNicService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type TemplateNicServiceGetRequest struct {
	TemplateNicService *TemplateNicService
	header             map[string]string
	query              map[string]string
	follow             *string
}

func (p *TemplateNicServiceGetRequest) Header(key, value string) *TemplateNicServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TemplateNicServiceGetRequest) Query(key, value string) *TemplateNicServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TemplateNicServiceGetRequest) Follow(follow string) *TemplateNicServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *TemplateNicServiceGetRequest) Send() (*TemplateNicServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TemplateNicService.connection.URL(), p.TemplateNicService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TemplateNicService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TemplateNicService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TemplateNicService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TemplateNicService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TemplateNicService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNicReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &TemplateNicServiceGetResponse{nic: result}, nil
}

func (p *TemplateNicServiceGetRequest) MustSend() *TemplateNicServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type TemplateNicServiceGetResponse struct {
	nic *Nic
}

func (p *TemplateNicServiceGetResponse) Nic() (*Nic, bool) {
	if p.nic != nil {
		return p.nic, true
	}
	return nil, false
}

func (p *TemplateNicServiceGetResponse) MustNic() *Nic {
	if p.nic == nil {
		panic("nic in response does not exist")
	}
	return p.nic
}

//
//
func (p *TemplateNicService) Get() *TemplateNicServiceGetRequest {
	return &TemplateNicServiceGetRequest{TemplateNicService: p}
}

//
//
type TemplateNicServiceRemoveRequest struct {
	TemplateNicService *TemplateNicService
	header             map[string]string
	query              map[string]string
	async              *bool
}

func (p *TemplateNicServiceRemoveRequest) Header(key, value string) *TemplateNicServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TemplateNicServiceRemoveRequest) Query(key, value string) *TemplateNicServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TemplateNicServiceRemoveRequest) Async(async bool) *TemplateNicServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *TemplateNicServiceRemoveRequest) Send() (*TemplateNicServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TemplateNicService.connection.URL(), p.TemplateNicService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TemplateNicService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TemplateNicService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TemplateNicService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TemplateNicService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TemplateNicService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(TemplateNicServiceRemoveResponse), nil
}

func (p *TemplateNicServiceRemoveRequest) MustSend() *TemplateNicServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type TemplateNicServiceRemoveResponse struct {
}

//
//
func (p *TemplateNicService) Remove() *TemplateNicServiceRemoveRequest {
	return &TemplateNicServiceRemoveRequest{TemplateNicService: p}
}

//
// Update the specified network interface card attached to the template.
//
type TemplateNicServiceUpdateRequest struct {
	TemplateNicService *TemplateNicService
	header             map[string]string
	query              map[string]string
	async              *bool
	nic                *Nic
}

func (p *TemplateNicServiceUpdateRequest) Header(key, value string) *TemplateNicServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TemplateNicServiceUpdateRequest) Query(key, value string) *TemplateNicServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TemplateNicServiceUpdateRequest) Async(async bool) *TemplateNicServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *TemplateNicServiceUpdateRequest) Nic(nic *Nic) *TemplateNicServiceUpdateRequest {
	p.nic = nic
	return p
}

func (p *TemplateNicServiceUpdateRequest) Send() (*TemplateNicServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TemplateNicService.connection.URL(), p.TemplateNicService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNicWriteOne(writer, p.nic, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TemplateNicService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TemplateNicService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TemplateNicService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TemplateNicService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TemplateNicService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNicReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &TemplateNicServiceUpdateResponse{nic: result}, nil
}

func (p *TemplateNicServiceUpdateRequest) MustSend() *TemplateNicServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified network interface card attached to the template.
//
type TemplateNicServiceUpdateResponse struct {
	nic *Nic
}

func (p *TemplateNicServiceUpdateResponse) Nic() (*Nic, bool) {
	if p.nic != nil {
		return p.nic, true
	}
	return nil, false
}

func (p *TemplateNicServiceUpdateResponse) MustNic() *Nic {
	if p.nic == nil {
		panic("nic in response does not exist")
	}
	return p.nic
}

//
// Update the specified network interface card attached to the template.
//
func (p *TemplateNicService) Update() *TemplateNicServiceUpdateRequest {
	return &TemplateNicServiceUpdateRequest{TemplateNicService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *TemplateNicService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *TemplateNicService) String() string {
	return fmt.Sprintf("TemplateNicService:%s", op.path)
}

//
//
type NetworkLabelService struct {
	BaseService
}

func NewNetworkLabelService(connection *Connection, path string) *NetworkLabelService {
	var result NetworkLabelService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type NetworkLabelServiceGetRequest struct {
	NetworkLabelService *NetworkLabelService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *NetworkLabelServiceGetRequest) Header(key, value string) *NetworkLabelServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkLabelServiceGetRequest) Query(key, value string) *NetworkLabelServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkLabelServiceGetRequest) Follow(follow string) *NetworkLabelServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *NetworkLabelServiceGetRequest) Send() (*NetworkLabelServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkLabelService.connection.URL(), p.NetworkLabelService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &NetworkLabelServiceGetResponse{label: result}, nil
}

func (p *NetworkLabelServiceGetRequest) MustSend() *NetworkLabelServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type NetworkLabelServiceGetResponse struct {
	label *NetworkLabel
}

func (p *NetworkLabelServiceGetResponse) Label() (*NetworkLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *NetworkLabelServiceGetResponse) MustLabel() *NetworkLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
//
func (p *NetworkLabelService) Get() *NetworkLabelServiceGetRequest {
	return &NetworkLabelServiceGetRequest{NetworkLabelService: p}
}

//
// Removes a label from a logical network.
// For example, to remove the label `exemplary` from a logical network having id `123` send the following request:
// [source]
// ----
// DELETE /ovirt-engine/api/networks/123/networklabels/exemplary
// ----
//
type NetworkLabelServiceRemoveRequest struct {
	NetworkLabelService *NetworkLabelService
	header              map[string]string
	query               map[string]string
	async               *bool
}

func (p *NetworkLabelServiceRemoveRequest) Header(key, value string) *NetworkLabelServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkLabelServiceRemoveRequest) Query(key, value string) *NetworkLabelServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkLabelServiceRemoveRequest) Async(async bool) *NetworkLabelServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *NetworkLabelServiceRemoveRequest) Send() (*NetworkLabelServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkLabelService.connection.URL(), p.NetworkLabelService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(NetworkLabelServiceRemoveResponse), nil
}

func (p *NetworkLabelServiceRemoveRequest) MustSend() *NetworkLabelServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a label from a logical network.
// For example, to remove the label `exemplary` from a logical network having id `123` send the following request:
// [source]
// ----
// DELETE /ovirt-engine/api/networks/123/networklabels/exemplary
// ----
//
type NetworkLabelServiceRemoveResponse struct {
}

//
// Removes a label from a logical network.
// For example, to remove the label `exemplary` from a logical network having id `123` send the following request:
// [source]
// ----
// DELETE /ovirt-engine/api/networks/123/networklabels/exemplary
// ----
//
func (p *NetworkLabelService) Remove() *NetworkLabelServiceRemoveRequest {
	return &NetworkLabelServiceRemoveRequest{NetworkLabelService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *NetworkLabelService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *NetworkLabelService) String() string {
	return fmt.Sprintf("NetworkLabelService:%s", op.path)
}

//
// Represents a service to manage collection of event-subscription of a user.
//
type EventSubscriptionsService struct {
	BaseService
}

func NewEventSubscriptionsService(connection *Connection, path string) *EventSubscriptionsService {
	var result EventSubscriptionsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new event-subscription to the system.
// An event-subscription is always added in the context of a user. For example, to add new
// event-subscription for `host_high_cpu_use` for user `123`, and have the notification
// sent to the e-mail address: `a@b.com`, send a request like this:
// ....
// POST /ovirt-engine/api/users/123/eventsubscriptions
// ....
// With a request body like this:
// [source,xml]
// ----
// <event_subscription>
//     <event>host_high_cpu_use</event>
//     <address>a@b.com</address>
// </event_subscription>
// ----
// The event name will become the ID of the new event-subscription entity:
// GET .../api/users/123/eventsubscriptions/host_high_cpu_use
// Note that no user id is provided in the request body. This is because the user-id (in this case 123)
// is already known to the API from the context. Note also that event-subscription entity contains
// notification-method field, but it is not provided either in the request body. This is because currently
// it's always set to SMTP as SNMP notifications are still unsupported by the API layer.
//
type EventSubscriptionsServiceAddRequest struct {
	EventSubscriptionsService *EventSubscriptionsService
	header                    map[string]string
	query                     map[string]string
	eventSubscription         *EventSubscription
}

func (p *EventSubscriptionsServiceAddRequest) Header(key, value string) *EventSubscriptionsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EventSubscriptionsServiceAddRequest) Query(key, value string) *EventSubscriptionsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EventSubscriptionsServiceAddRequest) EventSubscription(eventSubscription *EventSubscription) *EventSubscriptionsServiceAddRequest {
	p.eventSubscription = eventSubscription
	return p
}

func (p *EventSubscriptionsServiceAddRequest) Send() (*EventSubscriptionsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EventSubscriptionsService.connection.URL(), p.EventSubscriptionsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLEventSubscriptionWriteOne(writer, p.eventSubscription, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EventSubscriptionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EventSubscriptionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EventSubscriptionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EventSubscriptionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EventSubscriptionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLEventSubscriptionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &EventSubscriptionsServiceAddResponse{eventSubscription: result}, nil
}

func (p *EventSubscriptionsServiceAddRequest) MustSend() *EventSubscriptionsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new event-subscription to the system.
// An event-subscription is always added in the context of a user. For example, to add new
// event-subscription for `host_high_cpu_use` for user `123`, and have the notification
// sent to the e-mail address: `a@b.com`, send a request like this:
// ....
// POST /ovirt-engine/api/users/123/eventsubscriptions
// ....
// With a request body like this:
// [source,xml]
// ----
// <event_subscription>
//     <event>host_high_cpu_use</event>
//     <address>a@b.com</address>
// </event_subscription>
// ----
// The event name will become the ID of the new event-subscription entity:
// GET .../api/users/123/eventsubscriptions/host_high_cpu_use
// Note that no user id is provided in the request body. This is because the user-id (in this case 123)
// is already known to the API from the context. Note also that event-subscription entity contains
// notification-method field, but it is not provided either in the request body. This is because currently
// it's always set to SMTP as SNMP notifications are still unsupported by the API layer.
//
type EventSubscriptionsServiceAddResponse struct {
	eventSubscription *EventSubscription
}

func (p *EventSubscriptionsServiceAddResponse) EventSubscription() (*EventSubscription, bool) {
	if p.eventSubscription != nil {
		return p.eventSubscription, true
	}
	return nil, false
}

func (p *EventSubscriptionsServiceAddResponse) MustEventSubscription() *EventSubscription {
	if p.eventSubscription == nil {
		panic("eventSubscription in response does not exist")
	}
	return p.eventSubscription
}

//
// Add a new event-subscription to the system.
// An event-subscription is always added in the context of a user. For example, to add new
// event-subscription for `host_high_cpu_use` for user `123`, and have the notification
// sent to the e-mail address: `a@b.com`, send a request like this:
// ....
// POST /ovirt-engine/api/users/123/eventsubscriptions
// ....
// With a request body like this:
// [source,xml]
// ----
// <event_subscription>
//     <event>host_high_cpu_use</event>
//     <address>a@b.com</address>
// </event_subscription>
// ----
// The event name will become the ID of the new event-subscription entity:
// GET .../api/users/123/eventsubscriptions/host_high_cpu_use
// Note that no user id is provided in the request body. This is because the user-id (in this case 123)
// is already known to the API from the context. Note also that event-subscription entity contains
// notification-method field, but it is not provided either in the request body. This is because currently
// it's always set to SMTP as SNMP notifications are still unsupported by the API layer.
//
func (p *EventSubscriptionsService) Add() *EventSubscriptionsServiceAddRequest {
	return &EventSubscriptionsServiceAddRequest{EventSubscriptionsService: p}
}

//
// List the event-subscriptions for the provided user.
// For example to list event-subscriptions for user `123`:
// ....
// GET /ovirt-engine/api/users/123/event-subscriptions
// ....
// [source,xml]
// ----
// <event-subscriptions>
//   <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/host_install_failed">
//     <event>host_install_failed</event>
//     <notification_method>smtp</notification_method>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//     <address>a@b.com</address>
//   </event-subscription>
//   <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/vm_paused">
//     <event>vm_paused</event>
//     <notification_method>smtp</notification_method>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//     <address>a@b.com</address>
//   </event-subscription>
// </event-subscriptions>
// ----
//
type EventSubscriptionsServiceListRequest struct {
	EventSubscriptionsService *EventSubscriptionsService
	header                    map[string]string
	query                     map[string]string
	follow                    *string
	max                       *int64
}

func (p *EventSubscriptionsServiceListRequest) Header(key, value string) *EventSubscriptionsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EventSubscriptionsServiceListRequest) Query(key, value string) *EventSubscriptionsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EventSubscriptionsServiceListRequest) Follow(follow string) *EventSubscriptionsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *EventSubscriptionsServiceListRequest) Max(max int64) *EventSubscriptionsServiceListRequest {
	p.max = &max
	return p
}

func (p *EventSubscriptionsServiceListRequest) Send() (*EventSubscriptionsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EventSubscriptionsService.connection.URL(), p.EventSubscriptionsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EventSubscriptionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EventSubscriptionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EventSubscriptionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EventSubscriptionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EventSubscriptionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLEventSubscriptionReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &EventSubscriptionsServiceListResponse{eventSubscriptions: result}, nil
}

func (p *EventSubscriptionsServiceListRequest) MustSend() *EventSubscriptionsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List the event-subscriptions for the provided user.
// For example to list event-subscriptions for user `123`:
// ....
// GET /ovirt-engine/api/users/123/event-subscriptions
// ....
// [source,xml]
// ----
// <event-subscriptions>
//   <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/host_install_failed">
//     <event>host_install_failed</event>
//     <notification_method>smtp</notification_method>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//     <address>a@b.com</address>
//   </event-subscription>
//   <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/vm_paused">
//     <event>vm_paused</event>
//     <notification_method>smtp</notification_method>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//     <address>a@b.com</address>
//   </event-subscription>
// </event-subscriptions>
// ----
//
type EventSubscriptionsServiceListResponse struct {
	eventSubscriptions *EventSubscriptionSlice
}

func (p *EventSubscriptionsServiceListResponse) EventSubscriptions() (*EventSubscriptionSlice, bool) {
	if p.eventSubscriptions != nil {
		return p.eventSubscriptions, true
	}
	return nil, false
}

func (p *EventSubscriptionsServiceListResponse) MustEventSubscriptions() *EventSubscriptionSlice {
	if p.eventSubscriptions == nil {
		panic("eventSubscriptions in response does not exist")
	}
	return p.eventSubscriptions
}

//
// List the event-subscriptions for the provided user.
// For example to list event-subscriptions for user `123`:
// ....
// GET /ovirt-engine/api/users/123/event-subscriptions
// ....
// [source,xml]
// ----
// <event-subscriptions>
//   <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/host_install_failed">
//     <event>host_install_failed</event>
//     <notification_method>smtp</notification_method>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//     <address>a@b.com</address>
//   </event-subscription>
//   <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/vm_paused">
//     <event>vm_paused</event>
//     <notification_method>smtp</notification_method>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//     <address>a@b.com</address>
//   </event-subscription>
// </event-subscriptions>
// ----
//
func (p *EventSubscriptionsService) List() *EventSubscriptionsServiceListRequest {
	return &EventSubscriptionsServiceListRequest{EventSubscriptionsService: p}
}

//
// Reference to the service that manages a specific event-subscription.
//
func (op *EventSubscriptionsService) EventSubscriptionService(id string) *EventSubscriptionService {
	return NewEventSubscriptionService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *EventSubscriptionsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.EventSubscriptionService(path), nil
	}
	return op.EventSubscriptionService(path[:index]).Service(path[index+1:])
}

func (op *EventSubscriptionsService) String() string {
	return fmt.Sprintf("EventSubscriptionsService:%s", op.path)
}

//
// This service represents one label to entity assignment
// when accessed using the entities/affinitylabels subcollection.
//
type AssignedAffinityLabelService struct {
	BaseService
}

func NewAssignedAffinityLabelService(connection *Connection, path string) *AssignedAffinityLabelService {
	var result AssignedAffinityLabelService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves details about the attached label.
//
type AssignedAffinityLabelServiceGetRequest struct {
	AssignedAffinityLabelService *AssignedAffinityLabelService
	header                       map[string]string
	query                        map[string]string
	follow                       *string
}

func (p *AssignedAffinityLabelServiceGetRequest) Header(key, value string) *AssignedAffinityLabelServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedAffinityLabelServiceGetRequest) Query(key, value string) *AssignedAffinityLabelServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedAffinityLabelServiceGetRequest) Follow(follow string) *AssignedAffinityLabelServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AssignedAffinityLabelServiceGetRequest) Send() (*AssignedAffinityLabelServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedAffinityLabelService.connection.URL(), p.AssignedAffinityLabelService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedAffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedAffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedAffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedAffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedAffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedAffinityLabelServiceGetResponse{label: result}, nil
}

func (p *AssignedAffinityLabelServiceGetRequest) MustSend() *AssignedAffinityLabelServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves details about the attached label.
//
type AssignedAffinityLabelServiceGetResponse struct {
	label *AffinityLabel
}

func (p *AssignedAffinityLabelServiceGetResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AssignedAffinityLabelServiceGetResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Retrieves details about the attached label.
//
func (p *AssignedAffinityLabelService) Get() *AssignedAffinityLabelServiceGetRequest {
	return &AssignedAffinityLabelServiceGetRequest{AssignedAffinityLabelService: p}
}

//
// Removes the label from an entity. Does not touch the label itself.
//
type AssignedAffinityLabelServiceRemoveRequest struct {
	AssignedAffinityLabelService *AssignedAffinityLabelService
	header                       map[string]string
	query                        map[string]string
}

func (p *AssignedAffinityLabelServiceRemoveRequest) Header(key, value string) *AssignedAffinityLabelServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedAffinityLabelServiceRemoveRequest) Query(key, value string) *AssignedAffinityLabelServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedAffinityLabelServiceRemoveRequest) Send() (*AssignedAffinityLabelServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedAffinityLabelService.connection.URL(), p.AssignedAffinityLabelService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedAffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedAffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedAffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedAffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedAffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AssignedAffinityLabelServiceRemoveResponse), nil
}

func (p *AssignedAffinityLabelServiceRemoveRequest) MustSend() *AssignedAffinityLabelServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the label from an entity. Does not touch the label itself.
//
type AssignedAffinityLabelServiceRemoveResponse struct {
}

//
// Removes the label from an entity. Does not touch the label itself.
//
func (p *AssignedAffinityLabelService) Remove() *AssignedAffinityLabelServiceRemoveRequest {
	return &AssignedAffinityLabelServiceRemoveRequest{AssignedAffinityLabelService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedAffinityLabelService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AssignedAffinityLabelService) String() string {
	return fmt.Sprintf("AssignedAffinityLabelService:%s", op.path)
}

//
// Manages the ser of labels attached to a network or to a host NIC.
//
type NetworkLabelsService struct {
	BaseService
}

func NewNetworkLabelsService(connection *Connection, path string) *NetworkLabelsService {
	var result NetworkLabelsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Attaches label to logical network.
// You can attach labels to a logical network to automate the association of that logical network with physical host
// network interfaces to which the same label has been attached.
// For example, to attach the label `mylabel` to a logical network having id `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/networks/123/networklabels
// ----
// With a request body like this:
// [source,xml]
// ----
// <network_label id="mylabel"/>
// ----
//
type NetworkLabelsServiceAddRequest struct {
	NetworkLabelsService *NetworkLabelsService
	header               map[string]string
	query                map[string]string
	label                *NetworkLabel
}

func (p *NetworkLabelsServiceAddRequest) Header(key, value string) *NetworkLabelsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkLabelsServiceAddRequest) Query(key, value string) *NetworkLabelsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkLabelsServiceAddRequest) Label(label *NetworkLabel) *NetworkLabelsServiceAddRequest {
	p.label = label
	return p
}

func (p *NetworkLabelsServiceAddRequest) Send() (*NetworkLabelsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkLabelsService.connection.URL(), p.NetworkLabelsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkLabelWriteOne(writer, p.label, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &NetworkLabelsServiceAddResponse{label: result}, nil
}

func (p *NetworkLabelsServiceAddRequest) MustSend() *NetworkLabelsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Attaches label to logical network.
// You can attach labels to a logical network to automate the association of that logical network with physical host
// network interfaces to which the same label has been attached.
// For example, to attach the label `mylabel` to a logical network having id `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/networks/123/networklabels
// ----
// With a request body like this:
// [source,xml]
// ----
// <network_label id="mylabel"/>
// ----
//
type NetworkLabelsServiceAddResponse struct {
	label *NetworkLabel
}

func (p *NetworkLabelsServiceAddResponse) Label() (*NetworkLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *NetworkLabelsServiceAddResponse) MustLabel() *NetworkLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Attaches label to logical network.
// You can attach labels to a logical network to automate the association of that logical network with physical host
// network interfaces to which the same label has been attached.
// For example, to attach the label `mylabel` to a logical network having id `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/networks/123/networklabels
// ----
// With a request body like this:
// [source,xml]
// ----
// <network_label id="mylabel"/>
// ----
//
func (p *NetworkLabelsService) Add() *NetworkLabelsServiceAddRequest {
	return &NetworkLabelsServiceAddRequest{NetworkLabelsService: p}
}

//
// Returns the list of labels attached to the network or host NIC.
// The order of the returned list of labels isn't guaranteed.
//
type NetworkLabelsServiceListRequest struct {
	NetworkLabelsService *NetworkLabelsService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *NetworkLabelsServiceListRequest) Header(key, value string) *NetworkLabelsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkLabelsServiceListRequest) Query(key, value string) *NetworkLabelsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkLabelsServiceListRequest) Follow(follow string) *NetworkLabelsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *NetworkLabelsServiceListRequest) Max(max int64) *NetworkLabelsServiceListRequest {
	p.max = &max
	return p
}

func (p *NetworkLabelsServiceListRequest) Send() (*NetworkLabelsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkLabelsService.connection.URL(), p.NetworkLabelsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkLabelReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &NetworkLabelsServiceListResponse{labels: result}, nil
}

func (p *NetworkLabelsServiceListRequest) MustSend() *NetworkLabelsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of labels attached to the network or host NIC.
// The order of the returned list of labels isn't guaranteed.
//
type NetworkLabelsServiceListResponse struct {
	labels *NetworkLabelSlice
}

func (p *NetworkLabelsServiceListResponse) Labels() (*NetworkLabelSlice, bool) {
	if p.labels != nil {
		return p.labels, true
	}
	return nil, false
}

func (p *NetworkLabelsServiceListResponse) MustLabels() *NetworkLabelSlice {
	if p.labels == nil {
		panic("labels in response does not exist")
	}
	return p.labels
}

//
// Returns the list of labels attached to the network or host NIC.
// The order of the returned list of labels isn't guaranteed.
//
func (p *NetworkLabelsService) List() *NetworkLabelsServiceListRequest {
	return &NetworkLabelsServiceListRequest{NetworkLabelsService: p}
}

//
//
func (op *NetworkLabelsService) LabelService(id string) *NetworkLabelService {
	return NewNetworkLabelService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *NetworkLabelsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.LabelService(path), nil
	}
	return op.LabelService(path[:index]).Service(path[index+1:])
}

func (op *NetworkLabelsService) String() string {
	return fmt.Sprintf("NetworkLabelsService:%s", op.path)
}

//
//
type SnapshotDiskService struct {
	BaseService
}

func NewSnapshotDiskService(connection *Connection, path string) *SnapshotDiskService {
	var result SnapshotDiskService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type SnapshotDiskServiceGetRequest struct {
	SnapshotDiskService *SnapshotDiskService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *SnapshotDiskServiceGetRequest) Header(key, value string) *SnapshotDiskServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SnapshotDiskServiceGetRequest) Query(key, value string) *SnapshotDiskServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SnapshotDiskServiceGetRequest) Follow(follow string) *SnapshotDiskServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *SnapshotDiskServiceGetRequest) Send() (*SnapshotDiskServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SnapshotDiskService.connection.URL(), p.SnapshotDiskService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SnapshotDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SnapshotDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SnapshotDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SnapshotDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SnapshotDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SnapshotDiskServiceGetResponse{disk: result}, nil
}

func (p *SnapshotDiskServiceGetRequest) MustSend() *SnapshotDiskServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SnapshotDiskServiceGetResponse struct {
	disk *Disk
}

func (p *SnapshotDiskServiceGetResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *SnapshotDiskServiceGetResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
//
func (p *SnapshotDiskService) Get() *SnapshotDiskServiceGetRequest {
	return &SnapshotDiskServiceGetRequest{SnapshotDiskService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SnapshotDiskService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *SnapshotDiskService) String() string {
	return fmt.Sprintf("SnapshotDiskService:%s", op.path)
}

//
// Manages a group of users. Use this service to either get groups details or remove groups. In order
// to add new groups please use <<services/groups, service>> that manages the collection of groups.
//
type GroupService struct {
	BaseService
}

func NewGroupService(connection *Connection, path string) *GroupService {
	var result GroupService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets the system group information.
// Usage:
// ....
// GET /ovirt-engine/api/groups/123
// ....
// Will return the group information:
// [source,xml]
// ----
// <group href="/ovirt-engine/api/groups/123" id="123">
//   <name>mygroup</name>
//   <link href="/ovirt-engine/api/groups/123/roles" rel="roles"/>
//   <link href="/ovirt-engine/api/groups/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/groups/123/tags" rel="tags"/>
//   <domain_entry_id>476652557A382F67696B6D2B32762B37796E46476D513D3D</domain_entry_id>
//   <namespace>DC=example,DC=com</namespace>
//   <domain href="/ovirt-engine/api/domains/ABCDEF" id="ABCDEF">
//     <name>myextension-authz</name>
//   </domain>
// </group>
// ----
//
type GroupServiceGetRequest struct {
	GroupService *GroupService
	header       map[string]string
	query        map[string]string
	follow       *string
}

func (p *GroupServiceGetRequest) Header(key, value string) *GroupServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GroupServiceGetRequest) Query(key, value string) *GroupServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GroupServiceGetRequest) Follow(follow string) *GroupServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *GroupServiceGetRequest) Send() (*GroupServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GroupService.connection.URL(), p.GroupService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GroupService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GroupService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GroupService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GroupService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GroupService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGroupReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &GroupServiceGetResponse{get: result}, nil
}

func (p *GroupServiceGetRequest) MustSend() *GroupServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the system group information.
// Usage:
// ....
// GET /ovirt-engine/api/groups/123
// ....
// Will return the group information:
// [source,xml]
// ----
// <group href="/ovirt-engine/api/groups/123" id="123">
//   <name>mygroup</name>
//   <link href="/ovirt-engine/api/groups/123/roles" rel="roles"/>
//   <link href="/ovirt-engine/api/groups/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/groups/123/tags" rel="tags"/>
//   <domain_entry_id>476652557A382F67696B6D2B32762B37796E46476D513D3D</domain_entry_id>
//   <namespace>DC=example,DC=com</namespace>
//   <domain href="/ovirt-engine/api/domains/ABCDEF" id="ABCDEF">
//     <name>myextension-authz</name>
//   </domain>
// </group>
// ----
//
type GroupServiceGetResponse struct {
	get *Group
}

func (p *GroupServiceGetResponse) Get() (*Group, bool) {
	if p.get != nil {
		return p.get, true
	}
	return nil, false
}

func (p *GroupServiceGetResponse) MustGet() *Group {
	if p.get == nil {
		panic("get in response does not exist")
	}
	return p.get
}

//
// Gets the system group information.
// Usage:
// ....
// GET /ovirt-engine/api/groups/123
// ....
// Will return the group information:
// [source,xml]
// ----
// <group href="/ovirt-engine/api/groups/123" id="123">
//   <name>mygroup</name>
//   <link href="/ovirt-engine/api/groups/123/roles" rel="roles"/>
//   <link href="/ovirt-engine/api/groups/123/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/groups/123/tags" rel="tags"/>
//   <domain_entry_id>476652557A382F67696B6D2B32762B37796E46476D513D3D</domain_entry_id>
//   <namespace>DC=example,DC=com</namespace>
//   <domain href="/ovirt-engine/api/domains/ABCDEF" id="ABCDEF">
//     <name>myextension-authz</name>
//   </domain>
// </group>
// ----
//
func (p *GroupService) Get() *GroupServiceGetRequest {
	return &GroupServiceGetRequest{GroupService: p}
}

//
// Removes the system group.
// Usage:
// ....
// DELETE /ovirt-engine/api/groups/123
// ....
//
type GroupServiceRemoveRequest struct {
	GroupService *GroupService
	header       map[string]string
	query        map[string]string
	async        *bool
}

func (p *GroupServiceRemoveRequest) Header(key, value string) *GroupServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GroupServiceRemoveRequest) Query(key, value string) *GroupServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GroupServiceRemoveRequest) Async(async bool) *GroupServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *GroupServiceRemoveRequest) Send() (*GroupServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GroupService.connection.URL(), p.GroupService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GroupService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GroupService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GroupService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GroupService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GroupService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(GroupServiceRemoveResponse), nil
}

func (p *GroupServiceRemoveRequest) MustSend() *GroupServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the system group.
// Usage:
// ....
// DELETE /ovirt-engine/api/groups/123
// ....
//
type GroupServiceRemoveResponse struct {
}

//
// Removes the system group.
// Usage:
// ....
// DELETE /ovirt-engine/api/groups/123
// ....
//
func (p *GroupService) Remove() *GroupServiceRemoveRequest {
	return &GroupServiceRemoveRequest{GroupService: p}
}

//
// Reference to the service that manages the collection of permissions assigned to this system group.
//
func (op *GroupService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Reference to the service that manages the collection of roles assigned to this system group.
//
func (op *GroupService) RolesService() *AssignedRolesService {
	return NewAssignedRolesService(op.connection, fmt.Sprintf("%s/roles", op.path))
}

//
// Reference to the service that manages the collection of tags assigned to this system group.
//
func (op *GroupService) TagsService() *AssignedTagsService {
	return NewAssignedTagsService(op.connection, fmt.Sprintf("%s/tags", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GroupService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "roles" {
		return op.RolesService(), nil
	}
	if strings.HasPrefix(path, "roles/") {
		return op.RolesService().Service(path[6:])
	}
	if path == "tags" {
		return op.TagsService(), nil
	}
	if strings.HasPrefix(path, "tags/") {
		return op.TagsService().Service(path[5:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *GroupService) String() string {
	return fmt.Sprintf("GroupService:%s", op.path)
}

//
// A service to list all authentication domains in the system.
//
type DomainsService struct {
	BaseService
}

func NewDomainsService(connection *Connection, path string) *DomainsService {
	var result DomainsService
	result.connection = connection
	result.path = path
	return &result
}

//
// List all the authentication domains in the system.
// Usage:
// ....
// GET /ovirt-engine/api/domains
// ....
// Will return the list of domains:
// [source,xml]
// ----
// <domains>
//   <domain href="/ovirt-engine/api/domains/5678" id="5678">
//     <name>internal-authz</name>
//     <link href="/ovirt-engine/api/domains/5678/users" rel="users"/>
//     <link href="/ovirt-engine/api/domains/5678/groups" rel="groups"/>
//     <link href="/ovirt-engine/api/domains/5678/users?search={query}" rel="users/search"/>
//     <link href="/ovirt-engine/api/domains/5678/groups?search={query}" rel="groups/search"/>
//   </domain>
// </domains>
// ----
// The order of the returned list of domains isn't guaranteed.
//
type DomainsServiceListRequest struct {
	DomainsService *DomainsService
	header         map[string]string
	query          map[string]string
	follow         *string
	max            *int64
}

func (p *DomainsServiceListRequest) Header(key, value string) *DomainsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainsServiceListRequest) Query(key, value string) *DomainsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainsServiceListRequest) Follow(follow string) *DomainsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DomainsServiceListRequest) Max(max int64) *DomainsServiceListRequest {
	p.max = &max
	return p
}

func (p *DomainsServiceListRequest) Send() (*DomainsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainsService.connection.URL(), p.DomainsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDomainReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DomainsServiceListResponse{domains: result}, nil
}

func (p *DomainsServiceListRequest) MustSend() *DomainsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all the authentication domains in the system.
// Usage:
// ....
// GET /ovirt-engine/api/domains
// ....
// Will return the list of domains:
// [source,xml]
// ----
// <domains>
//   <domain href="/ovirt-engine/api/domains/5678" id="5678">
//     <name>internal-authz</name>
//     <link href="/ovirt-engine/api/domains/5678/users" rel="users"/>
//     <link href="/ovirt-engine/api/domains/5678/groups" rel="groups"/>
//     <link href="/ovirt-engine/api/domains/5678/users?search={query}" rel="users/search"/>
//     <link href="/ovirt-engine/api/domains/5678/groups?search={query}" rel="groups/search"/>
//   </domain>
// </domains>
// ----
// The order of the returned list of domains isn't guaranteed.
//
type DomainsServiceListResponse struct {
	domains *DomainSlice
}

func (p *DomainsServiceListResponse) Domains() (*DomainSlice, bool) {
	if p.domains != nil {
		return p.domains, true
	}
	return nil, false
}

func (p *DomainsServiceListResponse) MustDomains() *DomainSlice {
	if p.domains == nil {
		panic("domains in response does not exist")
	}
	return p.domains
}

//
// List all the authentication domains in the system.
// Usage:
// ....
// GET /ovirt-engine/api/domains
// ....
// Will return the list of domains:
// [source,xml]
// ----
// <domains>
//   <domain href="/ovirt-engine/api/domains/5678" id="5678">
//     <name>internal-authz</name>
//     <link href="/ovirt-engine/api/domains/5678/users" rel="users"/>
//     <link href="/ovirt-engine/api/domains/5678/groups" rel="groups"/>
//     <link href="/ovirt-engine/api/domains/5678/users?search={query}" rel="users/search"/>
//     <link href="/ovirt-engine/api/domains/5678/groups?search={query}" rel="groups/search"/>
//   </domain>
// </domains>
// ----
// The order of the returned list of domains isn't guaranteed.
//
func (p *DomainsService) List() *DomainsServiceListRequest {
	return &DomainsServiceListRequest{DomainsService: p}
}

//
// Reference to a service to view details of a domain.
//
func (op *DomainsService) DomainService(id string) *DomainService {
	return NewDomainService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DomainService(path), nil
	}
	return op.DomainService(path[:index]).Service(path[index+1:])
}

func (op *DomainsService) String() string {
	return fmt.Sprintf("DomainsService:%s", op.path)
}

//
//
type SshPublicKeysService struct {
	BaseService
}

func NewSshPublicKeysService(connection *Connection, path string) *SshPublicKeysService {
	var result SshPublicKeysService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type SshPublicKeysServiceAddRequest struct {
	SshPublicKeysService *SshPublicKeysService
	header               map[string]string
	query                map[string]string
	key                  *SshPublicKey
}

func (p *SshPublicKeysServiceAddRequest) Header(key, value string) *SshPublicKeysServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SshPublicKeysServiceAddRequest) Query(key, value string) *SshPublicKeysServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SshPublicKeysServiceAddRequest) Key(key *SshPublicKey) *SshPublicKeysServiceAddRequest {
	p.key = key
	return p
}

func (p *SshPublicKeysServiceAddRequest) Send() (*SshPublicKeysServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SshPublicKeysService.connection.URL(), p.SshPublicKeysService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLSshPublicKeyWriteOne(writer, p.key, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SshPublicKeysService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SshPublicKeysService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SshPublicKeysService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SshPublicKeysService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SshPublicKeysService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSshPublicKeyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SshPublicKeysServiceAddResponse{key: result}, nil
}

func (p *SshPublicKeysServiceAddRequest) MustSend() *SshPublicKeysServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SshPublicKeysServiceAddResponse struct {
	key *SshPublicKey
}

func (p *SshPublicKeysServiceAddResponse) Key() (*SshPublicKey, bool) {
	if p.key != nil {
		return p.key, true
	}
	return nil, false
}

func (p *SshPublicKeysServiceAddResponse) MustKey() *SshPublicKey {
	if p.key == nil {
		panic("key in response does not exist")
	}
	return p.key
}

//
//
func (p *SshPublicKeysService) Add() *SshPublicKeysServiceAddRequest {
	return &SshPublicKeysServiceAddRequest{SshPublicKeysService: p}
}

//
// Returns a list of SSH public keys of the user.
// For example, to retrieve the list of SSH keys of user with identifier `123`,
// send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/users/123/sshpublickeys
// ----
// The result will be the following XML document:
// [source,xml]
// ----
// <ssh_public_keys>
//   <ssh_public_key href="/ovirt-engine/api/users/123/sshpublickeys/456" id="456">
//     <content>ssh-rsa ...</content>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//   </ssh_public_key>
// </ssh_public_keys>
// ----
// Or the following JSON object
// [source,json]
// ----
// {
//   "ssh_public_key": [
//     {
//       "content": "ssh-rsa ...",
//       "user": {
//         "href": "/ovirt-engine/api/users/123",
//         "id": "123"
//       },
//       "href": "/ovirt-engine/api/users/123/sshpublickeys/456",
//       "id": "456"
//     }
//   ]
// }
// ----
// The order of the returned list of keys is not guaranteed.
//
type SshPublicKeysServiceListRequest struct {
	SshPublicKeysService *SshPublicKeysService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *SshPublicKeysServiceListRequest) Header(key, value string) *SshPublicKeysServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SshPublicKeysServiceListRequest) Query(key, value string) *SshPublicKeysServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SshPublicKeysServiceListRequest) Follow(follow string) *SshPublicKeysServiceListRequest {
	p.follow = &follow
	return p
}

func (p *SshPublicKeysServiceListRequest) Max(max int64) *SshPublicKeysServiceListRequest {
	p.max = &max
	return p
}

func (p *SshPublicKeysServiceListRequest) Send() (*SshPublicKeysServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SshPublicKeysService.connection.URL(), p.SshPublicKeysService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SshPublicKeysService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SshPublicKeysService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SshPublicKeysService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SshPublicKeysService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SshPublicKeysService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSshPublicKeyReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &SshPublicKeysServiceListResponse{keys: result}, nil
}

func (p *SshPublicKeysServiceListRequest) MustSend() *SshPublicKeysServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns a list of SSH public keys of the user.
// For example, to retrieve the list of SSH keys of user with identifier `123`,
// send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/users/123/sshpublickeys
// ----
// The result will be the following XML document:
// [source,xml]
// ----
// <ssh_public_keys>
//   <ssh_public_key href="/ovirt-engine/api/users/123/sshpublickeys/456" id="456">
//     <content>ssh-rsa ...</content>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//   </ssh_public_key>
// </ssh_public_keys>
// ----
// Or the following JSON object
// [source,json]
// ----
// {
//   "ssh_public_key": [
//     {
//       "content": "ssh-rsa ...",
//       "user": {
//         "href": "/ovirt-engine/api/users/123",
//         "id": "123"
//       },
//       "href": "/ovirt-engine/api/users/123/sshpublickeys/456",
//       "id": "456"
//     }
//   ]
// }
// ----
// The order of the returned list of keys is not guaranteed.
//
type SshPublicKeysServiceListResponse struct {
	keys *SshPublicKeySlice
}

func (p *SshPublicKeysServiceListResponse) Keys() (*SshPublicKeySlice, bool) {
	if p.keys != nil {
		return p.keys, true
	}
	return nil, false
}

func (p *SshPublicKeysServiceListResponse) MustKeys() *SshPublicKeySlice {
	if p.keys == nil {
		panic("keys in response does not exist")
	}
	return p.keys
}

//
// Returns a list of SSH public keys of the user.
// For example, to retrieve the list of SSH keys of user with identifier `123`,
// send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/users/123/sshpublickeys
// ----
// The result will be the following XML document:
// [source,xml]
// ----
// <ssh_public_keys>
//   <ssh_public_key href="/ovirt-engine/api/users/123/sshpublickeys/456" id="456">
//     <content>ssh-rsa ...</content>
//     <user href="/ovirt-engine/api/users/123" id="123"/>
//   </ssh_public_key>
// </ssh_public_keys>
// ----
// Or the following JSON object
// [source,json]
// ----
// {
//   "ssh_public_key": [
//     {
//       "content": "ssh-rsa ...",
//       "user": {
//         "href": "/ovirt-engine/api/users/123",
//         "id": "123"
//       },
//       "href": "/ovirt-engine/api/users/123/sshpublickeys/456",
//       "id": "456"
//     }
//   ]
// }
// ----
// The order of the returned list of keys is not guaranteed.
//
func (p *SshPublicKeysService) List() *SshPublicKeysServiceListRequest {
	return &SshPublicKeysServiceListRequest{SshPublicKeysService: p}
}

//
//
func (op *SshPublicKeysService) KeyService(id string) *SshPublicKeyService {
	return NewSshPublicKeyService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SshPublicKeysService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.KeyService(path), nil
	}
	return op.KeyService(path[:index]).Service(path[index+1:])
}

func (op *SshPublicKeysService) String() string {
	return fmt.Sprintf("SshPublicKeysService:%s", op.path)
}

//
// A service to view a domain user in the system.
//
type DomainUserService struct {
	BaseService
}

func NewDomainUserService(connection *Connection, path string) *DomainUserService {
	var result DomainUserService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets the domain user information.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678/users/1234
// ....
// Will return the domain user information:
// [source,xml]
// ----
// <user href="/ovirt-engine/api/users/1234" id="1234">
//   <name>admin</name>
//   <namespace>*</namespace>
//   <principal>admin</principal>
//   <user_name>admin@internal-authz</user_name>
//   <domain href="/ovirt-engine/api/domains/5678" id="5678">
//     <name>internal-authz</name>
//   </domain>
//   <groups/>
// </user>
// ----
//
type DomainUserServiceGetRequest struct {
	DomainUserService *DomainUserService
	header            map[string]string
	query             map[string]string
	follow            *string
}

func (p *DomainUserServiceGetRequest) Header(key, value string) *DomainUserServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainUserServiceGetRequest) Query(key, value string) *DomainUserServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainUserServiceGetRequest) Follow(follow string) *DomainUserServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DomainUserServiceGetRequest) Send() (*DomainUserServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainUserService.connection.URL(), p.DomainUserService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainUserService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainUserService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainUserService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainUserService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainUserService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLUserReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DomainUserServiceGetResponse{user: result}, nil
}

func (p *DomainUserServiceGetRequest) MustSend() *DomainUserServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the domain user information.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678/users/1234
// ....
// Will return the domain user information:
// [source,xml]
// ----
// <user href="/ovirt-engine/api/users/1234" id="1234">
//   <name>admin</name>
//   <namespace>*</namespace>
//   <principal>admin</principal>
//   <user_name>admin@internal-authz</user_name>
//   <domain href="/ovirt-engine/api/domains/5678" id="5678">
//     <name>internal-authz</name>
//   </domain>
//   <groups/>
// </user>
// ----
//
type DomainUserServiceGetResponse struct {
	user *User
}

func (p *DomainUserServiceGetResponse) User() (*User, bool) {
	if p.user != nil {
		return p.user, true
	}
	return nil, false
}

func (p *DomainUserServiceGetResponse) MustUser() *User {
	if p.user == nil {
		panic("user in response does not exist")
	}
	return p.user
}

//
// Gets the domain user information.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678/users/1234
// ....
// Will return the domain user information:
// [source,xml]
// ----
// <user href="/ovirt-engine/api/users/1234" id="1234">
//   <name>admin</name>
//   <namespace>*</namespace>
//   <principal>admin</principal>
//   <user_name>admin@internal-authz</user_name>
//   <domain href="/ovirt-engine/api/domains/5678" id="5678">
//     <name>internal-authz</name>
//   </domain>
//   <groups/>
// </user>
// ----
//
func (p *DomainUserService) Get() *DomainUserServiceGetRequest {
	return &DomainUserServiceGetRequest{DomainUserService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainUserService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DomainUserService) String() string {
	return fmt.Sprintf("DomainUserService:%s", op.path)
}

//
//
type DomainGroupsService struct {
	BaseService
}

func NewDomainGroupsService(connection *Connection, path string) *DomainGroupsService {
	var result DomainGroupsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of groups.
// The order of the returned list of groups isn't guaranteed.
//
type DomainGroupsServiceListRequest struct {
	DomainGroupsService *DomainGroupsService
	header              map[string]string
	query               map[string]string
	caseSensitive       *bool
	follow              *string
	max                 *int64
	search              *string
}

func (p *DomainGroupsServiceListRequest) Header(key, value string) *DomainGroupsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainGroupsServiceListRequest) Query(key, value string) *DomainGroupsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainGroupsServiceListRequest) CaseSensitive(caseSensitive bool) *DomainGroupsServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *DomainGroupsServiceListRequest) Follow(follow string) *DomainGroupsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DomainGroupsServiceListRequest) Max(max int64) *DomainGroupsServiceListRequest {
	p.max = &max
	return p
}

func (p *DomainGroupsServiceListRequest) Search(search string) *DomainGroupsServiceListRequest {
	p.search = &search
	return p
}

func (p *DomainGroupsServiceListRequest) Send() (*DomainGroupsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainGroupsService.connection.URL(), p.DomainGroupsService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainGroupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainGroupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainGroupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainGroupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainGroupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGroupReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DomainGroupsServiceListResponse{groups: result}, nil
}

func (p *DomainGroupsServiceListRequest) MustSend() *DomainGroupsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of groups.
// The order of the returned list of groups isn't guaranteed.
//
type DomainGroupsServiceListResponse struct {
	groups *GroupSlice
}

func (p *DomainGroupsServiceListResponse) Groups() (*GroupSlice, bool) {
	if p.groups != nil {
		return p.groups, true
	}
	return nil, false
}

func (p *DomainGroupsServiceListResponse) MustGroups() *GroupSlice {
	if p.groups == nil {
		panic("groups in response does not exist")
	}
	return p.groups
}

//
// Returns the list of groups.
// The order of the returned list of groups isn't guaranteed.
//
func (p *DomainGroupsService) List() *DomainGroupsServiceListRequest {
	return &DomainGroupsServiceListRequest{DomainGroupsService: p}
}

//
//
func (op *DomainGroupsService) GroupService(id string) *DomainGroupService {
	return NewDomainGroupService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainGroupsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.GroupService(path), nil
	}
	return op.GroupService(path[:index]).Service(path[index+1:])
}

func (op *DomainGroupsService) String() string {
	return fmt.Sprintf("DomainGroupsService:%s", op.path)
}

//
// A service to manage a user in the system.
// Use this service to either get users details or remove users.
// In order to add new users please use
// <<services/users>>.
//
type UserService struct {
	BaseService
}

func NewUserService(connection *Connection, path string) *UserService {
	var result UserService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets the system user information.
// Usage:
// ....
// GET /ovirt-engine/api/users/1234
// ....
// Will return the user information:
// [source,xml]
// ----
// <user href="/ovirt-engine/api/users/1234" id="1234">
//   <name>admin</name>
//   <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
//   <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
//   <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
//   <department></department>
//   <domain_entry_id>23456</domain_entry_id>
//   <email>user1@domain.com</email>
//   <last_name>Lastname</last_name>
//   <namespace>*</namespace>
//   <principal>user1</principal>
//   <user_name>user1@domain-authz</user_name>
//   <domain href="/ovirt-engine/api/domains/45678" id="45678">
//     <name>domain-authz</name>
//   </domain>
// </user>
// ----
//
type UserServiceGetRequest struct {
	UserService *UserService
	header      map[string]string
	query       map[string]string
	follow      *string
}

func (p *UserServiceGetRequest) Header(key, value string) *UserServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *UserServiceGetRequest) Query(key, value string) *UserServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *UserServiceGetRequest) Follow(follow string) *UserServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *UserServiceGetRequest) Send() (*UserServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.UserService.connection.URL(), p.UserService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.UserService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.UserService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.UserService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.UserService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.UserService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLUserReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &UserServiceGetResponse{user: result}, nil
}

func (p *UserServiceGetRequest) MustSend() *UserServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the system user information.
// Usage:
// ....
// GET /ovirt-engine/api/users/1234
// ....
// Will return the user information:
// [source,xml]
// ----
// <user href="/ovirt-engine/api/users/1234" id="1234">
//   <name>admin</name>
//   <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
//   <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
//   <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
//   <department></department>
//   <domain_entry_id>23456</domain_entry_id>
//   <email>user1@domain.com</email>
//   <last_name>Lastname</last_name>
//   <namespace>*</namespace>
//   <principal>user1</principal>
//   <user_name>user1@domain-authz</user_name>
//   <domain href="/ovirt-engine/api/domains/45678" id="45678">
//     <name>domain-authz</name>
//   </domain>
// </user>
// ----
//
type UserServiceGetResponse struct {
	user *User
}

func (p *UserServiceGetResponse) User() (*User, bool) {
	if p.user != nil {
		return p.user, true
	}
	return nil, false
}

func (p *UserServiceGetResponse) MustUser() *User {
	if p.user == nil {
		panic("user in response does not exist")
	}
	return p.user
}

//
// Gets the system user information.
// Usage:
// ....
// GET /ovirt-engine/api/users/1234
// ....
// Will return the user information:
// [source,xml]
// ----
// <user href="/ovirt-engine/api/users/1234" id="1234">
//   <name>admin</name>
//   <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
//   <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
//   <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
//   <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
//   <department></department>
//   <domain_entry_id>23456</domain_entry_id>
//   <email>user1@domain.com</email>
//   <last_name>Lastname</last_name>
//   <namespace>*</namespace>
//   <principal>user1</principal>
//   <user_name>user1@domain-authz</user_name>
//   <domain href="/ovirt-engine/api/domains/45678" id="45678">
//     <name>domain-authz</name>
//   </domain>
// </user>
// ----
//
func (p *UserService) Get() *UserServiceGetRequest {
	return &UserServiceGetRequest{UserService: p}
}

//
// Removes the system user.
// Usage:
// ....
// DELETE /ovirt-engine/api/users/1234
// ....
//
type UserServiceRemoveRequest struct {
	UserService *UserService
	header      map[string]string
	query       map[string]string
	async       *bool
}

func (p *UserServiceRemoveRequest) Header(key, value string) *UserServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *UserServiceRemoveRequest) Query(key, value string) *UserServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *UserServiceRemoveRequest) Async(async bool) *UserServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *UserServiceRemoveRequest) Send() (*UserServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.UserService.connection.URL(), p.UserService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.UserService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.UserService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.UserService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.UserService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.UserService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(UserServiceRemoveResponse), nil
}

func (p *UserServiceRemoveRequest) MustSend() *UserServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the system user.
// Usage:
// ....
// DELETE /ovirt-engine/api/users/1234
// ....
//
type UserServiceRemoveResponse struct {
}

//
// Removes the system user.
// Usage:
// ....
// DELETE /ovirt-engine/api/users/1234
// ....
//
func (p *UserService) Remove() *UserServiceRemoveRequest {
	return &UserServiceRemoveRequest{UserService: p}
}

//
// Updates information about the user.
// Only the `user_options` field can be updated.
// For example, to update user options:
// [source]
// ----
// PUT /ovirt-engine/api/users/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <user>
//    <user_options>
//       <property>
//          <name>test</name>
//          <value>test1</value>
//       </property>
//    </user_options>
// </user>
// ----
//
type UserServiceUpdateRequest struct {
	UserService *UserService
	header      map[string]string
	query       map[string]string
	user        *User
}

func (p *UserServiceUpdateRequest) Header(key, value string) *UserServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *UserServiceUpdateRequest) Query(key, value string) *UserServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *UserServiceUpdateRequest) User(user *User) *UserServiceUpdateRequest {
	p.user = user
	return p
}

func (p *UserServiceUpdateRequest) Send() (*UserServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.UserService.connection.URL(), p.UserService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLUserWriteOne(writer, p.user, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.UserService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.UserService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.UserService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.UserService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.UserService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLUserReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &UserServiceUpdateResponse{user: result}, nil
}

func (p *UserServiceUpdateRequest) MustSend() *UserServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates information about the user.
// Only the `user_options` field can be updated.
// For example, to update user options:
// [source]
// ----
// PUT /ovirt-engine/api/users/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <user>
//    <user_options>
//       <property>
//          <name>test</name>
//          <value>test1</value>
//       </property>
//    </user_options>
// </user>
// ----
//
type UserServiceUpdateResponse struct {
	user *User
}

func (p *UserServiceUpdateResponse) User() (*User, bool) {
	if p.user != nil {
		return p.user, true
	}
	return nil, false
}

func (p *UserServiceUpdateResponse) MustUser() *User {
	if p.user == nil {
		panic("user in response does not exist")
	}
	return p.user
}

//
// Updates information about the user.
// Only the `user_options` field can be updated.
// For example, to update user options:
// [source]
// ----
// PUT /ovirt-engine/api/users/123
// ----
// With a request body like this:
// [source,xml]
// ----
// <user>
//    <user_options>
//       <property>
//          <name>test</name>
//          <value>test1</value>
//       </property>
//    </user_options>
// </user>
// ----
//
func (p *UserService) Update() *UserServiceUpdateRequest {
	return &UserServiceUpdateRequest{UserService: p}
}

//
// List of event-subscriptions for this user.
//
func (op *UserService) EventSubscriptionsService() *EventSubscriptionsService {
	return NewEventSubscriptionsService(op.connection, fmt.Sprintf("%s/eventsubscriptions", op.path))
}

//
//
func (op *UserService) GroupsService() *DomainUserGroupsService {
	return NewDomainUserGroupsService(op.connection, fmt.Sprintf("%s/groups", op.path))
}

//
//
func (op *UserService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
//
func (op *UserService) RolesService() *AssignedRolesService {
	return NewAssignedRolesService(op.connection, fmt.Sprintf("%s/roles", op.path))
}

//
//
func (op *UserService) SshPublicKeysService() *SshPublicKeysService {
	return NewSshPublicKeysService(op.connection, fmt.Sprintf("%s/sshpublickeys", op.path))
}

//
//
func (op *UserService) TagsService() *AssignedTagsService {
	return NewAssignedTagsService(op.connection, fmt.Sprintf("%s/tags", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *UserService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "eventsubscriptions" {
		return op.EventSubscriptionsService(), nil
	}
	if strings.HasPrefix(path, "eventsubscriptions/") {
		return op.EventSubscriptionsService().Service(path[19:])
	}
	if path == "groups" {
		return op.GroupsService(), nil
	}
	if strings.HasPrefix(path, "groups/") {
		return op.GroupsService().Service(path[7:])
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "roles" {
		return op.RolesService(), nil
	}
	if strings.HasPrefix(path, "roles/") {
		return op.RolesService().Service(path[6:])
	}
	if path == "sshpublickeys" {
		return op.SshPublicKeysService(), nil
	}
	if strings.HasPrefix(path, "sshpublickeys/") {
		return op.SshPublicKeysService().Service(path[14:])
	}
	if path == "tags" {
		return op.TagsService(), nil
	}
	if strings.HasPrefix(path, "tags/") {
		return op.TagsService().Service(path[5:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *UserService) String() string {
	return fmt.Sprintf("UserService:%s", op.path)
}

//
//
type DomainGroupService struct {
	BaseService
}

func NewDomainGroupService(connection *Connection, path string) *DomainGroupService {
	var result DomainGroupService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type DomainGroupServiceGetRequest struct {
	DomainGroupService *DomainGroupService
	header             map[string]string
	query              map[string]string
	follow             *string
}

func (p *DomainGroupServiceGetRequest) Header(key, value string) *DomainGroupServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainGroupServiceGetRequest) Query(key, value string) *DomainGroupServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainGroupServiceGetRequest) Follow(follow string) *DomainGroupServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DomainGroupServiceGetRequest) Send() (*DomainGroupServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainGroupService.connection.URL(), p.DomainGroupService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainGroupService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainGroupService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainGroupService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainGroupService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainGroupService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGroupReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DomainGroupServiceGetResponse{get: result}, nil
}

func (p *DomainGroupServiceGetRequest) MustSend() *DomainGroupServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DomainGroupServiceGetResponse struct {
	get *Group
}

func (p *DomainGroupServiceGetResponse) Get() (*Group, bool) {
	if p.get != nil {
		return p.get, true
	}
	return nil, false
}

func (p *DomainGroupServiceGetResponse) MustGet() *Group {
	if p.get == nil {
		panic("get in response does not exist")
	}
	return p.get
}

//
//
func (p *DomainGroupService) Get() *DomainGroupServiceGetRequest {
	return &DomainGroupServiceGetRequest{DomainGroupService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainGroupService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DomainGroupService) String() string {
	return fmt.Sprintf("DomainGroupService:%s", op.path)
}

//
// A service to view details of an authentication domain in the system.
//
type DomainService struct {
	BaseService
}

func NewDomainService(connection *Connection, path string) *DomainService {
	var result DomainService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets the authentication domain information.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678
// ....
// Will return the domain information:
// [source,xml]
// ----
// <domain href="/ovirt-engine/api/domains/5678" id="5678">
//   <name>internal-authz</name>
//   <link href="/ovirt-engine/api/domains/5678/users" rel="users"/>
//   <link href="/ovirt-engine/api/domains/5678/groups" rel="groups"/>
//   <link href="/ovirt-engine/api/domains/5678/users?search={query}" rel="users/search"/>
//   <link href="/ovirt-engine/api/domains/5678/groups?search={query}" rel="groups/search"/>
// </domain>
// ----
//
type DomainServiceGetRequest struct {
	DomainService *DomainService
	header        map[string]string
	query         map[string]string
	follow        *string
}

func (p *DomainServiceGetRequest) Header(key, value string) *DomainServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainServiceGetRequest) Query(key, value string) *DomainServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainServiceGetRequest) Follow(follow string) *DomainServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DomainServiceGetRequest) Send() (*DomainServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainService.connection.URL(), p.DomainService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDomainReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DomainServiceGetResponse{domain: result}, nil
}

func (p *DomainServiceGetRequest) MustSend() *DomainServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the authentication domain information.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678
// ....
// Will return the domain information:
// [source,xml]
// ----
// <domain href="/ovirt-engine/api/domains/5678" id="5678">
//   <name>internal-authz</name>
//   <link href="/ovirt-engine/api/domains/5678/users" rel="users"/>
//   <link href="/ovirt-engine/api/domains/5678/groups" rel="groups"/>
//   <link href="/ovirt-engine/api/domains/5678/users?search={query}" rel="users/search"/>
//   <link href="/ovirt-engine/api/domains/5678/groups?search={query}" rel="groups/search"/>
// </domain>
// ----
//
type DomainServiceGetResponse struct {
	domain *Domain
}

func (p *DomainServiceGetResponse) Domain() (*Domain, bool) {
	if p.domain != nil {
		return p.domain, true
	}
	return nil, false
}

func (p *DomainServiceGetResponse) MustDomain() *Domain {
	if p.domain == nil {
		panic("domain in response does not exist")
	}
	return p.domain
}

//
// Gets the authentication domain information.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678
// ....
// Will return the domain information:
// [source,xml]
// ----
// <domain href="/ovirt-engine/api/domains/5678" id="5678">
//   <name>internal-authz</name>
//   <link href="/ovirt-engine/api/domains/5678/users" rel="users"/>
//   <link href="/ovirt-engine/api/domains/5678/groups" rel="groups"/>
//   <link href="/ovirt-engine/api/domains/5678/users?search={query}" rel="users/search"/>
//   <link href="/ovirt-engine/api/domains/5678/groups?search={query}" rel="groups/search"/>
// </domain>
// ----
//
func (p *DomainService) Get() *DomainServiceGetRequest {
	return &DomainServiceGetRequest{DomainService: p}
}

//
// Reference to a service to manage domain groups.
//
func (op *DomainService) GroupsService() *DomainGroupsService {
	return NewDomainGroupsService(op.connection, fmt.Sprintf("%s/groups", op.path))
}

//
// Reference to a service to manage domain users.
//
func (op *DomainService) UsersService() *DomainUsersService {
	return NewDomainUsersService(op.connection, fmt.Sprintf("%s/users", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "groups" {
		return op.GroupsService(), nil
	}
	if strings.HasPrefix(path, "groups/") {
		return op.GroupsService().Service(path[7:])
	}
	if path == "users" {
		return op.UsersService(), nil
	}
	if strings.HasPrefix(path, "users/") {
		return op.UsersService().Service(path[6:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DomainService) String() string {
	return fmt.Sprintf("DomainService:%s", op.path)
}

//
//
type SshPublicKeyService struct {
	BaseService
}

func NewSshPublicKeyService(connection *Connection, path string) *SshPublicKeyService {
	var result SshPublicKeyService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type SshPublicKeyServiceGetRequest struct {
	SshPublicKeyService *SshPublicKeyService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *SshPublicKeyServiceGetRequest) Header(key, value string) *SshPublicKeyServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SshPublicKeyServiceGetRequest) Query(key, value string) *SshPublicKeyServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SshPublicKeyServiceGetRequest) Follow(follow string) *SshPublicKeyServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *SshPublicKeyServiceGetRequest) Send() (*SshPublicKeyServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SshPublicKeyService.connection.URL(), p.SshPublicKeyService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SshPublicKeyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SshPublicKeyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SshPublicKeyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SshPublicKeyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SshPublicKeyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSshPublicKeyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SshPublicKeyServiceGetResponse{key: result}, nil
}

func (p *SshPublicKeyServiceGetRequest) MustSend() *SshPublicKeyServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SshPublicKeyServiceGetResponse struct {
	key *SshPublicKey
}

func (p *SshPublicKeyServiceGetResponse) Key() (*SshPublicKey, bool) {
	if p.key != nil {
		return p.key, true
	}
	return nil, false
}

func (p *SshPublicKeyServiceGetResponse) MustKey() *SshPublicKey {
	if p.key == nil {
		panic("key in response does not exist")
	}
	return p.key
}

//
//
func (p *SshPublicKeyService) Get() *SshPublicKeyServiceGetRequest {
	return &SshPublicKeyServiceGetRequest{SshPublicKeyService: p}
}

//
//
type SshPublicKeyServiceRemoveRequest struct {
	SshPublicKeyService *SshPublicKeyService
	header              map[string]string
	query               map[string]string
	async               *bool
}

func (p *SshPublicKeyServiceRemoveRequest) Header(key, value string) *SshPublicKeyServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SshPublicKeyServiceRemoveRequest) Query(key, value string) *SshPublicKeyServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SshPublicKeyServiceRemoveRequest) Async(async bool) *SshPublicKeyServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *SshPublicKeyServiceRemoveRequest) Send() (*SshPublicKeyServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SshPublicKeyService.connection.URL(), p.SshPublicKeyService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SshPublicKeyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SshPublicKeyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SshPublicKeyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SshPublicKeyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SshPublicKeyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(SshPublicKeyServiceRemoveResponse), nil
}

func (p *SshPublicKeyServiceRemoveRequest) MustSend() *SshPublicKeyServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SshPublicKeyServiceRemoveResponse struct {
}

//
//
func (p *SshPublicKeyService) Remove() *SshPublicKeyServiceRemoveRequest {
	return &SshPublicKeyServiceRemoveRequest{SshPublicKeyService: p}
}

//
//
type SshPublicKeyServiceUpdateRequest struct {
	SshPublicKeyService *SshPublicKeyService
	header              map[string]string
	query               map[string]string
	async               *bool
	key                 *SshPublicKey
}

func (p *SshPublicKeyServiceUpdateRequest) Header(key, value string) *SshPublicKeyServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SshPublicKeyServiceUpdateRequest) Query(key, value string) *SshPublicKeyServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SshPublicKeyServiceUpdateRequest) Async(async bool) *SshPublicKeyServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *SshPublicKeyServiceUpdateRequest) Key(key *SshPublicKey) *SshPublicKeyServiceUpdateRequest {
	p.key = key
	return p
}

func (p *SshPublicKeyServiceUpdateRequest) Send() (*SshPublicKeyServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SshPublicKeyService.connection.URL(), p.SshPublicKeyService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLSshPublicKeyWriteOne(writer, p.key, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SshPublicKeyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SshPublicKeyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SshPublicKeyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SshPublicKeyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SshPublicKeyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSshPublicKeyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SshPublicKeyServiceUpdateResponse{key: result}, nil
}

func (p *SshPublicKeyServiceUpdateRequest) MustSend() *SshPublicKeyServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SshPublicKeyServiceUpdateResponse struct {
	key *SshPublicKey
}

func (p *SshPublicKeyServiceUpdateResponse) Key() (*SshPublicKey, bool) {
	if p.key != nil {
		return p.key, true
	}
	return nil, false
}

func (p *SshPublicKeyServiceUpdateResponse) MustKey() *SshPublicKey {
	if p.key == nil {
		panic("key in response does not exist")
	}
	return p.key
}

//
//
func (p *SshPublicKeyService) Update() *SshPublicKeyServiceUpdateRequest {
	return &SshPublicKeyServiceUpdateRequest{SshPublicKeyService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SshPublicKeyService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *SshPublicKeyService) String() string {
	return fmt.Sprintf("SshPublicKeyService:%s", op.path)
}

//
// A service that shows a user's group membership in the AAA extension.
//
type DomainUserGroupsService struct {
	BaseService
}

func NewDomainUserGroupsService(connection *Connection, path string) *DomainUserGroupsService {
	var result DomainUserGroupsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of groups that the user is a member of.
//
type DomainUserGroupsServiceListRequest struct {
	DomainUserGroupsService *DomainUserGroupsService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
}

func (p *DomainUserGroupsServiceListRequest) Header(key, value string) *DomainUserGroupsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainUserGroupsServiceListRequest) Query(key, value string) *DomainUserGroupsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainUserGroupsServiceListRequest) Follow(follow string) *DomainUserGroupsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DomainUserGroupsServiceListRequest) Send() (*DomainUserGroupsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainUserGroupsService.connection.URL(), p.DomainUserGroupsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainUserGroupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainUserGroupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainUserGroupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainUserGroupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainUserGroupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGroupReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DomainUserGroupsServiceListResponse{groups: result}, nil
}

func (p *DomainUserGroupsServiceListRequest) MustSend() *DomainUserGroupsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of groups that the user is a member of.
//
type DomainUserGroupsServiceListResponse struct {
	groups *GroupSlice
}

func (p *DomainUserGroupsServiceListResponse) Groups() (*GroupSlice, bool) {
	if p.groups != nil {
		return p.groups, true
	}
	return nil, false
}

func (p *DomainUserGroupsServiceListResponse) MustGroups() *GroupSlice {
	if p.groups == nil {
		panic("groups in response does not exist")
	}
	return p.groups
}

//
// Returns the list of groups that the user is a member of.
//
func (p *DomainUserGroupsService) List() *DomainUserGroupsServiceListRequest {
	return &DomainUserGroupsServiceListRequest{DomainUserGroupsService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainUserGroupsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DomainUserGroupsService) String() string {
	return fmt.Sprintf("DomainUserGroupsService:%s", op.path)
}

//
// Manages the collection of groups of users.
//
type GroupsService struct {
	BaseService
}

func NewGroupsService(connection *Connection, path string) *GroupsService {
	var result GroupsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add group from a directory service. Please note that domain name is name of the authorization provider.
// For example, to add the `Developers` group from the `internal-authz` authorization provider send a request
// like this:
// [source]
// ----
// POST /ovirt-engine/api/groups
// ----
// With a request body like this:
// [source,xml]
// ----
// <group>
//   <name>Developers</name>
//   <domain>
//     <name>internal-authz</name>
//   </domain>
// </group>
// ----
//
type GroupsServiceAddRequest struct {
	GroupsService *GroupsService
	header        map[string]string
	query         map[string]string
	group         *Group
}

func (p *GroupsServiceAddRequest) Header(key, value string) *GroupsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GroupsServiceAddRequest) Query(key, value string) *GroupsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GroupsServiceAddRequest) Group(group *Group) *GroupsServiceAddRequest {
	p.group = group
	return p
}

func (p *GroupsServiceAddRequest) Send() (*GroupsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GroupsService.connection.URL(), p.GroupsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLGroupWriteOne(writer, p.group, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GroupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GroupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GroupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GroupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GroupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGroupReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &GroupsServiceAddResponse{group: result}, nil
}

func (p *GroupsServiceAddRequest) MustSend() *GroupsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add group from a directory service. Please note that domain name is name of the authorization provider.
// For example, to add the `Developers` group from the `internal-authz` authorization provider send a request
// like this:
// [source]
// ----
// POST /ovirt-engine/api/groups
// ----
// With a request body like this:
// [source,xml]
// ----
// <group>
//   <name>Developers</name>
//   <domain>
//     <name>internal-authz</name>
//   </domain>
// </group>
// ----
//
type GroupsServiceAddResponse struct {
	group *Group
}

func (p *GroupsServiceAddResponse) Group() (*Group, bool) {
	if p.group != nil {
		return p.group, true
	}
	return nil, false
}

func (p *GroupsServiceAddResponse) MustGroup() *Group {
	if p.group == nil {
		panic("group in response does not exist")
	}
	return p.group
}

//
// Add group from a directory service. Please note that domain name is name of the authorization provider.
// For example, to add the `Developers` group from the `internal-authz` authorization provider send a request
// like this:
// [source]
// ----
// POST /ovirt-engine/api/groups
// ----
// With a request body like this:
// [source,xml]
// ----
// <group>
//   <name>Developers</name>
//   <domain>
//     <name>internal-authz</name>
//   </domain>
// </group>
// ----
//
func (p *GroupsService) Add() *GroupsServiceAddRequest {
	return &GroupsServiceAddRequest{GroupsService: p}
}

//
// List all the groups in the system.
// Usage:
// ....
// GET /ovirt-engine/api/groups
// ....
// Will return the list of groups:
// [source,xml]
// ----
// <groups>
//   <group href="/ovirt-engine/api/groups/123" id="123">
//     <name>mygroup</name>
//     <link href="/ovirt-engine/api/groups/123/roles" rel="roles"/>
//     <link href="/ovirt-engine/api/groups/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/groups/123/tags" rel="tags"/>
//     <domain_entry_id>476652557A382F67696B6D2B32762B37796E46476D513D3D</domain_entry_id>
//     <namespace>DC=example,DC=com</namespace>
//     <domain href="/ovirt-engine/api/domains/ABCDEF" id="ABCDEF">
//       <name>myextension-authz</name>
//     </domain>
//   </group>
//   ...
// </groups>
// ----
// The order of the returned list of groups isn't guaranteed.
//
type GroupsServiceListRequest struct {
	GroupsService *GroupsService
	header        map[string]string
	query         map[string]string
	caseSensitive *bool
	follow        *string
	max           *int64
	search        *string
}

func (p *GroupsServiceListRequest) Header(key, value string) *GroupsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GroupsServiceListRequest) Query(key, value string) *GroupsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GroupsServiceListRequest) CaseSensitive(caseSensitive bool) *GroupsServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *GroupsServiceListRequest) Follow(follow string) *GroupsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *GroupsServiceListRequest) Max(max int64) *GroupsServiceListRequest {
	p.max = &max
	return p
}

func (p *GroupsServiceListRequest) Search(search string) *GroupsServiceListRequest {
	p.search = &search
	return p
}

func (p *GroupsServiceListRequest) Send() (*GroupsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GroupsService.connection.URL(), p.GroupsService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GroupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GroupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GroupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GroupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GroupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGroupReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &GroupsServiceListResponse{groups: result}, nil
}

func (p *GroupsServiceListRequest) MustSend() *GroupsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all the groups in the system.
// Usage:
// ....
// GET /ovirt-engine/api/groups
// ....
// Will return the list of groups:
// [source,xml]
// ----
// <groups>
//   <group href="/ovirt-engine/api/groups/123" id="123">
//     <name>mygroup</name>
//     <link href="/ovirt-engine/api/groups/123/roles" rel="roles"/>
//     <link href="/ovirt-engine/api/groups/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/groups/123/tags" rel="tags"/>
//     <domain_entry_id>476652557A382F67696B6D2B32762B37796E46476D513D3D</domain_entry_id>
//     <namespace>DC=example,DC=com</namespace>
//     <domain href="/ovirt-engine/api/domains/ABCDEF" id="ABCDEF">
//       <name>myextension-authz</name>
//     </domain>
//   </group>
//   ...
// </groups>
// ----
// The order of the returned list of groups isn't guaranteed.
//
type GroupsServiceListResponse struct {
	groups *GroupSlice
}

func (p *GroupsServiceListResponse) Groups() (*GroupSlice, bool) {
	if p.groups != nil {
		return p.groups, true
	}
	return nil, false
}

func (p *GroupsServiceListResponse) MustGroups() *GroupSlice {
	if p.groups == nil {
		panic("groups in response does not exist")
	}
	return p.groups
}

//
// List all the groups in the system.
// Usage:
// ....
// GET /ovirt-engine/api/groups
// ....
// Will return the list of groups:
// [source,xml]
// ----
// <groups>
//   <group href="/ovirt-engine/api/groups/123" id="123">
//     <name>mygroup</name>
//     <link href="/ovirt-engine/api/groups/123/roles" rel="roles"/>
//     <link href="/ovirt-engine/api/groups/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/groups/123/tags" rel="tags"/>
//     <domain_entry_id>476652557A382F67696B6D2B32762B37796E46476D513D3D</domain_entry_id>
//     <namespace>DC=example,DC=com</namespace>
//     <domain href="/ovirt-engine/api/domains/ABCDEF" id="ABCDEF">
//       <name>myextension-authz</name>
//     </domain>
//   </group>
//   ...
// </groups>
// ----
// The order of the returned list of groups isn't guaranteed.
//
func (p *GroupsService) List() *GroupsServiceListRequest {
	return &GroupsServiceListRequest{GroupsService: p}
}

//
// Reference to the service that manages a specific group.
//
func (op *GroupsService) GroupService(id string) *GroupService {
	return NewGroupService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GroupsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.GroupService(path), nil
	}
	return op.GroupService(path[:index]).Service(path[index+1:])
}

func (op *GroupsService) String() string {
	return fmt.Sprintf("GroupsService:%s", op.path)
}

//
// A service to manage the users in the system.
//
type UsersService struct {
	BaseService
}

func NewUsersService(connection *Connection, path string) *UsersService {
	var result UsersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add user from a directory service.
// For example, to add the `myuser` user from the `myextension-authz` authorization provider send a request
// like this:
// [source]
// ----
// POST /ovirt-engine/api/users
// ----
// With a request body like this:
// [source,xml]
// ----
// <user>
//   <user_name>myuser@myextension-authz</user_name>
//   <domain>
//     <name>myextension-authz</name>
//   </domain>
// </user>
// ----
// In case you are working with Active Directory you have to pass user principal name (UPN) as `username`, followed
// by authorization provider name. Due to https://bugzilla.redhat.com/1147900[bug 1147900] you need to provide
// also `principal` parameter set to UPN of the user.
// For example, to add the user with UPN `myuser@mysubdomain.mydomain.com` from the `myextension-authz`
// authorization provider send a request body like this:
// [source,xml]
// ----
// <user>
//   <principal>myuser@mysubdomain.mydomain.com</principal>
//   <user_name>myuser@mysubdomain.mydomain.com@myextension-authz</user_name>
//   <domain>
//     <name>myextension-authz</name>
//   </domain>
// </user>
// ----
//
type UsersServiceAddRequest struct {
	UsersService *UsersService
	header       map[string]string
	query        map[string]string
	user         *User
}

func (p *UsersServiceAddRequest) Header(key, value string) *UsersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *UsersServiceAddRequest) Query(key, value string) *UsersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *UsersServiceAddRequest) User(user *User) *UsersServiceAddRequest {
	p.user = user
	return p
}

func (p *UsersServiceAddRequest) Send() (*UsersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.UsersService.connection.URL(), p.UsersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLUserWriteOne(writer, p.user, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.UsersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.UsersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.UsersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.UsersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.UsersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLUserReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &UsersServiceAddResponse{user: result}, nil
}

func (p *UsersServiceAddRequest) MustSend() *UsersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add user from a directory service.
// For example, to add the `myuser` user from the `myextension-authz` authorization provider send a request
// like this:
// [source]
// ----
// POST /ovirt-engine/api/users
// ----
// With a request body like this:
// [source,xml]
// ----
// <user>
//   <user_name>myuser@myextension-authz</user_name>
//   <domain>
//     <name>myextension-authz</name>
//   </domain>
// </user>
// ----
// In case you are working with Active Directory you have to pass user principal name (UPN) as `username`, followed
// by authorization provider name. Due to https://bugzilla.redhat.com/1147900[bug 1147900] you need to provide
// also `principal` parameter set to UPN of the user.
// For example, to add the user with UPN `myuser@mysubdomain.mydomain.com` from the `myextension-authz`
// authorization provider send a request body like this:
// [source,xml]
// ----
// <user>
//   <principal>myuser@mysubdomain.mydomain.com</principal>
//   <user_name>myuser@mysubdomain.mydomain.com@myextension-authz</user_name>
//   <domain>
//     <name>myextension-authz</name>
//   </domain>
// </user>
// ----
//
type UsersServiceAddResponse struct {
	user *User
}

func (p *UsersServiceAddResponse) User() (*User, bool) {
	if p.user != nil {
		return p.user, true
	}
	return nil, false
}

func (p *UsersServiceAddResponse) MustUser() *User {
	if p.user == nil {
		panic("user in response does not exist")
	}
	return p.user
}

//
// Add user from a directory service.
// For example, to add the `myuser` user from the `myextension-authz` authorization provider send a request
// like this:
// [source]
// ----
// POST /ovirt-engine/api/users
// ----
// With a request body like this:
// [source,xml]
// ----
// <user>
//   <user_name>myuser@myextension-authz</user_name>
//   <domain>
//     <name>myextension-authz</name>
//   </domain>
// </user>
// ----
// In case you are working with Active Directory you have to pass user principal name (UPN) as `username`, followed
// by authorization provider name. Due to https://bugzilla.redhat.com/1147900[bug 1147900] you need to provide
// also `principal` parameter set to UPN of the user.
// For example, to add the user with UPN `myuser@mysubdomain.mydomain.com` from the `myextension-authz`
// authorization provider send a request body like this:
// [source,xml]
// ----
// <user>
//   <principal>myuser@mysubdomain.mydomain.com</principal>
//   <user_name>myuser@mysubdomain.mydomain.com@myextension-authz</user_name>
//   <domain>
//     <name>myextension-authz</name>
//   </domain>
// </user>
// ----
//
func (p *UsersService) Add() *UsersServiceAddRequest {
	return &UsersServiceAddRequest{UsersService: p}
}

//
// List all the users in the system.
// Usage:
// ....
// GET /ovirt-engine/api/users
// ....
// Will return the list of users:
// [source,xml]
// ----
// <users>
//   <user href="/ovirt-engine/api/users/1234" id="1234">
//     <name>admin</name>
//     <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
//     <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
//     <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
//     <domain_entry_id>23456</domain_entry_id>
//     <namespace>*</namespace>
//     <principal>user1</principal>
//     <user_name>user1@domain-authz</user_name>
//     <domain href="/ovirt-engine/api/domains/45678" id="45678">
//       <name>domain-authz</name>
//     </domain>
//   </user>
// </users>
// ----
// The order of the returned list of users isn't guaranteed.
//
type UsersServiceListRequest struct {
	UsersService  *UsersService
	header        map[string]string
	query         map[string]string
	caseSensitive *bool
	follow        *string
	max           *int64
	search        *string
}

func (p *UsersServiceListRequest) Header(key, value string) *UsersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *UsersServiceListRequest) Query(key, value string) *UsersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *UsersServiceListRequest) CaseSensitive(caseSensitive bool) *UsersServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *UsersServiceListRequest) Follow(follow string) *UsersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *UsersServiceListRequest) Max(max int64) *UsersServiceListRequest {
	p.max = &max
	return p
}

func (p *UsersServiceListRequest) Search(search string) *UsersServiceListRequest {
	p.search = &search
	return p
}

func (p *UsersServiceListRequest) Send() (*UsersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.UsersService.connection.URL(), p.UsersService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.UsersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.UsersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.UsersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.UsersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.UsersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLUserReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &UsersServiceListResponse{users: result}, nil
}

func (p *UsersServiceListRequest) MustSend() *UsersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all the users in the system.
// Usage:
// ....
// GET /ovirt-engine/api/users
// ....
// Will return the list of users:
// [source,xml]
// ----
// <users>
//   <user href="/ovirt-engine/api/users/1234" id="1234">
//     <name>admin</name>
//     <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
//     <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
//     <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
//     <domain_entry_id>23456</domain_entry_id>
//     <namespace>*</namespace>
//     <principal>user1</principal>
//     <user_name>user1@domain-authz</user_name>
//     <domain href="/ovirt-engine/api/domains/45678" id="45678">
//       <name>domain-authz</name>
//     </domain>
//   </user>
// </users>
// ----
// The order of the returned list of users isn't guaranteed.
//
type UsersServiceListResponse struct {
	users *UserSlice
}

func (p *UsersServiceListResponse) Users() (*UserSlice, bool) {
	if p.users != nil {
		return p.users, true
	}
	return nil, false
}

func (p *UsersServiceListResponse) MustUsers() *UserSlice {
	if p.users == nil {
		panic("users in response does not exist")
	}
	return p.users
}

//
// List all the users in the system.
// Usage:
// ....
// GET /ovirt-engine/api/users
// ....
// Will return the list of users:
// [source,xml]
// ----
// <users>
//   <user href="/ovirt-engine/api/users/1234" id="1234">
//     <name>admin</name>
//     <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
//     <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
//     <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
//     <domain_entry_id>23456</domain_entry_id>
//     <namespace>*</namespace>
//     <principal>user1</principal>
//     <user_name>user1@domain-authz</user_name>
//     <domain href="/ovirt-engine/api/domains/45678" id="45678">
//       <name>domain-authz</name>
//     </domain>
//   </user>
// </users>
// ----
// The order of the returned list of users isn't guaranteed.
//
func (p *UsersService) List() *UsersServiceListRequest {
	return &UsersServiceListRequest{UsersService: p}
}

//
//
func (op *UsersService) UserService(id string) *UserService {
	return NewUserService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *UsersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.UserService(path), nil
	}
	return op.UserService(path[:index]).Service(path[index+1:])
}

func (op *UsersService) String() string {
	return fmt.Sprintf("UsersService:%s", op.path)
}

//
// A service to list all domain users in the system.
//
type DomainUsersService struct {
	BaseService
}

func NewDomainUsersService(connection *Connection, path string) *DomainUsersService {
	var result DomainUsersService
	result.connection = connection
	result.path = path
	return &result
}

//
// List all the users in the domain.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678/users
// ....
// Will return the list of users in the domain:
// [source,xml]
// ----
// <users>
//   <user href="/ovirt-engine/api/domains/5678/users/1234" id="1234">
//     <name>admin</name>
//     <namespace>*</namespace>
//     <principal>admin</principal>
//     <user_name>admin@internal-authz</user_name>
//     <domain href="/ovirt-engine/api/domains/5678" id="5678">
//       <name>internal-authz</name>
//     </domain>
//     <groups/>
//   </user>
// </users>
// ----
// The order of the returned list of users isn't guaranteed.
//
type DomainUsersServiceListRequest struct {
	DomainUsersService *DomainUsersService
	header             map[string]string
	query              map[string]string
	caseSensitive      *bool
	follow             *string
	max                *int64
	search             *string
}

func (p *DomainUsersServiceListRequest) Header(key, value string) *DomainUsersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DomainUsersServiceListRequest) Query(key, value string) *DomainUsersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DomainUsersServiceListRequest) CaseSensitive(caseSensitive bool) *DomainUsersServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *DomainUsersServiceListRequest) Follow(follow string) *DomainUsersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DomainUsersServiceListRequest) Max(max int64) *DomainUsersServiceListRequest {
	p.max = &max
	return p
}

func (p *DomainUsersServiceListRequest) Search(search string) *DomainUsersServiceListRequest {
	p.search = &search
	return p
}

func (p *DomainUsersServiceListRequest) Send() (*DomainUsersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DomainUsersService.connection.URL(), p.DomainUsersService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DomainUsersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DomainUsersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DomainUsersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DomainUsersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DomainUsersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLUserReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DomainUsersServiceListResponse{users: result}, nil
}

func (p *DomainUsersServiceListRequest) MustSend() *DomainUsersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all the users in the domain.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678/users
// ....
// Will return the list of users in the domain:
// [source,xml]
// ----
// <users>
//   <user href="/ovirt-engine/api/domains/5678/users/1234" id="1234">
//     <name>admin</name>
//     <namespace>*</namespace>
//     <principal>admin</principal>
//     <user_name>admin@internal-authz</user_name>
//     <domain href="/ovirt-engine/api/domains/5678" id="5678">
//       <name>internal-authz</name>
//     </domain>
//     <groups/>
//   </user>
// </users>
// ----
// The order of the returned list of users isn't guaranteed.
//
type DomainUsersServiceListResponse struct {
	users *UserSlice
}

func (p *DomainUsersServiceListResponse) Users() (*UserSlice, bool) {
	if p.users != nil {
		return p.users, true
	}
	return nil, false
}

func (p *DomainUsersServiceListResponse) MustUsers() *UserSlice {
	if p.users == nil {
		panic("users in response does not exist")
	}
	return p.users
}

//
// List all the users in the domain.
// Usage:
// ....
// GET /ovirt-engine/api/domains/5678/users
// ....
// Will return the list of users in the domain:
// [source,xml]
// ----
// <users>
//   <user href="/ovirt-engine/api/domains/5678/users/1234" id="1234">
//     <name>admin</name>
//     <namespace>*</namespace>
//     <principal>admin</principal>
//     <user_name>admin@internal-authz</user_name>
//     <domain href="/ovirt-engine/api/domains/5678" id="5678">
//       <name>internal-authz</name>
//     </domain>
//     <groups/>
//   </user>
// </users>
// ----
// The order of the returned list of users isn't guaranteed.
//
func (p *DomainUsersService) List() *DomainUsersServiceListRequest {
	return &DomainUsersServiceListRequest{DomainUsersService: p}
}

//
// Reference to a service to view details of a domain user.
//
func (op *DomainUsersService) UserService(id string) *DomainUserService {
	return NewDomainUserService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DomainUsersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.UserService(path), nil
	}
	return op.UserService(path[:index]).Service(path[index+1:])
}

func (op *DomainUsersService) String() string {
	return fmt.Sprintf("DomainUsersService:%s", op.path)
}

//
// A service to manage host storages.
//
type HostStorageService struct {
	BaseService
}

func NewHostStorageService(connection *Connection, path string) *HostStorageService {
	var result HostStorageService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get list of storages.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/storage
// ----
// The XML response you get will be like this one:
// [source,xml]
// ----
// <host_storages>
//   <host_storage id="123">
//     ...
//   </host_storage>
//   ...
// </host_storages>
// ----
// The order of the returned list of storages isn't guaranteed.
//
type HostStorageServiceListRequest struct {
	HostStorageService *HostStorageService
	header             map[string]string
	query              map[string]string
	follow             *string
	reportStatus       *bool
}

func (p *HostStorageServiceListRequest) Header(key, value string) *HostStorageServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostStorageServiceListRequest) Query(key, value string) *HostStorageServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostStorageServiceListRequest) Follow(follow string) *HostStorageServiceListRequest {
	p.follow = &follow
	return p
}

func (p *HostStorageServiceListRequest) ReportStatus(reportStatus bool) *HostStorageServiceListRequest {
	p.reportStatus = &reportStatus
	return p
}

func (p *HostStorageServiceListRequest) Send() (*HostStorageServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostStorageService.connection.URL(), p.HostStorageService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.reportStatus != nil {
		values["report_status"] = []string{fmt.Sprintf("%v", *p.reportStatus)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostStorageService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostStorageService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostStorageService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostStorageService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostStorageService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostStorageReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &HostStorageServiceListResponse{storages: result}, nil
}

func (p *HostStorageServiceListRequest) MustSend() *HostStorageServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get list of storages.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/storage
// ----
// The XML response you get will be like this one:
// [source,xml]
// ----
// <host_storages>
//   <host_storage id="123">
//     ...
//   </host_storage>
//   ...
// </host_storages>
// ----
// The order of the returned list of storages isn't guaranteed.
//
type HostStorageServiceListResponse struct {
	storages *HostStorageSlice
}

func (p *HostStorageServiceListResponse) Storages() (*HostStorageSlice, bool) {
	if p.storages != nil {
		return p.storages, true
	}
	return nil, false
}

func (p *HostStorageServiceListResponse) MustStorages() *HostStorageSlice {
	if p.storages == nil {
		panic("storages in response does not exist")
	}
	return p.storages
}

//
// Get list of storages.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/storage
// ----
// The XML response you get will be like this one:
// [source,xml]
// ----
// <host_storages>
//   <host_storage id="123">
//     ...
//   </host_storage>
//   ...
// </host_storages>
// ----
// The order of the returned list of storages isn't guaranteed.
//
func (p *HostStorageService) List() *HostStorageServiceListRequest {
	return &HostStorageServiceListRequest{HostStorageService: p}
}

//
// Reference to a service managing the storage.
//
func (op *HostStorageService) StorageService(id string) *StorageService {
	return NewStorageService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostStorageService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.StorageService(path), nil
	}
	return op.StorageService(path[:index]).Service(path[index+1:])
}

func (op *HostStorageService) String() string {
	return fmt.Sprintf("HostStorageService:%s", op.path)
}

//
//
type DiskProfileService struct {
	BaseService
}

func NewDiskProfileService(connection *Connection, path string) *DiskProfileService {
	var result DiskProfileService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type DiskProfileServiceGetRequest struct {
	DiskProfileService *DiskProfileService
	header             map[string]string
	query              map[string]string
	follow             *string
}

func (p *DiskProfileServiceGetRequest) Header(key, value string) *DiskProfileServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskProfileServiceGetRequest) Query(key, value string) *DiskProfileServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskProfileServiceGetRequest) Follow(follow string) *DiskProfileServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DiskProfileServiceGetRequest) Send() (*DiskProfileServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskProfileService.connection.URL(), p.DiskProfileService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskProfileServiceGetResponse{profile: result}, nil
}

func (p *DiskProfileServiceGetRequest) MustSend() *DiskProfileServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskProfileServiceGetResponse struct {
	profile *DiskProfile
}

func (p *DiskProfileServiceGetResponse) Profile() (*DiskProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *DiskProfileServiceGetResponse) MustProfile() *DiskProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
//
func (p *DiskProfileService) Get() *DiskProfileServiceGetRequest {
	return &DiskProfileServiceGetRequest{DiskProfileService: p}
}

//
//
type DiskProfileServiceRemoveRequest struct {
	DiskProfileService *DiskProfileService
	header             map[string]string
	query              map[string]string
	async              *bool
}

func (p *DiskProfileServiceRemoveRequest) Header(key, value string) *DiskProfileServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskProfileServiceRemoveRequest) Query(key, value string) *DiskProfileServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskProfileServiceRemoveRequest) Async(async bool) *DiskProfileServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *DiskProfileServiceRemoveRequest) Send() (*DiskProfileServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskProfileService.connection.URL(), p.DiskProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(DiskProfileServiceRemoveResponse), nil
}

func (p *DiskProfileServiceRemoveRequest) MustSend() *DiskProfileServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskProfileServiceRemoveResponse struct {
}

//
//
func (p *DiskProfileService) Remove() *DiskProfileServiceRemoveRequest {
	return &DiskProfileServiceRemoveRequest{DiskProfileService: p}
}

//
// Update the specified disk profile in the system.
//
type DiskProfileServiceUpdateRequest struct {
	DiskProfileService *DiskProfileService
	header             map[string]string
	query              map[string]string
	async              *bool
	profile            *DiskProfile
}

func (p *DiskProfileServiceUpdateRequest) Header(key, value string) *DiskProfileServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskProfileServiceUpdateRequest) Query(key, value string) *DiskProfileServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskProfileServiceUpdateRequest) Async(async bool) *DiskProfileServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *DiskProfileServiceUpdateRequest) Profile(profile *DiskProfile) *DiskProfileServiceUpdateRequest {
	p.profile = profile
	return p
}

func (p *DiskProfileServiceUpdateRequest) Send() (*DiskProfileServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskProfileService.connection.URL(), p.DiskProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskProfileWriteOne(writer, p.profile, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskProfileServiceUpdateResponse{profile: result}, nil
}

func (p *DiskProfileServiceUpdateRequest) MustSend() *DiskProfileServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified disk profile in the system.
//
type DiskProfileServiceUpdateResponse struct {
	profile *DiskProfile
}

func (p *DiskProfileServiceUpdateResponse) Profile() (*DiskProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *DiskProfileServiceUpdateResponse) MustProfile() *DiskProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Update the specified disk profile in the system.
//
func (p *DiskProfileService) Update() *DiskProfileServiceUpdateRequest {
	return &DiskProfileServiceUpdateRequest{DiskProfileService: p}
}

//
//
func (op *DiskProfileService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DiskProfileService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DiskProfileService) String() string {
	return fmt.Sprintf("DiskProfileService:%s", op.path)
}

//
//
type WeightsService struct {
	BaseService
}

func NewWeightsService(connection *Connection, path string) *WeightsService {
	var result WeightsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a weight to a specified user defined scheduling policy.
//
type WeightsServiceAddRequest struct {
	WeightsService *WeightsService
	header         map[string]string
	query          map[string]string
	weight         *Weight
}

func (p *WeightsServiceAddRequest) Header(key, value string) *WeightsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *WeightsServiceAddRequest) Query(key, value string) *WeightsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *WeightsServiceAddRequest) Weight(weight *Weight) *WeightsServiceAddRequest {
	p.weight = weight
	return p
}

func (p *WeightsServiceAddRequest) Send() (*WeightsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.WeightsService.connection.URL(), p.WeightsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLWeightWriteOne(writer, p.weight, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.WeightsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.WeightsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.WeightsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.WeightsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.WeightsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLWeightReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &WeightsServiceAddResponse{weight: result}, nil
}

func (p *WeightsServiceAddRequest) MustSend() *WeightsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a weight to a specified user defined scheduling policy.
//
type WeightsServiceAddResponse struct {
	weight *Weight
}

func (p *WeightsServiceAddResponse) Weight() (*Weight, bool) {
	if p.weight != nil {
		return p.weight, true
	}
	return nil, false
}

func (p *WeightsServiceAddResponse) MustWeight() *Weight {
	if p.weight == nil {
		panic("weight in response does not exist")
	}
	return p.weight
}

//
// Add a weight to a specified user defined scheduling policy.
//
func (p *WeightsService) Add() *WeightsServiceAddRequest {
	return &WeightsServiceAddRequest{WeightsService: p}
}

//
// Returns the list of weights.
// The order of the returned list of weights isn't guaranteed.
//
type WeightsServiceListRequest struct {
	WeightsService *WeightsService
	header         map[string]string
	query          map[string]string
	filter         *bool
	follow         *string
	max            *int64
}

func (p *WeightsServiceListRequest) Header(key, value string) *WeightsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *WeightsServiceListRequest) Query(key, value string) *WeightsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *WeightsServiceListRequest) Filter(filter bool) *WeightsServiceListRequest {
	p.filter = &filter
	return p
}

func (p *WeightsServiceListRequest) Follow(follow string) *WeightsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *WeightsServiceListRequest) Max(max int64) *WeightsServiceListRequest {
	p.max = &max
	return p
}

func (p *WeightsServiceListRequest) Send() (*WeightsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.WeightsService.connection.URL(), p.WeightsService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.WeightsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.WeightsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.WeightsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.WeightsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.WeightsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLWeightReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &WeightsServiceListResponse{weights: result}, nil
}

func (p *WeightsServiceListRequest) MustSend() *WeightsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of weights.
// The order of the returned list of weights isn't guaranteed.
//
type WeightsServiceListResponse struct {
	weights *WeightSlice
}

func (p *WeightsServiceListResponse) Weights() (*WeightSlice, bool) {
	if p.weights != nil {
		return p.weights, true
	}
	return nil, false
}

func (p *WeightsServiceListResponse) MustWeights() *WeightSlice {
	if p.weights == nil {
		panic("weights in response does not exist")
	}
	return p.weights
}

//
// Returns the list of weights.
// The order of the returned list of weights isn't guaranteed.
//
func (p *WeightsService) List() *WeightsServiceListRequest {
	return &WeightsServiceListRequest{WeightsService: p}
}

//
//
func (op *WeightsService) WeightService(id string) *WeightService {
	return NewWeightService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *WeightsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.WeightService(path), nil
	}
	return op.WeightService(path[:index]).Service(path[index+1:])
}

func (op *WeightsService) String() string {
	return fmt.Sprintf("WeightsService:%s", op.path)
}

//
// Manages the set of quotas configured for a data center.
//
type QuotasService struct {
	BaseService
}

func NewQuotasService(connection *Connection, path string) *QuotasService {
	var result QuotasService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new quota.
// An example of creating a new quota:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/quotas
// ----
// [source,xml]
// ----
// <quota>
//   <name>myquota</name>
//   <description>My new quota for virtual machines</description>
// </quota>
// ----
//
type QuotasServiceAddRequest struct {
	QuotasService *QuotasService
	header        map[string]string
	query         map[string]string
	quota         *Quota
}

func (p *QuotasServiceAddRequest) Header(key, value string) *QuotasServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *QuotasServiceAddRequest) Query(key, value string) *QuotasServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *QuotasServiceAddRequest) Quota(quota *Quota) *QuotasServiceAddRequest {
	p.quota = quota
	return p
}

func (p *QuotasServiceAddRequest) Send() (*QuotasServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.QuotasService.connection.URL(), p.QuotasService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLQuotaWriteOne(writer, p.quota, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.QuotasService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.QuotasService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.QuotasService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.QuotasService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.QuotasService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLQuotaReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &QuotasServiceAddResponse{quota: result}, nil
}

func (p *QuotasServiceAddRequest) MustSend() *QuotasServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new quota.
// An example of creating a new quota:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/quotas
// ----
// [source,xml]
// ----
// <quota>
//   <name>myquota</name>
//   <description>My new quota for virtual machines</description>
// </quota>
// ----
//
type QuotasServiceAddResponse struct {
	quota *Quota
}

func (p *QuotasServiceAddResponse) Quota() (*Quota, bool) {
	if p.quota != nil {
		return p.quota, true
	}
	return nil, false
}

func (p *QuotasServiceAddResponse) MustQuota() *Quota {
	if p.quota == nil {
		panic("quota in response does not exist")
	}
	return p.quota
}

//
// Creates a new quota.
// An example of creating a new quota:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/quotas
// ----
// [source,xml]
// ----
// <quota>
//   <name>myquota</name>
//   <description>My new quota for virtual machines</description>
// </quota>
// ----
//
func (p *QuotasService) Add() *QuotasServiceAddRequest {
	return &QuotasServiceAddRequest{QuotasService: p}
}

//
// Lists quotas of a data center.
// The order of the returned list of quotas isn't guaranteed.
//
type QuotasServiceListRequest struct {
	QuotasService *QuotasService
	header        map[string]string
	query         map[string]string
	follow        *string
	max           *int64
}

func (p *QuotasServiceListRequest) Header(key, value string) *QuotasServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *QuotasServiceListRequest) Query(key, value string) *QuotasServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *QuotasServiceListRequest) Follow(follow string) *QuotasServiceListRequest {
	p.follow = &follow
	return p
}

func (p *QuotasServiceListRequest) Max(max int64) *QuotasServiceListRequest {
	p.max = &max
	return p
}

func (p *QuotasServiceListRequest) Send() (*QuotasServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.QuotasService.connection.URL(), p.QuotasService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.QuotasService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.QuotasService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.QuotasService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.QuotasService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.QuotasService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLQuotaReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &QuotasServiceListResponse{quotas: result}, nil
}

func (p *QuotasServiceListRequest) MustSend() *QuotasServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists quotas of a data center.
// The order of the returned list of quotas isn't guaranteed.
//
type QuotasServiceListResponse struct {
	quotas *QuotaSlice
}

func (p *QuotasServiceListResponse) Quotas() (*QuotaSlice, bool) {
	if p.quotas != nil {
		return p.quotas, true
	}
	return nil, false
}

func (p *QuotasServiceListResponse) MustQuotas() *QuotaSlice {
	if p.quotas == nil {
		panic("quotas in response does not exist")
	}
	return p.quotas
}

//
// Lists quotas of a data center.
// The order of the returned list of quotas isn't guaranteed.
//
func (p *QuotasService) List() *QuotasServiceListRequest {
	return &QuotasServiceListRequest{QuotasService: p}
}

//
//
func (op *QuotasService) QuotaService(id string) *QuotaService {
	return NewQuotaService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *QuotasService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.QuotaService(path), nil
	}
	return op.QuotaService(path[:index]).Service(path[index+1:])
}

func (op *QuotasService) String() string {
	return fmt.Sprintf("QuotasService:%s", op.path)
}

//
// Manages the set of CD-ROM devices of a virtual machine snapshot.
//
type SnapshotCdromsService struct {
	BaseService
}

func NewSnapshotCdromsService(connection *Connection, path string) *SnapshotCdromsService {
	var result SnapshotCdromsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of CD-ROM devices of the snapshot.
// The order of the returned list of CD-ROM devices isn't guaranteed.
//
type SnapshotCdromsServiceListRequest struct {
	SnapshotCdromsService *SnapshotCdromsService
	header                map[string]string
	query                 map[string]string
	follow                *string
	max                   *int64
}

func (p *SnapshotCdromsServiceListRequest) Header(key, value string) *SnapshotCdromsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SnapshotCdromsServiceListRequest) Query(key, value string) *SnapshotCdromsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SnapshotCdromsServiceListRequest) Follow(follow string) *SnapshotCdromsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *SnapshotCdromsServiceListRequest) Max(max int64) *SnapshotCdromsServiceListRequest {
	p.max = &max
	return p
}

func (p *SnapshotCdromsServiceListRequest) Send() (*SnapshotCdromsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SnapshotCdromsService.connection.URL(), p.SnapshotCdromsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SnapshotCdromsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SnapshotCdromsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SnapshotCdromsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SnapshotCdromsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SnapshotCdromsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCdromReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &SnapshotCdromsServiceListResponse{cdroms: result}, nil
}

func (p *SnapshotCdromsServiceListRequest) MustSend() *SnapshotCdromsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of CD-ROM devices of the snapshot.
// The order of the returned list of CD-ROM devices isn't guaranteed.
//
type SnapshotCdromsServiceListResponse struct {
	cdroms *CdromSlice
}

func (p *SnapshotCdromsServiceListResponse) Cdroms() (*CdromSlice, bool) {
	if p.cdroms != nil {
		return p.cdroms, true
	}
	return nil, false
}

func (p *SnapshotCdromsServiceListResponse) MustCdroms() *CdromSlice {
	if p.cdroms == nil {
		panic("cdroms in response does not exist")
	}
	return p.cdroms
}

//
// Returns the list of CD-ROM devices of the snapshot.
// The order of the returned list of CD-ROM devices isn't guaranteed.
//
func (p *SnapshotCdromsService) List() *SnapshotCdromsServiceListRequest {
	return &SnapshotCdromsServiceListRequest{SnapshotCdromsService: p}
}

//
//
func (op *SnapshotCdromsService) CdromService(id string) *SnapshotCdromService {
	return NewSnapshotCdromService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SnapshotCdromsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.CdromService(path), nil
	}
	return op.CdromService(path[:index]).Service(path[index+1:])
}

func (op *SnapshotCdromsService) String() string {
	return fmt.Sprintf("SnapshotCdromsService:%s", op.path)
}

//
// A service to manage individual host device attached to a virtual machine.
//
type VmHostDeviceService struct {
	BaseService
}

func NewVmHostDeviceService(connection *Connection, path string) *VmHostDeviceService {
	var result VmHostDeviceService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieve information about particular host device attached to given virtual machine.
// Example:
// [source]
// ----
// GET /ovirt-engine/api/vms/123/hostdevices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/543/devices/456" id="456">
//   <name>pci_0000_04_00_0</name>
//   <capability>pci</capability>
//   <iommu_group>30</iommu_group>
//   <placeholder>true</placeholder>
//   <product id="0x13ba">
//     <name>GM107GL [Quadro K2200]</name>
//   </product>
//   <vendor id="0x10de">
//     <name>NVIDIA Corporation</name>
//   </vendor>
//   <host href="/ovirt-engine/api/hosts/543" id="543"/>
//   <parent_device href="/ovirt-engine/api/hosts/543/devices/456" id="456">
//     <name>pci_0000_00_03_0</name>
//   </parent_device>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
// </host_device>
// ----
//
type VmHostDeviceServiceGetRequest struct {
	VmHostDeviceService *VmHostDeviceService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *VmHostDeviceServiceGetRequest) Header(key, value string) *VmHostDeviceServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmHostDeviceServiceGetRequest) Query(key, value string) *VmHostDeviceServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmHostDeviceServiceGetRequest) Follow(follow string) *VmHostDeviceServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *VmHostDeviceServiceGetRequest) Send() (*VmHostDeviceServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmHostDeviceService.connection.URL(), p.VmHostDeviceService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmHostDeviceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmHostDeviceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmHostDeviceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmHostDeviceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmHostDeviceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostDeviceReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmHostDeviceServiceGetResponse{device: result}, nil
}

func (p *VmHostDeviceServiceGetRequest) MustSend() *VmHostDeviceServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieve information about particular host device attached to given virtual machine.
// Example:
// [source]
// ----
// GET /ovirt-engine/api/vms/123/hostdevices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/543/devices/456" id="456">
//   <name>pci_0000_04_00_0</name>
//   <capability>pci</capability>
//   <iommu_group>30</iommu_group>
//   <placeholder>true</placeholder>
//   <product id="0x13ba">
//     <name>GM107GL [Quadro K2200]</name>
//   </product>
//   <vendor id="0x10de">
//     <name>NVIDIA Corporation</name>
//   </vendor>
//   <host href="/ovirt-engine/api/hosts/543" id="543"/>
//   <parent_device href="/ovirt-engine/api/hosts/543/devices/456" id="456">
//     <name>pci_0000_00_03_0</name>
//   </parent_device>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
// </host_device>
// ----
//
type VmHostDeviceServiceGetResponse struct {
	device *HostDevice
}

func (p *VmHostDeviceServiceGetResponse) Device() (*HostDevice, bool) {
	if p.device != nil {
		return p.device, true
	}
	return nil, false
}

func (p *VmHostDeviceServiceGetResponse) MustDevice() *HostDevice {
	if p.device == nil {
		panic("device in response does not exist")
	}
	return p.device
}

//
// Retrieve information about particular host device attached to given virtual machine.
// Example:
// [source]
// ----
// GET /ovirt-engine/api/vms/123/hostdevices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/543/devices/456" id="456">
//   <name>pci_0000_04_00_0</name>
//   <capability>pci</capability>
//   <iommu_group>30</iommu_group>
//   <placeholder>true</placeholder>
//   <product id="0x13ba">
//     <name>GM107GL [Quadro K2200]</name>
//   </product>
//   <vendor id="0x10de">
//     <name>NVIDIA Corporation</name>
//   </vendor>
//   <host href="/ovirt-engine/api/hosts/543" id="543"/>
//   <parent_device href="/ovirt-engine/api/hosts/543/devices/456" id="456">
//     <name>pci_0000_00_03_0</name>
//   </parent_device>
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
// </host_device>
// ----
//
func (p *VmHostDeviceService) Get() *VmHostDeviceServiceGetRequest {
	return &VmHostDeviceServiceGetRequest{VmHostDeviceService: p}
}

//
// Remove the attachment of this host device from given virtual machine.
// NOTE: In case this device serves as an IOMMU placeholder, it cannot be removed (remove will result only
// in setting its `placeholder` flag to `true`). Note that all IOMMU placeholder devices will be removed
// automatically as soon as there will be no more non-placeholder devices (all devices from given IOMMU
// group are detached).
// [source]
// ----
// DELETE /ovirt-engine/api/vms/123/hostdevices/456
// ----
//
type VmHostDeviceServiceRemoveRequest struct {
	VmHostDeviceService *VmHostDeviceService
	header              map[string]string
	query               map[string]string
	async               *bool
}

func (p *VmHostDeviceServiceRemoveRequest) Header(key, value string) *VmHostDeviceServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmHostDeviceServiceRemoveRequest) Query(key, value string) *VmHostDeviceServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmHostDeviceServiceRemoveRequest) Async(async bool) *VmHostDeviceServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *VmHostDeviceServiceRemoveRequest) Send() (*VmHostDeviceServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmHostDeviceService.connection.URL(), p.VmHostDeviceService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmHostDeviceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmHostDeviceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmHostDeviceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmHostDeviceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmHostDeviceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(VmHostDeviceServiceRemoveResponse), nil
}

func (p *VmHostDeviceServiceRemoveRequest) MustSend() *VmHostDeviceServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Remove the attachment of this host device from given virtual machine.
// NOTE: In case this device serves as an IOMMU placeholder, it cannot be removed (remove will result only
// in setting its `placeholder` flag to `true`). Note that all IOMMU placeholder devices will be removed
// automatically as soon as there will be no more non-placeholder devices (all devices from given IOMMU
// group are detached).
// [source]
// ----
// DELETE /ovirt-engine/api/vms/123/hostdevices/456
// ----
//
type VmHostDeviceServiceRemoveResponse struct {
}

//
// Remove the attachment of this host device from given virtual machine.
// NOTE: In case this device serves as an IOMMU placeholder, it cannot be removed (remove will result only
// in setting its `placeholder` flag to `true`). Note that all IOMMU placeholder devices will be removed
// automatically as soon as there will be no more non-placeholder devices (all devices from given IOMMU
// group are detached).
// [source]
// ----
// DELETE /ovirt-engine/api/vms/123/hostdevices/456
// ----
//
func (p *VmHostDeviceService) Remove() *VmHostDeviceServiceRemoveRequest {
	return &VmHostDeviceServiceRemoveRequest{VmHostDeviceService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmHostDeviceService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *VmHostDeviceService) String() string {
	return fmt.Sprintf("VmHostDeviceService:%s", op.path)
}

//
//
type HostNumaNodesService struct {
	BaseService
}

func NewHostNumaNodesService(connection *Connection, path string) *HostNumaNodesService {
	var result HostNumaNodesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of NUMA nodes of the host.
// The order of the returned list of NUMA nodes isn't guaranteed.
//
type HostNumaNodesServiceListRequest struct {
	HostNumaNodesService *HostNumaNodesService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *HostNumaNodesServiceListRequest) Header(key, value string) *HostNumaNodesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostNumaNodesServiceListRequest) Query(key, value string) *HostNumaNodesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostNumaNodesServiceListRequest) Follow(follow string) *HostNumaNodesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *HostNumaNodesServiceListRequest) Max(max int64) *HostNumaNodesServiceListRequest {
	p.max = &max
	return p
}

func (p *HostNumaNodesServiceListRequest) Send() (*HostNumaNodesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostNumaNodesService.connection.URL(), p.HostNumaNodesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostNumaNodesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostNumaNodesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostNumaNodesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostNumaNodesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostNumaNodesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNumaNodeReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &HostNumaNodesServiceListResponse{nodes: result}, nil
}

func (p *HostNumaNodesServiceListRequest) MustSend() *HostNumaNodesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of NUMA nodes of the host.
// The order of the returned list of NUMA nodes isn't guaranteed.
//
type HostNumaNodesServiceListResponse struct {
	nodes *NumaNodeSlice
}

func (p *HostNumaNodesServiceListResponse) Nodes() (*NumaNodeSlice, bool) {
	if p.nodes != nil {
		return p.nodes, true
	}
	return nil, false
}

func (p *HostNumaNodesServiceListResponse) MustNodes() *NumaNodeSlice {
	if p.nodes == nil {
		panic("nodes in response does not exist")
	}
	return p.nodes
}

//
// Returns the list of NUMA nodes of the host.
// The order of the returned list of NUMA nodes isn't guaranteed.
//
func (p *HostNumaNodesService) List() *HostNumaNodesServiceListRequest {
	return &HostNumaNodesServiceListRequest{HostNumaNodesService: p}
}

//
//
func (op *HostNumaNodesService) NodeService(id string) *HostNumaNodeService {
	return NewHostNumaNodeService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostNumaNodesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NodeService(path), nil
	}
	return op.NodeService(path[:index]).Service(path[index+1:])
}

func (op *HostNumaNodesService) String() string {
	return fmt.Sprintf("HostNumaNodesService:%s", op.path)
}

//
//
type InstanceTypeWatchdogsService struct {
	BaseService
}

func NewInstanceTypeWatchdogsService(connection *Connection, path string) *InstanceTypeWatchdogsService {
	var result InstanceTypeWatchdogsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add new watchdog to the instance type.
//
type InstanceTypeWatchdogsServiceAddRequest struct {
	InstanceTypeWatchdogsService *InstanceTypeWatchdogsService
	header                       map[string]string
	query                        map[string]string
	watchdog                     *Watchdog
}

func (p *InstanceTypeWatchdogsServiceAddRequest) Header(key, value string) *InstanceTypeWatchdogsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *InstanceTypeWatchdogsServiceAddRequest) Query(key, value string) *InstanceTypeWatchdogsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *InstanceTypeWatchdogsServiceAddRequest) Watchdog(watchdog *Watchdog) *InstanceTypeWatchdogsServiceAddRequest {
	p.watchdog = watchdog
	return p
}

func (p *InstanceTypeWatchdogsServiceAddRequest) Send() (*InstanceTypeWatchdogsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.InstanceTypeWatchdogsService.connection.URL(), p.InstanceTypeWatchdogsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLWatchdogWriteOne(writer, p.watchdog, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.InstanceTypeWatchdogsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.InstanceTypeWatchdogsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.InstanceTypeWatchdogsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.InstanceTypeWatchdogsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.InstanceTypeWatchdogsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLWatchdogReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &InstanceTypeWatchdogsServiceAddResponse{watchdog: result}, nil
}

func (p *InstanceTypeWatchdogsServiceAddRequest) MustSend() *InstanceTypeWatchdogsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add new watchdog to the instance type.
//
type InstanceTypeWatchdogsServiceAddResponse struct {
	watchdog *Watchdog
}

func (p *InstanceTypeWatchdogsServiceAddResponse) Watchdog() (*Watchdog, bool) {
	if p.watchdog != nil {
		return p.watchdog, true
	}
	return nil, false
}

func (p *InstanceTypeWatchdogsServiceAddResponse) MustWatchdog() *Watchdog {
	if p.watchdog == nil {
		panic("watchdog in response does not exist")
	}
	return p.watchdog
}

//
// Add new watchdog to the instance type.
//
func (p *InstanceTypeWatchdogsService) Add() *InstanceTypeWatchdogsServiceAddRequest {
	return &InstanceTypeWatchdogsServiceAddRequest{InstanceTypeWatchdogsService: p}
}

//
// Lists all the configured watchdogs of the instance type.
// The order of the returned list of watchdogs isn't guaranteed.
//
type InstanceTypeWatchdogsServiceListRequest struct {
	InstanceTypeWatchdogsService *InstanceTypeWatchdogsService
	header                       map[string]string
	query                        map[string]string
	follow                       *string
	max                          *int64
	search                       *string
}

func (p *InstanceTypeWatchdogsServiceListRequest) Header(key, value string) *InstanceTypeWatchdogsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *InstanceTypeWatchdogsServiceListRequest) Query(key, value string) *InstanceTypeWatchdogsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *InstanceTypeWatchdogsServiceListRequest) Follow(follow string) *InstanceTypeWatchdogsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *InstanceTypeWatchdogsServiceListRequest) Max(max int64) *InstanceTypeWatchdogsServiceListRequest {
	p.max = &max
	return p
}

func (p *InstanceTypeWatchdogsServiceListRequest) Search(search string) *InstanceTypeWatchdogsServiceListRequest {
	p.search = &search
	return p
}

func (p *InstanceTypeWatchdogsServiceListRequest) Send() (*InstanceTypeWatchdogsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.InstanceTypeWatchdogsService.connection.URL(), p.InstanceTypeWatchdogsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.InstanceTypeWatchdogsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.InstanceTypeWatchdogsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.InstanceTypeWatchdogsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.InstanceTypeWatchdogsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.InstanceTypeWatchdogsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLWatchdogReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &InstanceTypeWatchdogsServiceListResponse{watchdogs: result}, nil
}

func (p *InstanceTypeWatchdogsServiceListRequest) MustSend() *InstanceTypeWatchdogsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists all the configured watchdogs of the instance type.
// The order of the returned list of watchdogs isn't guaranteed.
//
type InstanceTypeWatchdogsServiceListResponse struct {
	watchdogs *WatchdogSlice
}

func (p *InstanceTypeWatchdogsServiceListResponse) Watchdogs() (*WatchdogSlice, bool) {
	if p.watchdogs != nil {
		return p.watchdogs, true
	}
	return nil, false
}

func (p *InstanceTypeWatchdogsServiceListResponse) MustWatchdogs() *WatchdogSlice {
	if p.watchdogs == nil {
		panic("watchdogs in response does not exist")
	}
	return p.watchdogs
}

//
// Lists all the configured watchdogs of the instance type.
// The order of the returned list of watchdogs isn't guaranteed.
//
func (p *InstanceTypeWatchdogsService) List() *InstanceTypeWatchdogsServiceListRequest {
	return &InstanceTypeWatchdogsServiceListRequest{InstanceTypeWatchdogsService: p}
}

//
//
func (op *InstanceTypeWatchdogsService) WatchdogService(id string) *InstanceTypeWatchdogService {
	return NewInstanceTypeWatchdogService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *InstanceTypeWatchdogsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.WatchdogService(path), nil
	}
	return op.WatchdogService(path[:index]).Service(path[index+1:])
}

func (op *InstanceTypeWatchdogsService) String() string {
	return fmt.Sprintf("InstanceTypeWatchdogsService:%s", op.path)
}

//
//
type VirtualFunctionAllowedNetworksService struct {
	BaseService
}

func NewVirtualFunctionAllowedNetworksService(connection *Connection, path string) *VirtualFunctionAllowedNetworksService {
	var result VirtualFunctionAllowedNetworksService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type VirtualFunctionAllowedNetworksServiceAddRequest struct {
	VirtualFunctionAllowedNetworksService *VirtualFunctionAllowedNetworksService
	header                                map[string]string
	query                                 map[string]string
	network                               *Network
}

func (p *VirtualFunctionAllowedNetworksServiceAddRequest) Header(key, value string) *VirtualFunctionAllowedNetworksServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VirtualFunctionAllowedNetworksServiceAddRequest) Query(key, value string) *VirtualFunctionAllowedNetworksServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VirtualFunctionAllowedNetworksServiceAddRequest) Network(network *Network) *VirtualFunctionAllowedNetworksServiceAddRequest {
	p.network = network
	return p
}

func (p *VirtualFunctionAllowedNetworksServiceAddRequest) Send() (*VirtualFunctionAllowedNetworksServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VirtualFunctionAllowedNetworksService.connection.URL(), p.VirtualFunctionAllowedNetworksService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkWriteOne(writer, p.network, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VirtualFunctionAllowedNetworksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VirtualFunctionAllowedNetworksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VirtualFunctionAllowedNetworksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VirtualFunctionAllowedNetworksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VirtualFunctionAllowedNetworksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VirtualFunctionAllowedNetworksServiceAddResponse{network: result}, nil
}

func (p *VirtualFunctionAllowedNetworksServiceAddRequest) MustSend() *VirtualFunctionAllowedNetworksServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VirtualFunctionAllowedNetworksServiceAddResponse struct {
	network *Network
}

func (p *VirtualFunctionAllowedNetworksServiceAddResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *VirtualFunctionAllowedNetworksServiceAddResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
//
func (p *VirtualFunctionAllowedNetworksService) Add() *VirtualFunctionAllowedNetworksServiceAddRequest {
	return &VirtualFunctionAllowedNetworksServiceAddRequest{VirtualFunctionAllowedNetworksService: p}
}

//
// Returns the list of networks.
// The order of the returned list of networks isn't guaranteed.
//
type VirtualFunctionAllowedNetworksServiceListRequest struct {
	VirtualFunctionAllowedNetworksService *VirtualFunctionAllowedNetworksService
	header                                map[string]string
	query                                 map[string]string
	follow                                *string
	max                                   *int64
}

func (p *VirtualFunctionAllowedNetworksServiceListRequest) Header(key, value string) *VirtualFunctionAllowedNetworksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VirtualFunctionAllowedNetworksServiceListRequest) Query(key, value string) *VirtualFunctionAllowedNetworksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VirtualFunctionAllowedNetworksServiceListRequest) Follow(follow string) *VirtualFunctionAllowedNetworksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *VirtualFunctionAllowedNetworksServiceListRequest) Max(max int64) *VirtualFunctionAllowedNetworksServiceListRequest {
	p.max = &max
	return p
}

func (p *VirtualFunctionAllowedNetworksServiceListRequest) Send() (*VirtualFunctionAllowedNetworksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VirtualFunctionAllowedNetworksService.connection.URL(), p.VirtualFunctionAllowedNetworksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VirtualFunctionAllowedNetworksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VirtualFunctionAllowedNetworksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VirtualFunctionAllowedNetworksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VirtualFunctionAllowedNetworksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VirtualFunctionAllowedNetworksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &VirtualFunctionAllowedNetworksServiceListResponse{networks: result}, nil
}

func (p *VirtualFunctionAllowedNetworksServiceListRequest) MustSend() *VirtualFunctionAllowedNetworksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of networks.
// The order of the returned list of networks isn't guaranteed.
//
type VirtualFunctionAllowedNetworksServiceListResponse struct {
	networks *NetworkSlice
}

func (p *VirtualFunctionAllowedNetworksServiceListResponse) Networks() (*NetworkSlice, bool) {
	if p.networks != nil {
		return p.networks, true
	}
	return nil, false
}

func (p *VirtualFunctionAllowedNetworksServiceListResponse) MustNetworks() *NetworkSlice {
	if p.networks == nil {
		panic("networks in response does not exist")
	}
	return p.networks
}

//
// Returns the list of networks.
// The order of the returned list of networks isn't guaranteed.
//
func (p *VirtualFunctionAllowedNetworksService) List() *VirtualFunctionAllowedNetworksServiceListRequest {
	return &VirtualFunctionAllowedNetworksServiceListRequest{VirtualFunctionAllowedNetworksService: p}
}

//
//
func (op *VirtualFunctionAllowedNetworksService) NetworkService(id string) *VirtualFunctionAllowedNetworkService {
	return NewVirtualFunctionAllowedNetworkService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VirtualFunctionAllowedNetworksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NetworkService(path), nil
	}
	return op.NetworkService(path[:index]).Service(path[index+1:])
}

func (op *VirtualFunctionAllowedNetworksService) String() string {
	return fmt.Sprintf("VirtualFunctionAllowedNetworksService:%s", op.path)
}

//
// This service manages the set of disks attached to a virtual machine. Each attached disk is represented by a
// <<types/disk_attachment,DiskAttachment>>, containing the bootable flag, the disk interface and the reference to
// the disk.
//
type DiskAttachmentsService struct {
	BaseService
}

func NewDiskAttachmentsService(connection *Connection, path string) *DiskAttachmentsService {
	var result DiskAttachmentsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a new disk attachment to the virtual machine. The `attachment` parameter can contain just a reference, if
// the disk already exists:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk id="123"/>
// </disk_attachment>
// ----
// Or it can contain the complete representation of the disk, if the disk doesn't exist yet:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk>
//     <name>mydisk</name>
//     <provisioned_size>1024</provisioned_size>
//     ...
//   </disk>
// </disk_attachment>
// ----
// In this case the disk will be created and then attached to the virtual machine.
// In both cases, use the following URL for a virtual machine with an id `345`:
// [source]
// ----
// POST /ovirt-engine/api/vms/345/diskattachments
// ----
// IMPORTANT: The server accepts requests that don't contain the `active` attribute, but the effect is
// undefined. In some cases the disk will be automatically activated and in other cases it won't. To
// avoid issues it is strongly recommended to always include the `active` attribute with the desired
// value.
//
type DiskAttachmentsServiceAddRequest struct {
	DiskAttachmentsService *DiskAttachmentsService
	header                 map[string]string
	query                  map[string]string
	attachment             *DiskAttachment
}

func (p *DiskAttachmentsServiceAddRequest) Header(key, value string) *DiskAttachmentsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddRequest) Query(key, value string) *DiskAttachmentsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddRequest) Attachment(attachment *DiskAttachment) *DiskAttachmentsServiceAddRequest {
	p.attachment = attachment
	return p
}

func (p *DiskAttachmentsServiceAddRequest) Send() (*DiskAttachmentsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskAttachmentsService.connection.URL(), p.DiskAttachmentsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskAttachmentWriteOne(writer, p.attachment, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskAttachmentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskAttachmentsServiceAddResponse{attachment: result}, nil
}

func (p *DiskAttachmentsServiceAddRequest) MustSend() *DiskAttachmentsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a new disk attachment to the virtual machine. The `attachment` parameter can contain just a reference, if
// the disk already exists:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk id="123"/>
// </disk_attachment>
// ----
// Or it can contain the complete representation of the disk, if the disk doesn't exist yet:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk>
//     <name>mydisk</name>
//     <provisioned_size>1024</provisioned_size>
//     ...
//   </disk>
// </disk_attachment>
// ----
// In this case the disk will be created and then attached to the virtual machine.
// In both cases, use the following URL for a virtual machine with an id `345`:
// [source]
// ----
// POST /ovirt-engine/api/vms/345/diskattachments
// ----
// IMPORTANT: The server accepts requests that don't contain the `active` attribute, but the effect is
// undefined. In some cases the disk will be automatically activated and in other cases it won't. To
// avoid issues it is strongly recommended to always include the `active` attribute with the desired
// value.
//
type DiskAttachmentsServiceAddResponse struct {
	attachment *DiskAttachment
}

func (p *DiskAttachmentsServiceAddResponse) Attachment() (*DiskAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *DiskAttachmentsServiceAddResponse) MustAttachment() *DiskAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
// Adds a new disk attachment to the virtual machine. The `attachment` parameter can contain just a reference, if
// the disk already exists:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk id="123"/>
// </disk_attachment>
// ----
// Or it can contain the complete representation of the disk, if the disk doesn't exist yet:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk>
//     <name>mydisk</name>
//     <provisioned_size>1024</provisioned_size>
//     ...
//   </disk>
// </disk_attachment>
// ----
// In this case the disk will be created and then attached to the virtual machine.
// In both cases, use the following URL for a virtual machine with an id `345`:
// [source]
// ----
// POST /ovirt-engine/api/vms/345/diskattachments
// ----
// IMPORTANT: The server accepts requests that don't contain the `active` attribute, but the effect is
// undefined. In some cases the disk will be automatically activated and in other cases it won't. To
// avoid issues it is strongly recommended to always include the `active` attribute with the desired
// value.
//
func (p *DiskAttachmentsService) Add() *DiskAttachmentsServiceAddRequest {
	return &DiskAttachmentsServiceAddRequest{DiskAttachmentsService: p}
}

//
// List the disk that are attached to the virtual machine.
// The order of the returned list of disks attachments isn't guaranteed.
//
type DiskAttachmentsServiceListRequest struct {
	DiskAttachmentsService *DiskAttachmentsService
	header                 map[string]string
	query                  map[string]string
	follow                 *string
}

func (p *DiskAttachmentsServiceListRequest) Header(key, value string) *DiskAttachmentsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentsServiceListRequest) Query(key, value string) *DiskAttachmentsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentsServiceListRequest) Follow(follow string) *DiskAttachmentsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DiskAttachmentsServiceListRequest) Send() (*DiskAttachmentsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskAttachmentsService.connection.URL(), p.DiskAttachmentsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskAttachmentReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DiskAttachmentsServiceListResponse{attachments: result}, nil
}

func (p *DiskAttachmentsServiceListRequest) MustSend() *DiskAttachmentsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List the disk that are attached to the virtual machine.
// The order of the returned list of disks attachments isn't guaranteed.
//
type DiskAttachmentsServiceListResponse struct {
	attachments *DiskAttachmentSlice
}

func (p *DiskAttachmentsServiceListResponse) Attachments() (*DiskAttachmentSlice, bool) {
	if p.attachments != nil {
		return p.attachments, true
	}
	return nil, false
}

func (p *DiskAttachmentsServiceListResponse) MustAttachments() *DiskAttachmentSlice {
	if p.attachments == nil {
		panic("attachments in response does not exist")
	}
	return p.attachments
}

//
// List the disk that are attached to the virtual machine.
// The order of the returned list of disks attachments isn't guaranteed.
//
func (p *DiskAttachmentsService) List() *DiskAttachmentsServiceListRequest {
	return &DiskAttachmentsServiceListRequest{DiskAttachmentsService: p}
}

//
//
type DiskAttachmentsServiceAddProvidingDiskIdRequest struct {
	DiskAttachmentsService *DiskAttachmentsService
	header                 map[string]string
	query                  map[string]string
	attachment             *DiskAttachment
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) Header(key, value string) *DiskAttachmentsServiceAddProvidingDiskIdRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) Query(key, value string) *DiskAttachmentsServiceAddProvidingDiskIdRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) Attachment(attachment *DiskAttachment) *DiskAttachmentsServiceAddProvidingDiskIdRequest {
	p.attachment = attachment
	return p
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) Send() (*DiskAttachmentsServiceAddProvidingDiskIdResponse, error) {
	rawURL := fmt.Sprintf("%s%s/providingdiskid", p.DiskAttachmentsService.connection.URL(), p.DiskAttachmentsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Attachment(p.attachment)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustAttachment()
	return &DiskAttachmentsServiceAddProvidingDiskIdResponse{attachment: result}, nil
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) MustSend() *DiskAttachmentsServiceAddProvidingDiskIdResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskAttachmentsServiceAddProvidingDiskIdResponse struct {
	attachment *DiskAttachment
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdResponse) Attachment() (*DiskAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdResponse) MustAttachment() *DiskAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
//
func (p *DiskAttachmentsService) AddProvidingDiskId() *DiskAttachmentsServiceAddProvidingDiskIdRequest {
	return &DiskAttachmentsServiceAddProvidingDiskIdRequest{DiskAttachmentsService: p}
}

//
//
type DiskAttachmentsServiceAddSignature1Request struct {
	DiskAttachmentsService *DiskAttachmentsService
	header                 map[string]string
	query                  map[string]string
	attachment             *DiskAttachment
}

func (p *DiskAttachmentsServiceAddSignature1Request) Header(key, value string) *DiskAttachmentsServiceAddSignature1Request {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddSignature1Request) Query(key, value string) *DiskAttachmentsServiceAddSignature1Request {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddSignature1Request) Attachment(attachment *DiskAttachment) *DiskAttachmentsServiceAddSignature1Request {
	p.attachment = attachment
	return p
}

func (p *DiskAttachmentsServiceAddSignature1Request) Send() (*DiskAttachmentsServiceAddSignature1Response, error) {
	rawURL := fmt.Sprintf("%s%s/signature1", p.DiskAttachmentsService.connection.URL(), p.DiskAttachmentsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Attachment(p.attachment)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustAttachment()
	return &DiskAttachmentsServiceAddSignature1Response{attachment: result}, nil
}

func (p *DiskAttachmentsServiceAddSignature1Request) MustSend() *DiskAttachmentsServiceAddSignature1Response {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskAttachmentsServiceAddSignature1Response struct {
	attachment *DiskAttachment
}

func (p *DiskAttachmentsServiceAddSignature1Response) Attachment() (*DiskAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *DiskAttachmentsServiceAddSignature1Response) MustAttachment() *DiskAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
//
func (p *DiskAttachmentsService) AddSignature1() *DiskAttachmentsServiceAddSignature1Request {
	return &DiskAttachmentsServiceAddSignature1Request{DiskAttachmentsService: p}
}

//
// Reference to the service that manages a specific attachment.
//
func (op *DiskAttachmentsService) AttachmentService(id string) *DiskAttachmentService {
	return NewDiskAttachmentService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DiskAttachmentsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.AttachmentService(path), nil
	}
	return op.AttachmentService(path[:index]).Service(path[index+1:])
}

func (op *DiskAttachmentsService) String() string {
	return fmt.Sprintf("DiskAttachmentsService:%s", op.path)
}

//
//
type StorageServerConnectionExtensionService struct {
	BaseService
}

func NewStorageServerConnectionExtensionService(connection *Connection, path string) *StorageServerConnectionExtensionService {
	var result StorageServerConnectionExtensionService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type StorageServerConnectionExtensionServiceGetRequest struct {
	StorageServerConnectionExtensionService *StorageServerConnectionExtensionService
	header                                  map[string]string
	query                                   map[string]string
	follow                                  *string
}

func (p *StorageServerConnectionExtensionServiceGetRequest) Header(key, value string) *StorageServerConnectionExtensionServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionExtensionServiceGetRequest) Query(key, value string) *StorageServerConnectionExtensionServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionExtensionServiceGetRequest) Follow(follow string) *StorageServerConnectionExtensionServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *StorageServerConnectionExtensionServiceGetRequest) Send() (*StorageServerConnectionExtensionServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageServerConnectionExtensionService.connection.URL(), p.StorageServerConnectionExtensionService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionExtensionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionExtensionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionExtensionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionExtensionService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionExtensionService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageConnectionExtensionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageServerConnectionExtensionServiceGetResponse{extension: result}, nil
}

func (p *StorageServerConnectionExtensionServiceGetRequest) MustSend() *StorageServerConnectionExtensionServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type StorageServerConnectionExtensionServiceGetResponse struct {
	extension *StorageConnectionExtension
}

func (p *StorageServerConnectionExtensionServiceGetResponse) Extension() (*StorageConnectionExtension, bool) {
	if p.extension != nil {
		return p.extension, true
	}
	return nil, false
}

func (p *StorageServerConnectionExtensionServiceGetResponse) MustExtension() *StorageConnectionExtension {
	if p.extension == nil {
		panic("extension in response does not exist")
	}
	return p.extension
}

//
//
func (p *StorageServerConnectionExtensionService) Get() *StorageServerConnectionExtensionServiceGetRequest {
	return &StorageServerConnectionExtensionServiceGetRequest{StorageServerConnectionExtensionService: p}
}

//
//
type StorageServerConnectionExtensionServiceRemoveRequest struct {
	StorageServerConnectionExtensionService *StorageServerConnectionExtensionService
	header                                  map[string]string
	query                                   map[string]string
	async                                   *bool
}

func (p *StorageServerConnectionExtensionServiceRemoveRequest) Header(key, value string) *StorageServerConnectionExtensionServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionExtensionServiceRemoveRequest) Query(key, value string) *StorageServerConnectionExtensionServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionExtensionServiceRemoveRequest) Async(async bool) *StorageServerConnectionExtensionServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *StorageServerConnectionExtensionServiceRemoveRequest) Send() (*StorageServerConnectionExtensionServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageServerConnectionExtensionService.connection.URL(), p.StorageServerConnectionExtensionService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionExtensionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionExtensionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionExtensionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionExtensionService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionExtensionService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(StorageServerConnectionExtensionServiceRemoveResponse), nil
}

func (p *StorageServerConnectionExtensionServiceRemoveRequest) MustSend() *StorageServerConnectionExtensionServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type StorageServerConnectionExtensionServiceRemoveResponse struct {
}

//
//
func (p *StorageServerConnectionExtensionService) Remove() *StorageServerConnectionExtensionServiceRemoveRequest {
	return &StorageServerConnectionExtensionServiceRemoveRequest{StorageServerConnectionExtensionService: p}
}

//
// Update a storage server connection extension for the given host.
// To update the storage connection `456` of host `123` send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/hosts/123/storageconnectionextensions/456
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_connection_extension>
//   <target>iqn.2016-01.com.example:mytarget</target>
//   <username>myuser</username>
//   <password>mypassword</password>
// </storage_connection_extension>
// ----
//
type StorageServerConnectionExtensionServiceUpdateRequest struct {
	StorageServerConnectionExtensionService *StorageServerConnectionExtensionService
	header                                  map[string]string
	query                                   map[string]string
	async                                   *bool
	extension                               *StorageConnectionExtension
}

func (p *StorageServerConnectionExtensionServiceUpdateRequest) Header(key, value string) *StorageServerConnectionExtensionServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionExtensionServiceUpdateRequest) Query(key, value string) *StorageServerConnectionExtensionServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionExtensionServiceUpdateRequest) Async(async bool) *StorageServerConnectionExtensionServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *StorageServerConnectionExtensionServiceUpdateRequest) Extension(extension *StorageConnectionExtension) *StorageServerConnectionExtensionServiceUpdateRequest {
	p.extension = extension
	return p
}

func (p *StorageServerConnectionExtensionServiceUpdateRequest) Send() (*StorageServerConnectionExtensionServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageServerConnectionExtensionService.connection.URL(), p.StorageServerConnectionExtensionService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLStorageConnectionExtensionWriteOne(writer, p.extension, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionExtensionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionExtensionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionExtensionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionExtensionService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionExtensionService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageConnectionExtensionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageServerConnectionExtensionServiceUpdateResponse{extension: result}, nil
}

func (p *StorageServerConnectionExtensionServiceUpdateRequest) MustSend() *StorageServerConnectionExtensionServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update a storage server connection extension for the given host.
// To update the storage connection `456` of host `123` send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/hosts/123/storageconnectionextensions/456
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_connection_extension>
//   <target>iqn.2016-01.com.example:mytarget</target>
//   <username>myuser</username>
//   <password>mypassword</password>
// </storage_connection_extension>
// ----
//
type StorageServerConnectionExtensionServiceUpdateResponse struct {
	extension *StorageConnectionExtension
}

func (p *StorageServerConnectionExtensionServiceUpdateResponse) Extension() (*StorageConnectionExtension, bool) {
	if p.extension != nil {
		return p.extension, true
	}
	return nil, false
}

func (p *StorageServerConnectionExtensionServiceUpdateResponse) MustExtension() *StorageConnectionExtension {
	if p.extension == nil {
		panic("extension in response does not exist")
	}
	return p.extension
}

//
// Update a storage server connection extension for the given host.
// To update the storage connection `456` of host `123` send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/hosts/123/storageconnectionextensions/456
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_connection_extension>
//   <target>iqn.2016-01.com.example:mytarget</target>
//   <username>myuser</username>
//   <password>mypassword</password>
// </storage_connection_extension>
// ----
//
func (p *StorageServerConnectionExtensionService) Update() *StorageServerConnectionExtensionServiceUpdateRequest {
	return &StorageServerConnectionExtensionServiceUpdateRequest{StorageServerConnectionExtensionService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageServerConnectionExtensionService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *StorageServerConnectionExtensionService) String() string {
	return fmt.Sprintf("StorageServerConnectionExtensionService:%s", op.path)
}

//
// This service manages image transfers, for performing Image I/O API in {product-name}.
// Please refer to <<services/image_transfer, image transfer>> for further
// documentation.
//
type ImageTransfersService struct {
	BaseService
}

func NewImageTransfersService(connection *Connection, path string) *ImageTransfersService {
	var result ImageTransfersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new image transfer. An image, disk or disk snapshot needs to be specified
// in order to make a new transfer.
// IMPORTANT: The `image` attribute is deprecated since version 4.2 of the engine.
// Use the `disk` or `snapshot` attributes instead.
// *Creating a new image transfer for downloading or uploading a `disk`:*
// To create an image transfer to download or upload a disk with id `123`,
// send the following request:
// [source]
// ----
// POST /ovirt-engine/api/imagetransfers
// ----
// With a request body like this:
// [source,xml]
// ----
// <image_transfer>
//   <disk id="123"/>
//   <direction>upload|download</direction>
// </image_transfer>
// ----
// *Creating a new image transfer for downloading or uploading a `disk_snapshot`:*
// To create an image transfer to download or upload a `disk_snapshot` with id `456`,
// send the following request:
// [source]
// ----
// POST /ovirt-engine/api/imagetransfers
// ----
// With a request body like this:
// [source,xml]
// ----
// <image_transfer>
//   <snapshot id="456"/>
//   <direction>download|upload</direction>
// </image_transfer>
// ----
//
type ImageTransfersServiceAddRequest struct {
	ImageTransfersService *ImageTransfersService
	header                map[string]string
	query                 map[string]string
	imageTransfer         *ImageTransfer
}

func (p *ImageTransfersServiceAddRequest) Header(key, value string) *ImageTransfersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ImageTransfersServiceAddRequest) Query(key, value string) *ImageTransfersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ImageTransfersServiceAddRequest) ImageTransfer(imageTransfer *ImageTransfer) *ImageTransfersServiceAddRequest {
	p.imageTransfer = imageTransfer
	return p
}

func (p *ImageTransfersServiceAddRequest) Send() (*ImageTransfersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ImageTransfersService.connection.URL(), p.ImageTransfersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLImageTransferWriteOne(writer, p.imageTransfer, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ImageTransfersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ImageTransfersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ImageTransfersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ImageTransfersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ImageTransfersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLImageTransferReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ImageTransfersServiceAddResponse{imageTransfer: result}, nil
}

func (p *ImageTransfersServiceAddRequest) MustSend() *ImageTransfersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new image transfer. An image, disk or disk snapshot needs to be specified
// in order to make a new transfer.
// IMPORTANT: The `image` attribute is deprecated since version 4.2 of the engine.
// Use the `disk` or `snapshot` attributes instead.
// *Creating a new image transfer for downloading or uploading a `disk`:*
// To create an image transfer to download or upload a disk with id `123`,
// send the following request:
// [source]
// ----
// POST /ovirt-engine/api/imagetransfers
// ----
// With a request body like this:
// [source,xml]
// ----
// <image_transfer>
//   <disk id="123"/>
//   <direction>upload|download</direction>
// </image_transfer>
// ----
// *Creating a new image transfer for downloading or uploading a `disk_snapshot`:*
// To create an image transfer to download or upload a `disk_snapshot` with id `456`,
// send the following request:
// [source]
// ----
// POST /ovirt-engine/api/imagetransfers
// ----
// With a request body like this:
// [source,xml]
// ----
// <image_transfer>
//   <snapshot id="456"/>
//   <direction>download|upload</direction>
// </image_transfer>
// ----
//
type ImageTransfersServiceAddResponse struct {
	imageTransfer *ImageTransfer
}

func (p *ImageTransfersServiceAddResponse) ImageTransfer() (*ImageTransfer, bool) {
	if p.imageTransfer != nil {
		return p.imageTransfer, true
	}
	return nil, false
}

func (p *ImageTransfersServiceAddResponse) MustImageTransfer() *ImageTransfer {
	if p.imageTransfer == nil {
		panic("imageTransfer in response does not exist")
	}
	return p.imageTransfer
}

//
// Add a new image transfer. An image, disk or disk snapshot needs to be specified
// in order to make a new transfer.
// IMPORTANT: The `image` attribute is deprecated since version 4.2 of the engine.
// Use the `disk` or `snapshot` attributes instead.
// *Creating a new image transfer for downloading or uploading a `disk`:*
// To create an image transfer to download or upload a disk with id `123`,
// send the following request:
// [source]
// ----
// POST /ovirt-engine/api/imagetransfers
// ----
// With a request body like this:
// [source,xml]
// ----
// <image_transfer>
//   <disk id="123"/>
//   <direction>upload|download</direction>
// </image_transfer>
// ----
// *Creating a new image transfer for downloading or uploading a `disk_snapshot`:*
// To create an image transfer to download or upload a `disk_snapshot` with id `456`,
// send the following request:
// [source]
// ----
// POST /ovirt-engine/api/imagetransfers
// ----
// With a request body like this:
// [source,xml]
// ----
// <image_transfer>
//   <snapshot id="456"/>
//   <direction>download|upload</direction>
// </image_transfer>
// ----
//
func (p *ImageTransfersService) Add() *ImageTransfersServiceAddRequest {
	return &ImageTransfersServiceAddRequest{ImageTransfersService: p}
}

//
//
type ImageTransfersServiceAddForDiskRequest struct {
	ImageTransfersService *ImageTransfersService
	header                map[string]string
	query                 map[string]string
	imageTransfer         *ImageTransfer
}

func (p *ImageTransfersServiceAddForDiskRequest) Header(key, value string) *ImageTransfersServiceAddForDiskRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ImageTransfersServiceAddForDiskRequest) Query(key, value string) *ImageTransfersServiceAddForDiskRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ImageTransfersServiceAddForDiskRequest) ImageTransfer(imageTransfer *ImageTransfer) *ImageTransfersServiceAddForDiskRequest {
	p.imageTransfer = imageTransfer
	return p
}

func (p *ImageTransfersServiceAddForDiskRequest) Send() (*ImageTransfersServiceAddForDiskResponse, error) {
	rawURL := fmt.Sprintf("%s%s/fordisk", p.ImageTransfersService.connection.URL(), p.ImageTransfersService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.ImageTransfer(p.imageTransfer)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ImageTransfersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ImageTransfersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ImageTransfersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ImageTransfersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ImageTransfersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustImageTransfer()
	return &ImageTransfersServiceAddForDiskResponse{imageTransfer: result}, nil
}

func (p *ImageTransfersServiceAddForDiskRequest) MustSend() *ImageTransfersServiceAddForDiskResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type ImageTransfersServiceAddForDiskResponse struct {
	imageTransfer *ImageTransfer
}

func (p *ImageTransfersServiceAddForDiskResponse) ImageTransfer() (*ImageTransfer, bool) {
	if p.imageTransfer != nil {
		return p.imageTransfer, true
	}
	return nil, false
}

func (p *ImageTransfersServiceAddForDiskResponse) MustImageTransfer() *ImageTransfer {
	if p.imageTransfer == nil {
		panic("imageTransfer in response does not exist")
	}
	return p.imageTransfer
}

//
//
func (p *ImageTransfersService) AddForDisk() *ImageTransfersServiceAddForDiskRequest {
	return &ImageTransfersServiceAddForDiskRequest{ImageTransfersService: p}
}

//
//
type ImageTransfersServiceAddForImageRequest struct {
	ImageTransfersService *ImageTransfersService
	header                map[string]string
	query                 map[string]string
	imageTransfer         *ImageTransfer
}

func (p *ImageTransfersServiceAddForImageRequest) Header(key, value string) *ImageTransfersServiceAddForImageRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ImageTransfersServiceAddForImageRequest) Query(key, value string) *ImageTransfersServiceAddForImageRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ImageTransfersServiceAddForImageRequest) ImageTransfer(imageTransfer *ImageTransfer) *ImageTransfersServiceAddForImageRequest {
	p.imageTransfer = imageTransfer
	return p
}

func (p *ImageTransfersServiceAddForImageRequest) Send() (*ImageTransfersServiceAddForImageResponse, error) {
	rawURL := fmt.Sprintf("%s%s/forimage", p.ImageTransfersService.connection.URL(), p.ImageTransfersService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.ImageTransfer(p.imageTransfer)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ImageTransfersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ImageTransfersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ImageTransfersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ImageTransfersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ImageTransfersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustImageTransfer()
	return &ImageTransfersServiceAddForImageResponse{imageTransfer: result}, nil
}

func (p *ImageTransfersServiceAddForImageRequest) MustSend() *ImageTransfersServiceAddForImageResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type ImageTransfersServiceAddForImageResponse struct {
	imageTransfer *ImageTransfer
}

func (p *ImageTransfersServiceAddForImageResponse) ImageTransfer() (*ImageTransfer, bool) {
	if p.imageTransfer != nil {
		return p.imageTransfer, true
	}
	return nil, false
}

func (p *ImageTransfersServiceAddForImageResponse) MustImageTransfer() *ImageTransfer {
	if p.imageTransfer == nil {
		panic("imageTransfer in response does not exist")
	}
	return p.imageTransfer
}

//
//
func (p *ImageTransfersService) AddForImage() *ImageTransfersServiceAddForImageRequest {
	return &ImageTransfersServiceAddForImageRequest{ImageTransfersService: p}
}

//
//
type ImageTransfersServiceAddForSnapshotRequest struct {
	ImageTransfersService *ImageTransfersService
	header                map[string]string
	query                 map[string]string
	imageTransfer         *ImageTransfer
}

func (p *ImageTransfersServiceAddForSnapshotRequest) Header(key, value string) *ImageTransfersServiceAddForSnapshotRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ImageTransfersServiceAddForSnapshotRequest) Query(key, value string) *ImageTransfersServiceAddForSnapshotRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ImageTransfersServiceAddForSnapshotRequest) ImageTransfer(imageTransfer *ImageTransfer) *ImageTransfersServiceAddForSnapshotRequest {
	p.imageTransfer = imageTransfer
	return p
}

func (p *ImageTransfersServiceAddForSnapshotRequest) Send() (*ImageTransfersServiceAddForSnapshotResponse, error) {
	rawURL := fmt.Sprintf("%s%s/forsnapshot", p.ImageTransfersService.connection.URL(), p.ImageTransfersService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.ImageTransfer(p.imageTransfer)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ImageTransfersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ImageTransfersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ImageTransfersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ImageTransfersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ImageTransfersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustImageTransfer()
	return &ImageTransfersServiceAddForSnapshotResponse{imageTransfer: result}, nil
}

func (p *ImageTransfersServiceAddForSnapshotRequest) MustSend() *ImageTransfersServiceAddForSnapshotResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type ImageTransfersServiceAddForSnapshotResponse struct {
	imageTransfer *ImageTransfer
}

func (p *ImageTransfersServiceAddForSnapshotResponse) ImageTransfer() (*ImageTransfer, bool) {
	if p.imageTransfer != nil {
		return p.imageTransfer, true
	}
	return nil, false
}

func (p *ImageTransfersServiceAddForSnapshotResponse) MustImageTransfer() *ImageTransfer {
	if p.imageTransfer == nil {
		panic("imageTransfer in response does not exist")
	}
	return p.imageTransfer
}

//
//
func (p *ImageTransfersService) AddForSnapshot() *ImageTransfersServiceAddForSnapshotRequest {
	return &ImageTransfersServiceAddForSnapshotRequest{ImageTransfersService: p}
}

//
// Retrieves the list of image transfers that are currently
// being performed.
// The order of the returned list of image transfers is not guaranteed.
//
type ImageTransfersServiceListRequest struct {
	ImageTransfersService *ImageTransfersService
	header                map[string]string
	query                 map[string]string
	follow                *string
}

func (p *ImageTransfersServiceListRequest) Header(key, value string) *ImageTransfersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ImageTransfersServiceListRequest) Query(key, value string) *ImageTransfersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ImageTransfersServiceListRequest) Follow(follow string) *ImageTransfersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ImageTransfersServiceListRequest) Send() (*ImageTransfersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ImageTransfersService.connection.URL(), p.ImageTransfersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ImageTransfersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ImageTransfersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ImageTransfersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ImageTransfersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ImageTransfersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLImageTransferReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ImageTransfersServiceListResponse{imageTransfer: result}, nil
}

func (p *ImageTransfersServiceListRequest) MustSend() *ImageTransfersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the list of image transfers that are currently
// being performed.
// The order of the returned list of image transfers is not guaranteed.
//
type ImageTransfersServiceListResponse struct {
	imageTransfer *ImageTransferSlice
}

func (p *ImageTransfersServiceListResponse) ImageTransfer() (*ImageTransferSlice, bool) {
	if p.imageTransfer != nil {
		return p.imageTransfer, true
	}
	return nil, false
}

func (p *ImageTransfersServiceListResponse) MustImageTransfer() *ImageTransferSlice {
	if p.imageTransfer == nil {
		panic("imageTransfer in response does not exist")
	}
	return p.imageTransfer
}

//
// Retrieves the list of image transfers that are currently
// being performed.
// The order of the returned list of image transfers is not guaranteed.
//
func (p *ImageTransfersService) List() *ImageTransfersServiceListRequest {
	return &ImageTransfersServiceListRequest{ImageTransfersService: p}
}

//
// Returns a reference to the service that manages an
// specific image transfer.
//
func (op *ImageTransfersService) ImageTransferService(id string) *ImageTransferService {
	return NewImageTransferService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ImageTransfersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ImageTransferService(path), nil
	}
	return op.ImageTransferService(path[:index]).Service(path[index+1:])
}

func (op *ImageTransfersService) String() string {
	return fmt.Sprintf("ImageTransfersService:%s", op.path)
}

//
// A service to manage a specific permit of the role.
//
type PermitService struct {
	BaseService
}

func NewPermitService(connection *Connection, path string) *PermitService {
	var result PermitService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets the information about the permit of the role.
// For example to retrieve the information about the permit with the id `456` of the role with the id `123`
// send a request like this:
// ....
// GET /ovirt-engine/api/roles/123/permits/456
// ....
// [source,xml]
// ----
// <permit href="/ovirt-engine/api/roles/123/permits/456" id="456">
//   <name>change_vm_cd</name>
//   <administrative>false</administrative>
//   <role href="/ovirt-engine/api/roles/123" id="123"/>
// </permit>
// ----
//
type PermitServiceGetRequest struct {
	PermitService *PermitService
	header        map[string]string
	query         map[string]string
	follow        *string
}

func (p *PermitServiceGetRequest) Header(key, value string) *PermitServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *PermitServiceGetRequest) Query(key, value string) *PermitServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *PermitServiceGetRequest) Follow(follow string) *PermitServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *PermitServiceGetRequest) Send() (*PermitServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.PermitService.connection.URL(), p.PermitService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.PermitService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.PermitService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.PermitService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.PermitService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.PermitService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLPermitReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &PermitServiceGetResponse{permit: result}, nil
}

func (p *PermitServiceGetRequest) MustSend() *PermitServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the information about the permit of the role.
// For example to retrieve the information about the permit with the id `456` of the role with the id `123`
// send a request like this:
// ....
// GET /ovirt-engine/api/roles/123/permits/456
// ....
// [source,xml]
// ----
// <permit href="/ovirt-engine/api/roles/123/permits/456" id="456">
//   <name>change_vm_cd</name>
//   <administrative>false</administrative>
//   <role href="/ovirt-engine/api/roles/123" id="123"/>
// </permit>
// ----
//
type PermitServiceGetResponse struct {
	permit *Permit
}

func (p *PermitServiceGetResponse) Permit() (*Permit, bool) {
	if p.permit != nil {
		return p.permit, true
	}
	return nil, false
}

func (p *PermitServiceGetResponse) MustPermit() *Permit {
	if p.permit == nil {
		panic("permit in response does not exist")
	}
	return p.permit
}

//
// Gets the information about the permit of the role.
// For example to retrieve the information about the permit with the id `456` of the role with the id `123`
// send a request like this:
// ....
// GET /ovirt-engine/api/roles/123/permits/456
// ....
// [source,xml]
// ----
// <permit href="/ovirt-engine/api/roles/123/permits/456" id="456">
//   <name>change_vm_cd</name>
//   <administrative>false</administrative>
//   <role href="/ovirt-engine/api/roles/123" id="123"/>
// </permit>
// ----
//
func (p *PermitService) Get() *PermitServiceGetRequest {
	return &PermitServiceGetRequest{PermitService: p}
}

//
// Removes the permit from the role.
// For example to remove the permit with id `456` from the role with id `123` send a request like this:
// ....
// DELETE /ovirt-engine/api/roles/123/permits/456
// ....
//
type PermitServiceRemoveRequest struct {
	PermitService *PermitService
	header        map[string]string
	query         map[string]string
	async         *bool
}

func (p *PermitServiceRemoveRequest) Header(key, value string) *PermitServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *PermitServiceRemoveRequest) Query(key, value string) *PermitServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *PermitServiceRemoveRequest) Async(async bool) *PermitServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *PermitServiceRemoveRequest) Send() (*PermitServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.PermitService.connection.URL(), p.PermitService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.PermitService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.PermitService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.PermitService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.PermitService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.PermitService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(PermitServiceRemoveResponse), nil
}

func (p *PermitServiceRemoveRequest) MustSend() *PermitServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the permit from the role.
// For example to remove the permit with id `456` from the role with id `123` send a request like this:
// ....
// DELETE /ovirt-engine/api/roles/123/permits/456
// ....
//
type PermitServiceRemoveResponse struct {
}

//
// Removes the permit from the role.
// For example to remove the permit with id `456` from the role with id `123` send a request like this:
// ....
// DELETE /ovirt-engine/api/roles/123/permits/456
// ....
//
func (p *PermitService) Remove() *PermitServiceRemoveRequest {
	return &PermitServiceRemoveRequest{PermitService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *PermitService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *PermitService) String() string {
	return fmt.Sprintf("PermitService:%s", op.path)
}

//
//
type AssignedDiskProfilesService struct {
	BaseService
}

func NewAssignedDiskProfilesService(connection *Connection, path string) *AssignedDiskProfilesService {
	var result AssignedDiskProfilesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new disk profile for the storage domain.
//
type AssignedDiskProfilesServiceAddRequest struct {
	AssignedDiskProfilesService *AssignedDiskProfilesService
	header                      map[string]string
	query                       map[string]string
	profile                     *DiskProfile
}

func (p *AssignedDiskProfilesServiceAddRequest) Header(key, value string) *AssignedDiskProfilesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedDiskProfilesServiceAddRequest) Query(key, value string) *AssignedDiskProfilesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedDiskProfilesServiceAddRequest) Profile(profile *DiskProfile) *AssignedDiskProfilesServiceAddRequest {
	p.profile = profile
	return p
}

func (p *AssignedDiskProfilesServiceAddRequest) Send() (*AssignedDiskProfilesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedDiskProfilesService.connection.URL(), p.AssignedDiskProfilesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskProfileWriteOne(writer, p.profile, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedDiskProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedDiskProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedDiskProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedDiskProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedDiskProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedDiskProfilesServiceAddResponse{profile: result}, nil
}

func (p *AssignedDiskProfilesServiceAddRequest) MustSend() *AssignedDiskProfilesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new disk profile for the storage domain.
//
type AssignedDiskProfilesServiceAddResponse struct {
	profile *DiskProfile
}

func (p *AssignedDiskProfilesServiceAddResponse) Profile() (*DiskProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *AssignedDiskProfilesServiceAddResponse) MustProfile() *DiskProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Add a new disk profile for the storage domain.
//
func (p *AssignedDiskProfilesService) Add() *AssignedDiskProfilesServiceAddRequest {
	return &AssignedDiskProfilesServiceAddRequest{AssignedDiskProfilesService: p}
}

//
// Returns the list of disk profiles assigned to the storage domain.
// The order of the returned disk profiles isn't guaranteed.
//
type AssignedDiskProfilesServiceListRequest struct {
	AssignedDiskProfilesService *AssignedDiskProfilesService
	header                      map[string]string
	query                       map[string]string
	follow                      *string
	max                         *int64
}

func (p *AssignedDiskProfilesServiceListRequest) Header(key, value string) *AssignedDiskProfilesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedDiskProfilesServiceListRequest) Query(key, value string) *AssignedDiskProfilesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedDiskProfilesServiceListRequest) Follow(follow string) *AssignedDiskProfilesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AssignedDiskProfilesServiceListRequest) Max(max int64) *AssignedDiskProfilesServiceListRequest {
	p.max = &max
	return p
}

func (p *AssignedDiskProfilesServiceListRequest) Send() (*AssignedDiskProfilesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedDiskProfilesService.connection.URL(), p.AssignedDiskProfilesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedDiskProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedDiskProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedDiskProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedDiskProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedDiskProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskProfileReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AssignedDiskProfilesServiceListResponse{profiles: result}, nil
}

func (p *AssignedDiskProfilesServiceListRequest) MustSend() *AssignedDiskProfilesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of disk profiles assigned to the storage domain.
// The order of the returned disk profiles isn't guaranteed.
//
type AssignedDiskProfilesServiceListResponse struct {
	profiles *DiskProfileSlice
}

func (p *AssignedDiskProfilesServiceListResponse) Profiles() (*DiskProfileSlice, bool) {
	if p.profiles != nil {
		return p.profiles, true
	}
	return nil, false
}

func (p *AssignedDiskProfilesServiceListResponse) MustProfiles() *DiskProfileSlice {
	if p.profiles == nil {
		panic("profiles in response does not exist")
	}
	return p.profiles
}

//
// Returns the list of disk profiles assigned to the storage domain.
// The order of the returned disk profiles isn't guaranteed.
//
func (p *AssignedDiskProfilesService) List() *AssignedDiskProfilesServiceListRequest {
	return &AssignedDiskProfilesServiceListRequest{AssignedDiskProfilesService: p}
}

//
//
func (op *AssignedDiskProfilesService) ProfileService(id string) *AssignedDiskProfileService {
	return NewAssignedDiskProfileService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedDiskProfilesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProfileService(path), nil
	}
	return op.ProfileService(path[:index]).Service(path[index+1:])
}

func (op *AssignedDiskProfilesService) String() string {
	return fmt.Sprintf("AssignedDiskProfilesService:%s", op.path)
}

//
//
type GlusterHookService struct {
	BaseService
}

func NewGlusterHookService(connection *Connection, path string) *GlusterHookService {
	var result GlusterHookService
	result.connection = connection
	result.path = path
	return &result
}

//
// Resolves status conflict of hook among servers in cluster by disabling Gluster hook in all servers of the
// cluster. This updates the hook status to `DISABLED` in database.
//
type GlusterHookServiceDisableRequest struct {
	GlusterHookService *GlusterHookService
	header             map[string]string
	query              map[string]string
	async              *bool
}

func (p *GlusterHookServiceDisableRequest) Header(key, value string) *GlusterHookServiceDisableRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterHookServiceDisableRequest) Query(key, value string) *GlusterHookServiceDisableRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterHookServiceDisableRequest) Async(async bool) *GlusterHookServiceDisableRequest {
	p.async = &async
	return p
}

func (p *GlusterHookServiceDisableRequest) Send() (*GlusterHookServiceDisableResponse, error) {
	rawURL := fmt.Sprintf("%s%s/disable", p.GlusterHookService.connection.URL(), p.GlusterHookService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterHookService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterHookService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterHookService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterHookService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterHookService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterHookServiceDisableResponse), nil
}

func (p *GlusterHookServiceDisableRequest) MustSend() *GlusterHookServiceDisableResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Resolves status conflict of hook among servers in cluster by disabling Gluster hook in all servers of the
// cluster. This updates the hook status to `DISABLED` in database.
//
type GlusterHookServiceDisableResponse struct {
}

//
// Resolves status conflict of hook among servers in cluster by disabling Gluster hook in all servers of the
// cluster. This updates the hook status to `DISABLED` in database.
//
func (p *GlusterHookService) Disable() *GlusterHookServiceDisableRequest {
	return &GlusterHookServiceDisableRequest{GlusterHookService: p}
}

//
// Resolves status conflict of hook among servers in cluster by disabling Gluster hook in all servers of the
// cluster. This updates the hook status to `DISABLED` in database.
//
type GlusterHookServiceEnableRequest struct {
	GlusterHookService *GlusterHookService
	header             map[string]string
	query              map[string]string
	async              *bool
}

func (p *GlusterHookServiceEnableRequest) Header(key, value string) *GlusterHookServiceEnableRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterHookServiceEnableRequest) Query(key, value string) *GlusterHookServiceEnableRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterHookServiceEnableRequest) Async(async bool) *GlusterHookServiceEnableRequest {
	p.async = &async
	return p
}

func (p *GlusterHookServiceEnableRequest) Send() (*GlusterHookServiceEnableResponse, error) {
	rawURL := fmt.Sprintf("%s%s/enable", p.GlusterHookService.connection.URL(), p.GlusterHookService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterHookService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterHookService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterHookService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterHookService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterHookService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterHookServiceEnableResponse), nil
}

func (p *GlusterHookServiceEnableRequest) MustSend() *GlusterHookServiceEnableResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Resolves status conflict of hook among servers in cluster by disabling Gluster hook in all servers of the
// cluster. This updates the hook status to `DISABLED` in database.
//
type GlusterHookServiceEnableResponse struct {
}

//
// Resolves status conflict of hook among servers in cluster by disabling Gluster hook in all servers of the
// cluster. This updates the hook status to `DISABLED` in database.
//
func (p *GlusterHookService) Enable() *GlusterHookServiceEnableRequest {
	return &GlusterHookServiceEnableRequest{GlusterHookService: p}
}

//
//
type GlusterHookServiceGetRequest struct {
	GlusterHookService *GlusterHookService
	header             map[string]string
	query              map[string]string
	follow             *string
}

func (p *GlusterHookServiceGetRequest) Header(key, value string) *GlusterHookServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterHookServiceGetRequest) Query(key, value string) *GlusterHookServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterHookServiceGetRequest) Follow(follow string) *GlusterHookServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *GlusterHookServiceGetRequest) Send() (*GlusterHookServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterHookService.connection.URL(), p.GlusterHookService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterHookService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterHookService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterHookService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterHookService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterHookService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterHookReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &GlusterHookServiceGetResponse{hook: result}, nil
}

func (p *GlusterHookServiceGetRequest) MustSend() *GlusterHookServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type GlusterHookServiceGetResponse struct {
	hook *GlusterHook
}

func (p *GlusterHookServiceGetResponse) Hook() (*GlusterHook, bool) {
	if p.hook != nil {
		return p.hook, true
	}
	return nil, false
}

func (p *GlusterHookServiceGetResponse) MustHook() *GlusterHook {
	if p.hook == nil {
		panic("hook in response does not exist")
	}
	return p.hook
}

//
//
func (p *GlusterHookService) Get() *GlusterHookServiceGetRequest {
	return &GlusterHookServiceGetRequest{GlusterHookService: p}
}

//
// Removes the this Gluster hook from all servers in cluster and deletes it from the database.
//
type GlusterHookServiceRemoveRequest struct {
	GlusterHookService *GlusterHookService
	header             map[string]string
	query              map[string]string
	async              *bool
}

func (p *GlusterHookServiceRemoveRequest) Header(key, value string) *GlusterHookServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterHookServiceRemoveRequest) Query(key, value string) *GlusterHookServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterHookServiceRemoveRequest) Async(async bool) *GlusterHookServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *GlusterHookServiceRemoveRequest) Send() (*GlusterHookServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterHookService.connection.URL(), p.GlusterHookService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterHookService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterHookService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterHookService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterHookService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterHookService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(GlusterHookServiceRemoveResponse), nil
}

func (p *GlusterHookServiceRemoveRequest) MustSend() *GlusterHookServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the this Gluster hook from all servers in cluster and deletes it from the database.
//
type GlusterHookServiceRemoveResponse struct {
}

//
// Removes the this Gluster hook from all servers in cluster and deletes it from the database.
//
func (p *GlusterHookService) Remove() *GlusterHookServiceRemoveRequest {
	return &GlusterHookServiceRemoveRequest{GlusterHookService: p}
}

//
// Resolves missing hook conflict depending on the resolution type.
// For `ADD` resolves by copying hook stored in engine database to all servers where the hook is missing. The
// engine maintains a list of all servers where hook is missing.
// For `COPY` resolves conflict in hook content by copying hook stored in engine database to all servers where
// the hook is missing. The engine maintains a list of all servers where the content is conflicting. If a host
// id is passed as parameter, the hook content from the server is used as the master to copy to other servers
// in cluster.
//
type GlusterHookServiceResolveRequest struct {
	GlusterHookService *GlusterHookService
	header             map[string]string
	query              map[string]string
	async              *bool
	host               *Host
	resolutionType     *string
}

func (p *GlusterHookServiceResolveRequest) Header(key, value string) *GlusterHookServiceResolveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterHookServiceResolveRequest) Query(key, value string) *GlusterHookServiceResolveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterHookServiceResolveRequest) Async(async bool) *GlusterHookServiceResolveRequest {
	p.async = &async
	return p
}

func (p *GlusterHookServiceResolveRequest) Host(host *Host) *GlusterHookServiceResolveRequest {
	p.host = host
	return p
}

func (p *GlusterHookServiceResolveRequest) ResolutionType(resolutionType string) *GlusterHookServiceResolveRequest {
	p.resolutionType = &resolutionType
	return p
}

func (p *GlusterHookServiceResolveRequest) Send() (*GlusterHookServiceResolveResponse, error) {
	rawURL := fmt.Sprintf("%s%s/resolve", p.GlusterHookService.connection.URL(), p.GlusterHookService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Host(p.host)
	if p.resolutionType != nil {
		actionBuilder.ResolutionType(*p.resolutionType)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterHookService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterHookService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterHookService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterHookService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterHookService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterHookServiceResolveResponse), nil
}

func (p *GlusterHookServiceResolveRequest) MustSend() *GlusterHookServiceResolveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Resolves missing hook conflict depending on the resolution type.
// For `ADD` resolves by copying hook stored in engine database to all servers where the hook is missing. The
// engine maintains a list of all servers where hook is missing.
// For `COPY` resolves conflict in hook content by copying hook stored in engine database to all servers where
// the hook is missing. The engine maintains a list of all servers where the content is conflicting. If a host
// id is passed as parameter, the hook content from the server is used as the master to copy to other servers
// in cluster.
//
type GlusterHookServiceResolveResponse struct {
}

//
// Resolves missing hook conflict depending on the resolution type.
// For `ADD` resolves by copying hook stored in engine database to all servers where the hook is missing. The
// engine maintains a list of all servers where hook is missing.
// For `COPY` resolves conflict in hook content by copying hook stored in engine database to all servers where
// the hook is missing. The engine maintains a list of all servers where the content is conflicting. If a host
// id is passed as parameter, the hook content from the server is used as the master to copy to other servers
// in cluster.
//
func (p *GlusterHookService) Resolve() *GlusterHookServiceResolveRequest {
	return &GlusterHookServiceResolveRequest{GlusterHookService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GlusterHookService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *GlusterHookService) String() string {
	return fmt.Sprintf("GlusterHookService:%s", op.path)
}

//
//
type GlusterHooksService struct {
	BaseService
}

func NewGlusterHooksService(connection *Connection, path string) *GlusterHooksService {
	var result GlusterHooksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of hooks.
// The order of the returned list of hooks isn't guaranteed.
//
type GlusterHooksServiceListRequest struct {
	GlusterHooksService *GlusterHooksService
	header              map[string]string
	query               map[string]string
	follow              *string
	max                 *int64
}

func (p *GlusterHooksServiceListRequest) Header(key, value string) *GlusterHooksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterHooksServiceListRequest) Query(key, value string) *GlusterHooksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterHooksServiceListRequest) Follow(follow string) *GlusterHooksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *GlusterHooksServiceListRequest) Max(max int64) *GlusterHooksServiceListRequest {
	p.max = &max
	return p
}

func (p *GlusterHooksServiceListRequest) Send() (*GlusterHooksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterHooksService.connection.URL(), p.GlusterHooksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterHooksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterHooksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterHooksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterHooksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterHooksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterHookReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &GlusterHooksServiceListResponse{hooks: result}, nil
}

func (p *GlusterHooksServiceListRequest) MustSend() *GlusterHooksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of hooks.
// The order of the returned list of hooks isn't guaranteed.
//
type GlusterHooksServiceListResponse struct {
	hooks *GlusterHookSlice
}

func (p *GlusterHooksServiceListResponse) Hooks() (*GlusterHookSlice, bool) {
	if p.hooks != nil {
		return p.hooks, true
	}
	return nil, false
}

func (p *GlusterHooksServiceListResponse) MustHooks() *GlusterHookSlice {
	if p.hooks == nil {
		panic("hooks in response does not exist")
	}
	return p.hooks
}

//
// Returns the list of hooks.
// The order of the returned list of hooks isn't guaranteed.
//
func (p *GlusterHooksService) List() *GlusterHooksServiceListRequest {
	return &GlusterHooksServiceListRequest{GlusterHooksService: p}
}

//
//
func (op *GlusterHooksService) HookService(id string) *GlusterHookService {
	return NewGlusterHookService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GlusterHooksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.HookService(path), nil
	}
	return op.HookService(path[:index]).Service(path[index+1:])
}

func (op *GlusterHooksService) String() string {
	return fmt.Sprintf("GlusterHooksService:%s", op.path)
}

//
// This service manages a single gluster brick.
//
type GlusterBrickService struct {
	BaseService
}

func NewGlusterBrickService(connection *Connection, path string) *GlusterBrickService {
	var result GlusterBrickService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get details of a brick.
// Retrieves status details of brick from underlying gluster volume with header `All-Content` set to `true`. This is
// the equivalent of running `gluster volume status <volumename> <brickname> detail`.
// For example, to get the details of brick `234` of gluster volume `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/234
// ----
// Which will return a response body like this:
// [source,xml]
// ----
// <brick id="234">
//   <name>host1:/rhgs/data/brick1</name>
//   <brick_dir>/rhgs/data/brick1</brick_dir>
//   <server_id>111</server_id>
//   <status>up</status>
//   <device>/dev/mapper/RHGS_vg1-lv_vmaddldisks</device>
//   <fs_name>xfs</fs_name>
//   <gluster_clients>
//     <gluster_client>
//       <bytes_read>2818417648</bytes_read>
//       <bytes_written>1384694844</bytes_written>
//       <client_port>1011</client_port>
//       <host_name>client2</host_name>
//     </gluster_client>
//   </gluster_clients>
//   <memory_pools>
//     <memory_pool>
//       <name>data-server:fd_t</name>
//       <alloc_count>1626348</alloc_count>
//       <cold_count>1020</cold_count>
//       <hot_count>4</hot_count>
//       <max_alloc>23</max_alloc>
//       <max_stdalloc>0</max_stdalloc>
//       <padded_size>140</padded_size>
//       <pool_misses>0</pool_misses>
//     </memory_pool>
//   </memory_pools>
//   <mnt_options>rw,seclabel,noatime,nodiratime,attr2,inode64,sunit=512,swidth=2048,noquota</mnt_options>
//   <pid>25589</pid>
//   <port>49155</port>
// </brick>
// ----
//
type GlusterBrickServiceGetRequest struct {
	GlusterBrickService *GlusterBrickService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *GlusterBrickServiceGetRequest) Header(key, value string) *GlusterBrickServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBrickServiceGetRequest) Query(key, value string) *GlusterBrickServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBrickServiceGetRequest) Follow(follow string) *GlusterBrickServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *GlusterBrickServiceGetRequest) Send() (*GlusterBrickServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterBrickService.connection.URL(), p.GlusterBrickService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBrickService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBrickService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBrickService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBrickService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBrickService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterBrickReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &GlusterBrickServiceGetResponse{brick: result}, nil
}

func (p *GlusterBrickServiceGetRequest) MustSend() *GlusterBrickServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get details of a brick.
// Retrieves status details of brick from underlying gluster volume with header `All-Content` set to `true`. This is
// the equivalent of running `gluster volume status <volumename> <brickname> detail`.
// For example, to get the details of brick `234` of gluster volume `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/234
// ----
// Which will return a response body like this:
// [source,xml]
// ----
// <brick id="234">
//   <name>host1:/rhgs/data/brick1</name>
//   <brick_dir>/rhgs/data/brick1</brick_dir>
//   <server_id>111</server_id>
//   <status>up</status>
//   <device>/dev/mapper/RHGS_vg1-lv_vmaddldisks</device>
//   <fs_name>xfs</fs_name>
//   <gluster_clients>
//     <gluster_client>
//       <bytes_read>2818417648</bytes_read>
//       <bytes_written>1384694844</bytes_written>
//       <client_port>1011</client_port>
//       <host_name>client2</host_name>
//     </gluster_client>
//   </gluster_clients>
//   <memory_pools>
//     <memory_pool>
//       <name>data-server:fd_t</name>
//       <alloc_count>1626348</alloc_count>
//       <cold_count>1020</cold_count>
//       <hot_count>4</hot_count>
//       <max_alloc>23</max_alloc>
//       <max_stdalloc>0</max_stdalloc>
//       <padded_size>140</padded_size>
//       <pool_misses>0</pool_misses>
//     </memory_pool>
//   </memory_pools>
//   <mnt_options>rw,seclabel,noatime,nodiratime,attr2,inode64,sunit=512,swidth=2048,noquota</mnt_options>
//   <pid>25589</pid>
//   <port>49155</port>
// </brick>
// ----
//
type GlusterBrickServiceGetResponse struct {
	brick *GlusterBrick
}

func (p *GlusterBrickServiceGetResponse) Brick() (*GlusterBrick, bool) {
	if p.brick != nil {
		return p.brick, true
	}
	return nil, false
}

func (p *GlusterBrickServiceGetResponse) MustBrick() *GlusterBrick {
	if p.brick == nil {
		panic("brick in response does not exist")
	}
	return p.brick
}

//
// Get details of a brick.
// Retrieves status details of brick from underlying gluster volume with header `All-Content` set to `true`. This is
// the equivalent of running `gluster volume status <volumename> <brickname> detail`.
// For example, to get the details of brick `234` of gluster volume `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/234
// ----
// Which will return a response body like this:
// [source,xml]
// ----
// <brick id="234">
//   <name>host1:/rhgs/data/brick1</name>
//   <brick_dir>/rhgs/data/brick1</brick_dir>
//   <server_id>111</server_id>
//   <status>up</status>
//   <device>/dev/mapper/RHGS_vg1-lv_vmaddldisks</device>
//   <fs_name>xfs</fs_name>
//   <gluster_clients>
//     <gluster_client>
//       <bytes_read>2818417648</bytes_read>
//       <bytes_written>1384694844</bytes_written>
//       <client_port>1011</client_port>
//       <host_name>client2</host_name>
//     </gluster_client>
//   </gluster_clients>
//   <memory_pools>
//     <memory_pool>
//       <name>data-server:fd_t</name>
//       <alloc_count>1626348</alloc_count>
//       <cold_count>1020</cold_count>
//       <hot_count>4</hot_count>
//       <max_alloc>23</max_alloc>
//       <max_stdalloc>0</max_stdalloc>
//       <padded_size>140</padded_size>
//       <pool_misses>0</pool_misses>
//     </memory_pool>
//   </memory_pools>
//   <mnt_options>rw,seclabel,noatime,nodiratime,attr2,inode64,sunit=512,swidth=2048,noquota</mnt_options>
//   <pid>25589</pid>
//   <port>49155</port>
// </brick>
// ----
//
func (p *GlusterBrickService) Get() *GlusterBrickServiceGetRequest {
	return &GlusterBrickServiceGetRequest{GlusterBrickService: p}
}

//
// Removes a brick.
// Removes a brick from the underlying gluster volume and deletes entries from database. This can be used only when
// removing a single brick without data migration. To remove multiple bricks and with data migration, use
// <<services/gluster_bricks/methods/migrate, migrate>> instead.
// For example, to delete brick `234` from gluster volume `123`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/234
// ----
//
type GlusterBrickServiceRemoveRequest struct {
	GlusterBrickService *GlusterBrickService
	header              map[string]string
	query               map[string]string
	async               *bool
}

func (p *GlusterBrickServiceRemoveRequest) Header(key, value string) *GlusterBrickServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBrickServiceRemoveRequest) Query(key, value string) *GlusterBrickServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBrickServiceRemoveRequest) Async(async bool) *GlusterBrickServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *GlusterBrickServiceRemoveRequest) Send() (*GlusterBrickServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterBrickService.connection.URL(), p.GlusterBrickService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBrickService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBrickService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBrickService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBrickService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBrickService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(GlusterBrickServiceRemoveResponse), nil
}

func (p *GlusterBrickServiceRemoveRequest) MustSend() *GlusterBrickServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a brick.
// Removes a brick from the underlying gluster volume and deletes entries from database. This can be used only when
// removing a single brick without data migration. To remove multiple bricks and with data migration, use
// <<services/gluster_bricks/methods/migrate, migrate>> instead.
// For example, to delete brick `234` from gluster volume `123`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/234
// ----
//
type GlusterBrickServiceRemoveResponse struct {
}

//
// Removes a brick.
// Removes a brick from the underlying gluster volume and deletes entries from database. This can be used only when
// removing a single brick without data migration. To remove multiple bricks and with data migration, use
// <<services/gluster_bricks/methods/migrate, migrate>> instead.
// For example, to delete brick `234` from gluster volume `123`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/234
// ----
//
func (p *GlusterBrickService) Remove() *GlusterBrickServiceRemoveRequest {
	return &GlusterBrickServiceRemoveRequest{GlusterBrickService: p}
}

//
// Replaces this brick with a new one.
// IMPORTANT: This operation has been deprecated since version 3.5 of the engine and will be removed in the future.
// Use <<services/gluster_bricks/methods/add, add brick(s)>> and
// <<services/gluster_bricks/methods/migrate, migrate brick(s)>> instead.
//
type GlusterBrickServiceReplaceRequest struct {
	GlusterBrickService *GlusterBrickService
	header              map[string]string
	query               map[string]string
	async               *bool
	force               *bool
}

func (p *GlusterBrickServiceReplaceRequest) Header(key, value string) *GlusterBrickServiceReplaceRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBrickServiceReplaceRequest) Query(key, value string) *GlusterBrickServiceReplaceRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBrickServiceReplaceRequest) Async(async bool) *GlusterBrickServiceReplaceRequest {
	p.async = &async
	return p
}

func (p *GlusterBrickServiceReplaceRequest) Force(force bool) *GlusterBrickServiceReplaceRequest {
	p.force = &force
	return p
}

func (p *GlusterBrickServiceReplaceRequest) Send() (*GlusterBrickServiceReplaceResponse, error) {
	rawURL := fmt.Sprintf("%s%s/replace", p.GlusterBrickService.connection.URL(), p.GlusterBrickService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBrickService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBrickService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBrickService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBrickService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBrickService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterBrickServiceReplaceResponse), nil
}

func (p *GlusterBrickServiceReplaceRequest) MustSend() *GlusterBrickServiceReplaceResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Replaces this brick with a new one.
// IMPORTANT: This operation has been deprecated since version 3.5 of the engine and will be removed in the future.
// Use <<services/gluster_bricks/methods/add, add brick(s)>> and
// <<services/gluster_bricks/methods/migrate, migrate brick(s)>> instead.
//
type GlusterBrickServiceReplaceResponse struct {
}

//
// Replaces this brick with a new one.
// IMPORTANT: This operation has been deprecated since version 3.5 of the engine and will be removed in the future.
// Use <<services/gluster_bricks/methods/add, add brick(s)>> and
// <<services/gluster_bricks/methods/migrate, migrate brick(s)>> instead.
//
func (p *GlusterBrickService) Replace() *GlusterBrickServiceReplaceRequest {
	return &GlusterBrickServiceReplaceRequest{GlusterBrickService: p}
}

//
//
func (op *GlusterBrickService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GlusterBrickService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *GlusterBrickService) String() string {
	return fmt.Sprintf("GlusterBrickService:%s", op.path)
}

//
// This service manages a collection of gluster volumes available in a cluster.
//
type GlusterVolumesService struct {
	BaseService
}

func NewGlusterVolumesService(connection *Connection, path string) *GlusterVolumesService {
	var result GlusterVolumesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new gluster volume.
// The volume is created based on properties of the `volume` parameter. The properties `name`, `volume_type` and
// `bricks` are required.
// For example, to add a volume with name `myvolume` to the cluster `123`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/glustervolumes
// ----
// With the following request body:
// [source,xml]
// ----
// <gluster_volume>
//   <name>myvolume</name>
//   <volume_type>replicate</volume_type>
//   <replica_count>3</replica_count>
//   <bricks>
//     <brick>
//       <server_id>server1</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//     <brick>
//       <server_id>server2</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//     <brick>
//       <server_id>server3</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//   <bricks>
// </gluster_volume>
// ----
//
type GlusterVolumesServiceAddRequest struct {
	GlusterVolumesService *GlusterVolumesService
	header                map[string]string
	query                 map[string]string
	volume                *GlusterVolume
}

func (p *GlusterVolumesServiceAddRequest) Header(key, value string) *GlusterVolumesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumesServiceAddRequest) Query(key, value string) *GlusterVolumesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumesServiceAddRequest) Volume(volume *GlusterVolume) *GlusterVolumesServiceAddRequest {
	p.volume = volume
	return p
}

func (p *GlusterVolumesServiceAddRequest) Send() (*GlusterVolumesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterVolumesService.connection.URL(), p.GlusterVolumesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLGlusterVolumeWriteOne(writer, p.volume, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterVolumeReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &GlusterVolumesServiceAddResponse{volume: result}, nil
}

func (p *GlusterVolumesServiceAddRequest) MustSend() *GlusterVolumesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new gluster volume.
// The volume is created based on properties of the `volume` parameter. The properties `name`, `volume_type` and
// `bricks` are required.
// For example, to add a volume with name `myvolume` to the cluster `123`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/glustervolumes
// ----
// With the following request body:
// [source,xml]
// ----
// <gluster_volume>
//   <name>myvolume</name>
//   <volume_type>replicate</volume_type>
//   <replica_count>3</replica_count>
//   <bricks>
//     <brick>
//       <server_id>server1</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//     <brick>
//       <server_id>server2</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//     <brick>
//       <server_id>server3</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//   <bricks>
// </gluster_volume>
// ----
//
type GlusterVolumesServiceAddResponse struct {
	volume *GlusterVolume
}

func (p *GlusterVolumesServiceAddResponse) Volume() (*GlusterVolume, bool) {
	if p.volume != nil {
		return p.volume, true
	}
	return nil, false
}

func (p *GlusterVolumesServiceAddResponse) MustVolume() *GlusterVolume {
	if p.volume == nil {
		panic("volume in response does not exist")
	}
	return p.volume
}

//
// Creates a new gluster volume.
// The volume is created based on properties of the `volume` parameter. The properties `name`, `volume_type` and
// `bricks` are required.
// For example, to add a volume with name `myvolume` to the cluster `123`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/glustervolumes
// ----
// With the following request body:
// [source,xml]
// ----
// <gluster_volume>
//   <name>myvolume</name>
//   <volume_type>replicate</volume_type>
//   <replica_count>3</replica_count>
//   <bricks>
//     <brick>
//       <server_id>server1</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//     <brick>
//       <server_id>server2</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//     <brick>
//       <server_id>server3</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//   <bricks>
// </gluster_volume>
// ----
//
func (p *GlusterVolumesService) Add() *GlusterVolumesServiceAddRequest {
	return &GlusterVolumesServiceAddRequest{GlusterVolumesService: p}
}

//
// Lists all gluster volumes in the cluster.
// For example, to list all Gluster Volumes in cluster `456`, send a request like
// this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/456/glustervolumes
// ----
// The order of the returned list of volumes isn't guaranteed.
//
type GlusterVolumesServiceListRequest struct {
	GlusterVolumesService *GlusterVolumesService
	header                map[string]string
	query                 map[string]string
	caseSensitive         *bool
	follow                *string
	max                   *int64
	search                *string
}

func (p *GlusterVolumesServiceListRequest) Header(key, value string) *GlusterVolumesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumesServiceListRequest) Query(key, value string) *GlusterVolumesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumesServiceListRequest) CaseSensitive(caseSensitive bool) *GlusterVolumesServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *GlusterVolumesServiceListRequest) Follow(follow string) *GlusterVolumesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *GlusterVolumesServiceListRequest) Max(max int64) *GlusterVolumesServiceListRequest {
	p.max = &max
	return p
}

func (p *GlusterVolumesServiceListRequest) Search(search string) *GlusterVolumesServiceListRequest {
	p.search = &search
	return p
}

func (p *GlusterVolumesServiceListRequest) Send() (*GlusterVolumesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterVolumesService.connection.URL(), p.GlusterVolumesService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterVolumeReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &GlusterVolumesServiceListResponse{volumes: result}, nil
}

func (p *GlusterVolumesServiceListRequest) MustSend() *GlusterVolumesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists all gluster volumes in the cluster.
// For example, to list all Gluster Volumes in cluster `456`, send a request like
// this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/456/glustervolumes
// ----
// The order of the returned list of volumes isn't guaranteed.
//
type GlusterVolumesServiceListResponse struct {
	volumes *GlusterVolumeSlice
}

func (p *GlusterVolumesServiceListResponse) Volumes() (*GlusterVolumeSlice, bool) {
	if p.volumes != nil {
		return p.volumes, true
	}
	return nil, false
}

func (p *GlusterVolumesServiceListResponse) MustVolumes() *GlusterVolumeSlice {
	if p.volumes == nil {
		panic("volumes in response does not exist")
	}
	return p.volumes
}

//
// Lists all gluster volumes in the cluster.
// For example, to list all Gluster Volumes in cluster `456`, send a request like
// this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/456/glustervolumes
// ----
// The order of the returned list of volumes isn't guaranteed.
//
func (p *GlusterVolumesService) List() *GlusterVolumesServiceListRequest {
	return &GlusterVolumesServiceListRequest{GlusterVolumesService: p}
}

//
// Reference to a service managing gluster volume.
//
func (op *GlusterVolumesService) VolumeService(id string) *GlusterVolumeService {
	return NewGlusterVolumeService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GlusterVolumesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.VolumeService(path), nil
	}
	return op.VolumeService(path[:index]).Service(path[index+1:])
}

func (op *GlusterVolumesService) String() string {
	return fmt.Sprintf("GlusterVolumesService:%s", op.path)
}

//
// This service manages a single gluster volume.
//
type GlusterVolumeService struct {
	BaseService
}

func NewGlusterVolumeService(connection *Connection, path string) *GlusterVolumeService {
	var result GlusterVolumeService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get the gluster volume details.
// For example, to get details of a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/456/glustervolumes/123
// ----
// This GET request will return the following output:
// [source,xml]
// ----
// <gluster_volume id="123">
//  <name>data</name>
//  <link href="/ovirt-engine/api/clusters/456/glustervolumes/123/glusterbricks" rel="glusterbricks"/>
//  <disperse_count>0</disperse_count>
//  <options>
//    <option>
//      <name>storage.owner-gid</name>
//      <value>36</value>
//    </option>
//    <option>
//      <name>performance.io-cache</name>
//      <value>off</value>
//    </option>
//    <option>
//      <name>cluster.data-self-heal-algorithm</name>
//      <value>full</value>
//    </option>
//  </options>
//  <redundancy_count>0</redundancy_count>
//  <replica_count>3</replica_count>
//  <status>up</status>
//  <stripe_count>0</stripe_count>
//  <transport_types>
//    <transport_type>tcp</transport_type>
//  </transport_types>
//  <volume_type>replicate</volume_type>
//  </gluster_volume>
// ----
//
type GlusterVolumeServiceGetRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	follow               *string
}

func (p *GlusterVolumeServiceGetRequest) Header(key, value string) *GlusterVolumeServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceGetRequest) Query(key, value string) *GlusterVolumeServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceGetRequest) Follow(follow string) *GlusterVolumeServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *GlusterVolumeServiceGetRequest) Send() (*GlusterVolumeServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterVolumeReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &GlusterVolumeServiceGetResponse{volume: result}, nil
}

func (p *GlusterVolumeServiceGetRequest) MustSend() *GlusterVolumeServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get the gluster volume details.
// For example, to get details of a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/456/glustervolumes/123
// ----
// This GET request will return the following output:
// [source,xml]
// ----
// <gluster_volume id="123">
//  <name>data</name>
//  <link href="/ovirt-engine/api/clusters/456/glustervolumes/123/glusterbricks" rel="glusterbricks"/>
//  <disperse_count>0</disperse_count>
//  <options>
//    <option>
//      <name>storage.owner-gid</name>
//      <value>36</value>
//    </option>
//    <option>
//      <name>performance.io-cache</name>
//      <value>off</value>
//    </option>
//    <option>
//      <name>cluster.data-self-heal-algorithm</name>
//      <value>full</value>
//    </option>
//  </options>
//  <redundancy_count>0</redundancy_count>
//  <replica_count>3</replica_count>
//  <status>up</status>
//  <stripe_count>0</stripe_count>
//  <transport_types>
//    <transport_type>tcp</transport_type>
//  </transport_types>
//  <volume_type>replicate</volume_type>
//  </gluster_volume>
// ----
//
type GlusterVolumeServiceGetResponse struct {
	volume *GlusterVolume
}

func (p *GlusterVolumeServiceGetResponse) Volume() (*GlusterVolume, bool) {
	if p.volume != nil {
		return p.volume, true
	}
	return nil, false
}

func (p *GlusterVolumeServiceGetResponse) MustVolume() *GlusterVolume {
	if p.volume == nil {
		panic("volume in response does not exist")
	}
	return p.volume
}

//
// Get the gluster volume details.
// For example, to get details of a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/456/glustervolumes/123
// ----
// This GET request will return the following output:
// [source,xml]
// ----
// <gluster_volume id="123">
//  <name>data</name>
//  <link href="/ovirt-engine/api/clusters/456/glustervolumes/123/glusterbricks" rel="glusterbricks"/>
//  <disperse_count>0</disperse_count>
//  <options>
//    <option>
//      <name>storage.owner-gid</name>
//      <value>36</value>
//    </option>
//    <option>
//      <name>performance.io-cache</name>
//      <value>off</value>
//    </option>
//    <option>
//      <name>cluster.data-self-heal-algorithm</name>
//      <value>full</value>
//    </option>
//  </options>
//  <redundancy_count>0</redundancy_count>
//  <replica_count>3</replica_count>
//  <status>up</status>
//  <stripe_count>0</stripe_count>
//  <transport_types>
//    <transport_type>tcp</transport_type>
//  </transport_types>
//  <volume_type>replicate</volume_type>
//  </gluster_volume>
// ----
//
func (p *GlusterVolumeService) Get() *GlusterVolumeServiceGetRequest {
	return &GlusterVolumeServiceGetRequest{GlusterVolumeService: p}
}

//
// Get gluster volume profile statistics.
// For example, to get profile statistics for a gluster volume with identifier `123` in cluster `456`, send a
// request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/getprofilestatistics
// ----
//
type GlusterVolumeServiceGetProfileStatisticsRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
}

func (p *GlusterVolumeServiceGetProfileStatisticsRequest) Header(key, value string) *GlusterVolumeServiceGetProfileStatisticsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceGetProfileStatisticsRequest) Query(key, value string) *GlusterVolumeServiceGetProfileStatisticsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceGetProfileStatisticsRequest) Send() (*GlusterVolumeServiceGetProfileStatisticsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/getprofilestatistics", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustDetails()
	return &GlusterVolumeServiceGetProfileStatisticsResponse{details: result}, nil
}

func (p *GlusterVolumeServiceGetProfileStatisticsRequest) MustSend() *GlusterVolumeServiceGetProfileStatisticsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get gluster volume profile statistics.
// For example, to get profile statistics for a gluster volume with identifier `123` in cluster `456`, send a
// request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/getprofilestatistics
// ----
//
type GlusterVolumeServiceGetProfileStatisticsResponse struct {
	details *GlusterVolumeProfileDetails
}

func (p *GlusterVolumeServiceGetProfileStatisticsResponse) Details() (*GlusterVolumeProfileDetails, bool) {
	if p.details != nil {
		return p.details, true
	}
	return nil, false
}

func (p *GlusterVolumeServiceGetProfileStatisticsResponse) MustDetails() *GlusterVolumeProfileDetails {
	if p.details == nil {
		panic("details in response does not exist")
	}
	return p.details
}

//
// Get gluster volume profile statistics.
// For example, to get profile statistics for a gluster volume with identifier `123` in cluster `456`, send a
// request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/getprofilestatistics
// ----
//
func (p *GlusterVolumeService) GetProfileStatistics() *GlusterVolumeServiceGetProfileStatisticsRequest {
	return &GlusterVolumeServiceGetProfileStatisticsRequest{GlusterVolumeService: p}
}

//
// Rebalance the gluster volume.
// Rebalancing a gluster volume helps to distribute the data evenly across all the bricks. After expanding or
// shrinking a gluster volume (without migrating data), we need to rebalance the data among the bricks. In a
// non-replicated volume, all bricks should be online to perform the rebalance operation. In a replicated volume, at
// least one of the bricks in the replica should be online.
// For example, to rebalance a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/rebalance
// ----
//
type GlusterVolumeServiceRebalanceRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
	fixLayout            *bool
	force                *bool
}

func (p *GlusterVolumeServiceRebalanceRequest) Header(key, value string) *GlusterVolumeServiceRebalanceRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceRebalanceRequest) Query(key, value string) *GlusterVolumeServiceRebalanceRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceRebalanceRequest) Async(async bool) *GlusterVolumeServiceRebalanceRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceRebalanceRequest) FixLayout(fixLayout bool) *GlusterVolumeServiceRebalanceRequest {
	p.fixLayout = &fixLayout
	return p
}

func (p *GlusterVolumeServiceRebalanceRequest) Force(force bool) *GlusterVolumeServiceRebalanceRequest {
	p.force = &force
	return p
}

func (p *GlusterVolumeServiceRebalanceRequest) Send() (*GlusterVolumeServiceRebalanceResponse, error) {
	rawURL := fmt.Sprintf("%s%s/rebalance", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.fixLayout != nil {
		actionBuilder.FixLayout(*p.fixLayout)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceRebalanceResponse), nil
}

func (p *GlusterVolumeServiceRebalanceRequest) MustSend() *GlusterVolumeServiceRebalanceResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Rebalance the gluster volume.
// Rebalancing a gluster volume helps to distribute the data evenly across all the bricks. After expanding or
// shrinking a gluster volume (without migrating data), we need to rebalance the data among the bricks. In a
// non-replicated volume, all bricks should be online to perform the rebalance operation. In a replicated volume, at
// least one of the bricks in the replica should be online.
// For example, to rebalance a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/rebalance
// ----
//
type GlusterVolumeServiceRebalanceResponse struct {
}

//
// Rebalance the gluster volume.
// Rebalancing a gluster volume helps to distribute the data evenly across all the bricks. After expanding or
// shrinking a gluster volume (without migrating data), we need to rebalance the data among the bricks. In a
// non-replicated volume, all bricks should be online to perform the rebalance operation. In a replicated volume, at
// least one of the bricks in the replica should be online.
// For example, to rebalance a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/rebalance
// ----
//
func (p *GlusterVolumeService) Rebalance() *GlusterVolumeServiceRebalanceRequest {
	return &GlusterVolumeServiceRebalanceRequest{GlusterVolumeService: p}
}

//
// Removes the gluster volume.
// For example, to remove a volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/456/glustervolumes/123
// ----
//
type GlusterVolumeServiceRemoveRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
}

func (p *GlusterVolumeServiceRemoveRequest) Header(key, value string) *GlusterVolumeServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceRemoveRequest) Query(key, value string) *GlusterVolumeServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceRemoveRequest) Async(async bool) *GlusterVolumeServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceRemoveRequest) Send() (*GlusterVolumeServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(GlusterVolumeServiceRemoveResponse), nil
}

func (p *GlusterVolumeServiceRemoveRequest) MustSend() *GlusterVolumeServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the gluster volume.
// For example, to remove a volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/456/glustervolumes/123
// ----
//
type GlusterVolumeServiceRemoveResponse struct {
}

//
// Removes the gluster volume.
// For example, to remove a volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/456/glustervolumes/123
// ----
//
func (p *GlusterVolumeService) Remove() *GlusterVolumeServiceRemoveRequest {
	return &GlusterVolumeServiceRemoveRequest{GlusterVolumeService: p}
}

//
// Resets all the options set in the gluster volume.
// For example, to reset all options in a gluster volume with identifier `123` in cluster `456`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/resetalloptions
// ----
//
type GlusterVolumeServiceResetAllOptionsRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
}

func (p *GlusterVolumeServiceResetAllOptionsRequest) Header(key, value string) *GlusterVolumeServiceResetAllOptionsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceResetAllOptionsRequest) Query(key, value string) *GlusterVolumeServiceResetAllOptionsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceResetAllOptionsRequest) Async(async bool) *GlusterVolumeServiceResetAllOptionsRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceResetAllOptionsRequest) Send() (*GlusterVolumeServiceResetAllOptionsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/resetalloptions", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceResetAllOptionsResponse), nil
}

func (p *GlusterVolumeServiceResetAllOptionsRequest) MustSend() *GlusterVolumeServiceResetAllOptionsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Resets all the options set in the gluster volume.
// For example, to reset all options in a gluster volume with identifier `123` in cluster `456`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/resetalloptions
// ----
//
type GlusterVolumeServiceResetAllOptionsResponse struct {
}

//
// Resets all the options set in the gluster volume.
// For example, to reset all options in a gluster volume with identifier `123` in cluster `456`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/resetalloptions
// ----
//
func (p *GlusterVolumeService) ResetAllOptions() *GlusterVolumeServiceResetAllOptionsRequest {
	return &GlusterVolumeServiceResetAllOptionsRequest{GlusterVolumeService: p}
}

//
// Resets a particular option in the gluster volume.
// For example, to reset a particular option `option1` in a gluster volume with identifier `123` in cluster `456`,
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/resetoption
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//  <option name="option1"/>
// </action>
// ----
//
type GlusterVolumeServiceResetOptionRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
	force                *bool
	option               *Option
}

func (p *GlusterVolumeServiceResetOptionRequest) Header(key, value string) *GlusterVolumeServiceResetOptionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceResetOptionRequest) Query(key, value string) *GlusterVolumeServiceResetOptionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceResetOptionRequest) Async(async bool) *GlusterVolumeServiceResetOptionRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceResetOptionRequest) Force(force bool) *GlusterVolumeServiceResetOptionRequest {
	p.force = &force
	return p
}

func (p *GlusterVolumeServiceResetOptionRequest) Option(option *Option) *GlusterVolumeServiceResetOptionRequest {
	p.option = option
	return p
}

func (p *GlusterVolumeServiceResetOptionRequest) Send() (*GlusterVolumeServiceResetOptionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/resetoption", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	actionBuilder.Option(p.option)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceResetOptionResponse), nil
}

func (p *GlusterVolumeServiceResetOptionRequest) MustSend() *GlusterVolumeServiceResetOptionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Resets a particular option in the gluster volume.
// For example, to reset a particular option `option1` in a gluster volume with identifier `123` in cluster `456`,
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/resetoption
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//  <option name="option1"/>
// </action>
// ----
//
type GlusterVolumeServiceResetOptionResponse struct {
}

//
// Resets a particular option in the gluster volume.
// For example, to reset a particular option `option1` in a gluster volume with identifier `123` in cluster `456`,
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/resetoption
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//  <option name="option1"/>
// </action>
// ----
//
func (p *GlusterVolumeService) ResetOption() *GlusterVolumeServiceResetOptionRequest {
	return &GlusterVolumeServiceResetOptionRequest{GlusterVolumeService: p}
}

//
// Sets a particular option in the gluster volume.
// For example, to set `option1` with value `value1` in a gluster volume with identifier `123` in cluster `456`,
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/setoption
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//  <option name="option1" value="value1"/>
// </action>
// ----
//
type GlusterVolumeServiceSetOptionRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
	option               *Option
}

func (p *GlusterVolumeServiceSetOptionRequest) Header(key, value string) *GlusterVolumeServiceSetOptionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceSetOptionRequest) Query(key, value string) *GlusterVolumeServiceSetOptionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceSetOptionRequest) Async(async bool) *GlusterVolumeServiceSetOptionRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceSetOptionRequest) Option(option *Option) *GlusterVolumeServiceSetOptionRequest {
	p.option = option
	return p
}

func (p *GlusterVolumeServiceSetOptionRequest) Send() (*GlusterVolumeServiceSetOptionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/setoption", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Option(p.option)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceSetOptionResponse), nil
}

func (p *GlusterVolumeServiceSetOptionRequest) MustSend() *GlusterVolumeServiceSetOptionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Sets a particular option in the gluster volume.
// For example, to set `option1` with value `value1` in a gluster volume with identifier `123` in cluster `456`,
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/setoption
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//  <option name="option1" value="value1"/>
// </action>
// ----
//
type GlusterVolumeServiceSetOptionResponse struct {
}

//
// Sets a particular option in the gluster volume.
// For example, to set `option1` with value `value1` in a gluster volume with identifier `123` in cluster `456`,
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/setoption
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//  <option name="option1" value="value1"/>
// </action>
// ----
//
func (p *GlusterVolumeService) SetOption() *GlusterVolumeServiceSetOptionRequest {
	return &GlusterVolumeServiceSetOptionRequest{GlusterVolumeService: p}
}

//
// Starts the gluster volume.
// A Gluster Volume should be started to read/write data. For example, to start a gluster volume with identifier
// `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/start
// ----
//
type GlusterVolumeServiceStartRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
	force                *bool
}

func (p *GlusterVolumeServiceStartRequest) Header(key, value string) *GlusterVolumeServiceStartRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceStartRequest) Query(key, value string) *GlusterVolumeServiceStartRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceStartRequest) Async(async bool) *GlusterVolumeServiceStartRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceStartRequest) Force(force bool) *GlusterVolumeServiceStartRequest {
	p.force = &force
	return p
}

func (p *GlusterVolumeServiceStartRequest) Send() (*GlusterVolumeServiceStartResponse, error) {
	rawURL := fmt.Sprintf("%s%s/start", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceStartResponse), nil
}

func (p *GlusterVolumeServiceStartRequest) MustSend() *GlusterVolumeServiceStartResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Starts the gluster volume.
// A Gluster Volume should be started to read/write data. For example, to start a gluster volume with identifier
// `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/start
// ----
//
type GlusterVolumeServiceStartResponse struct {
}

//
// Starts the gluster volume.
// A Gluster Volume should be started to read/write data. For example, to start a gluster volume with identifier
// `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/start
// ----
//
func (p *GlusterVolumeService) Start() *GlusterVolumeServiceStartRequest {
	return &GlusterVolumeServiceStartRequest{GlusterVolumeService: p}
}

//
// Start profiling the gluster volume.
// For example, to start profiling a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/startprofile
// ----
//
type GlusterVolumeServiceStartProfileRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
}

func (p *GlusterVolumeServiceStartProfileRequest) Header(key, value string) *GlusterVolumeServiceStartProfileRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceStartProfileRequest) Query(key, value string) *GlusterVolumeServiceStartProfileRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceStartProfileRequest) Async(async bool) *GlusterVolumeServiceStartProfileRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceStartProfileRequest) Send() (*GlusterVolumeServiceStartProfileResponse, error) {
	rawURL := fmt.Sprintf("%s%s/startprofile", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceStartProfileResponse), nil
}

func (p *GlusterVolumeServiceStartProfileRequest) MustSend() *GlusterVolumeServiceStartProfileResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Start profiling the gluster volume.
// For example, to start profiling a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/startprofile
// ----
//
type GlusterVolumeServiceStartProfileResponse struct {
}

//
// Start profiling the gluster volume.
// For example, to start profiling a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/startprofile
// ----
//
func (p *GlusterVolumeService) StartProfile() *GlusterVolumeServiceStartProfileRequest {
	return &GlusterVolumeServiceStartProfileRequest{GlusterVolumeService: p}
}

//
// Stops the gluster volume.
// Stopping a volume will make its data inaccessible.
// For example, to stop a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stop
// ----
//
type GlusterVolumeServiceStopRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
	force                *bool
}

func (p *GlusterVolumeServiceStopRequest) Header(key, value string) *GlusterVolumeServiceStopRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceStopRequest) Query(key, value string) *GlusterVolumeServiceStopRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceStopRequest) Async(async bool) *GlusterVolumeServiceStopRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceStopRequest) Force(force bool) *GlusterVolumeServiceStopRequest {
	p.force = &force
	return p
}

func (p *GlusterVolumeServiceStopRequest) Send() (*GlusterVolumeServiceStopResponse, error) {
	rawURL := fmt.Sprintf("%s%s/stop", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceStopResponse), nil
}

func (p *GlusterVolumeServiceStopRequest) MustSend() *GlusterVolumeServiceStopResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Stops the gluster volume.
// Stopping a volume will make its data inaccessible.
// For example, to stop a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stop
// ----
//
type GlusterVolumeServiceStopResponse struct {
}

//
// Stops the gluster volume.
// Stopping a volume will make its data inaccessible.
// For example, to stop a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stop
// ----
//
func (p *GlusterVolumeService) Stop() *GlusterVolumeServiceStopRequest {
	return &GlusterVolumeServiceStopRequest{GlusterVolumeService: p}
}

//
// Stop profiling the gluster volume.
// For example, to stop profiling a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stopprofile
// ----
//
type GlusterVolumeServiceStopProfileRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
}

func (p *GlusterVolumeServiceStopProfileRequest) Header(key, value string) *GlusterVolumeServiceStopProfileRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceStopProfileRequest) Query(key, value string) *GlusterVolumeServiceStopProfileRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceStopProfileRequest) Async(async bool) *GlusterVolumeServiceStopProfileRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceStopProfileRequest) Send() (*GlusterVolumeServiceStopProfileResponse, error) {
	rawURL := fmt.Sprintf("%s%s/stopprofile", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceStopProfileResponse), nil
}

func (p *GlusterVolumeServiceStopProfileRequest) MustSend() *GlusterVolumeServiceStopProfileResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Stop profiling the gluster volume.
// For example, to stop profiling a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stopprofile
// ----
//
type GlusterVolumeServiceStopProfileResponse struct {
}

//
// Stop profiling the gluster volume.
// For example, to stop profiling a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stopprofile
// ----
//
func (p *GlusterVolumeService) StopProfile() *GlusterVolumeServiceStopProfileRequest {
	return &GlusterVolumeServiceStopProfileRequest{GlusterVolumeService: p}
}

//
// Stop rebalancing the gluster volume.
// For example, to stop rebalancing a gluster volume with identifier `123` in cluster `456`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stoprebalance
// ----
//
type GlusterVolumeServiceStopRebalanceRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
}

func (p *GlusterVolumeServiceStopRebalanceRequest) Header(key, value string) *GlusterVolumeServiceStopRebalanceRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceStopRebalanceRequest) Query(key, value string) *GlusterVolumeServiceStopRebalanceRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceStopRebalanceRequest) Async(async bool) *GlusterVolumeServiceStopRebalanceRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceStopRebalanceRequest) Send() (*GlusterVolumeServiceStopRebalanceResponse, error) {
	rawURL := fmt.Sprintf("%s%s/stoprebalance", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceStopRebalanceResponse), nil
}

func (p *GlusterVolumeServiceStopRebalanceRequest) MustSend() *GlusterVolumeServiceStopRebalanceResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Stop rebalancing the gluster volume.
// For example, to stop rebalancing a gluster volume with identifier `123` in cluster `456`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stoprebalance
// ----
//
type GlusterVolumeServiceStopRebalanceResponse struct {
}

//
// Stop rebalancing the gluster volume.
// For example, to stop rebalancing a gluster volume with identifier `123` in cluster `456`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stoprebalance
// ----
//
func (p *GlusterVolumeService) StopRebalance() *GlusterVolumeServiceStopRebalanceRequest {
	return &GlusterVolumeServiceStopRebalanceRequest{GlusterVolumeService: p}
}

//
// Reference to a service managing gluster bricks.
//
func (op *GlusterVolumeService) GlusterBricksService() *GlusterBricksService {
	return NewGlusterBricksService(op.connection, fmt.Sprintf("%s/glusterbricks", op.path))
}

//
//
func (op *GlusterVolumeService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GlusterVolumeService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "glusterbricks" {
		return op.GlusterBricksService(), nil
	}
	if strings.HasPrefix(path, "glusterbricks/") {
		return op.GlusterBricksService().Service(path[14:])
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *GlusterVolumeService) String() string {
	return fmt.Sprintf("GlusterVolumeService:%s", op.path)
}

//
// This service manages the gluster bricks in a gluster volume
//
type GlusterBricksService struct {
	BaseService
}

func NewGlusterBricksService(connection *Connection, path string) *GlusterBricksService {
	var result GlusterBricksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Activate the bricks post data migration of remove brick operation.
// Used to activate brick(s) once the data migration from bricks is complete but user no longer wishes to remove
// bricks. The bricks that were previously marked for removal will now be used as normal bricks.
// For example, to retain the bricks that on glustervolume `123` from which data was migrated, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/activate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <bricks>
//     <brick>
//       <name>host1:/rhgs/brick1</name>
//     </brick>
//   </bricks>
// </action>
// ----
//
type GlusterBricksServiceActivateRequest struct {
	GlusterBricksService *GlusterBricksService
	header               map[string]string
	query                map[string]string
	async                *bool
	bricks               *GlusterBrickSlice
}

func (p *GlusterBricksServiceActivateRequest) Header(key, value string) *GlusterBricksServiceActivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBricksServiceActivateRequest) Query(key, value string) *GlusterBricksServiceActivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBricksServiceActivateRequest) Async(async bool) *GlusterBricksServiceActivateRequest {
	p.async = &async
	return p
}

func (p *GlusterBricksServiceActivateRequest) Bricks(bricks *GlusterBrickSlice) *GlusterBricksServiceActivateRequest {
	p.bricks = bricks
	return p
}

func (p *GlusterBricksServiceActivateRequest) BricksOfAny(anys ...*GlusterBrick) *GlusterBricksServiceActivateRequest {
	if p.bricks == nil {
		p.bricks = new(GlusterBrickSlice)
	}
	p.bricks.slice = append(p.bricks.slice, anys...)
	return p
}

func (p *GlusterBricksServiceActivateRequest) Send() (*GlusterBricksServiceActivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/activate", p.GlusterBricksService.connection.URL(), p.GlusterBricksService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Bricks(p.bricks)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBricksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBricksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBricksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBricksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBricksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterBricksServiceActivateResponse), nil
}

func (p *GlusterBricksServiceActivateRequest) MustSend() *GlusterBricksServiceActivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Activate the bricks post data migration of remove brick operation.
// Used to activate brick(s) once the data migration from bricks is complete but user no longer wishes to remove
// bricks. The bricks that were previously marked for removal will now be used as normal bricks.
// For example, to retain the bricks that on glustervolume `123` from which data was migrated, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/activate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <bricks>
//     <brick>
//       <name>host1:/rhgs/brick1</name>
//     </brick>
//   </bricks>
// </action>
// ----
//
type GlusterBricksServiceActivateResponse struct {
}

//
// Activate the bricks post data migration of remove brick operation.
// Used to activate brick(s) once the data migration from bricks is complete but user no longer wishes to remove
// bricks. The bricks that were previously marked for removal will now be used as normal bricks.
// For example, to retain the bricks that on glustervolume `123` from which data was migrated, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/activate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <bricks>
//     <brick>
//       <name>host1:/rhgs/brick1</name>
//     </brick>
//   </bricks>
// </action>
// ----
//
func (p *GlusterBricksService) Activate() *GlusterBricksServiceActivateRequest {
	return &GlusterBricksServiceActivateRequest{GlusterBricksService: p}
}

//
// Adds a list of bricks to gluster volume.
// Used to expand a gluster volume by adding bricks. For replicated volume types, the parameter `replica_count`
// needs to be passed. In case the replica count is being increased, then the number of bricks needs to be
// equivalent to the number of replica sets.
// For example, to add bricks to gluster volume `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <server_id>111</server_id>
//     <brick_dir>/export/data/brick3</brick_dir>
//   </brick>
// </bricks>
// ----
//
type GlusterBricksServiceAddRequest struct {
	GlusterBricksService *GlusterBricksService
	header               map[string]string
	query                map[string]string
	bricks               *GlusterBrickSlice
	replicaCount         *int64
	stripeCount          *int64
}

func (p *GlusterBricksServiceAddRequest) Header(key, value string) *GlusterBricksServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBricksServiceAddRequest) Query(key, value string) *GlusterBricksServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBricksServiceAddRequest) Bricks(bricks *GlusterBrickSlice) *GlusterBricksServiceAddRequest {
	p.bricks = bricks
	return p
}

func (p *GlusterBricksServiceAddRequest) BricksOfAny(anys ...*GlusterBrick) *GlusterBricksServiceAddRequest {
	if p.bricks == nil {
		p.bricks = new(GlusterBrickSlice)
	}
	p.bricks.slice = append(p.bricks.slice, anys...)
	return p
}

func (p *GlusterBricksServiceAddRequest) ReplicaCount(replicaCount int64) *GlusterBricksServiceAddRequest {
	p.replicaCount = &replicaCount
	return p
}

func (p *GlusterBricksServiceAddRequest) StripeCount(stripeCount int64) *GlusterBricksServiceAddRequest {
	p.stripeCount = &stripeCount
	return p
}

func (p *GlusterBricksServiceAddRequest) Send() (*GlusterBricksServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterBricksService.connection.URL(), p.GlusterBricksService.path)
	values := make(url.Values)
	if p.replicaCount != nil {
		values["replica_count"] = []string{fmt.Sprintf("%v", *p.replicaCount)}
	}

	if p.stripeCount != nil {
		values["stripe_count"] = []string{fmt.Sprintf("%v", *p.stripeCount)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLGlusterBrickWriteMany(writer, p.bricks, "", "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBricksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBricksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBricksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBricksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBricksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterBrickReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &GlusterBricksServiceAddResponse{bricks: result}, nil
}

func (p *GlusterBricksServiceAddRequest) MustSend() *GlusterBricksServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a list of bricks to gluster volume.
// Used to expand a gluster volume by adding bricks. For replicated volume types, the parameter `replica_count`
// needs to be passed. In case the replica count is being increased, then the number of bricks needs to be
// equivalent to the number of replica sets.
// For example, to add bricks to gluster volume `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <server_id>111</server_id>
//     <brick_dir>/export/data/brick3</brick_dir>
//   </brick>
// </bricks>
// ----
//
type GlusterBricksServiceAddResponse struct {
	bricks *GlusterBrickSlice
}

func (p *GlusterBricksServiceAddResponse) Bricks() (*GlusterBrickSlice, bool) {
	if p.bricks != nil {
		return p.bricks, true
	}
	return nil, false
}

func (p *GlusterBricksServiceAddResponse) MustBricks() *GlusterBrickSlice {
	if p.bricks == nil {
		panic("bricks in response does not exist")
	}
	return p.bricks
}

//
// Adds a list of bricks to gluster volume.
// Used to expand a gluster volume by adding bricks. For replicated volume types, the parameter `replica_count`
// needs to be passed. In case the replica count is being increased, then the number of bricks needs to be
// equivalent to the number of replica sets.
// For example, to add bricks to gluster volume `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <server_id>111</server_id>
//     <brick_dir>/export/data/brick3</brick_dir>
//   </brick>
// </bricks>
// ----
//
func (p *GlusterBricksService) Add() *GlusterBricksServiceAddRequest {
	return &GlusterBricksServiceAddRequest{GlusterBricksService: p}
}

//
// Lists the bricks of a gluster volume.
// For example, to list bricks of gluster volume `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// Provides an output as below:
// [source,xml]
// ----
// <bricks>
//   <brick id="234">
//     <name>host1:/rhgs/data/brick1</name>
//     <brick_dir>/rhgs/data/brick1</brick_dir>
//     <server_id>111</server_id>
//     <status>up</status>
//   </brick>
//   <brick id="233">
//     <name>host2:/rhgs/data/brick1</name>
//     <brick_dir>/rhgs/data/brick1</brick_dir>
//     <server_id>222</server_id>
//     <status>up</status>
//   </brick>
// </bricks>
// ----
// The order of the returned list is based on the brick order provided at gluster volume creation.
//
type GlusterBricksServiceListRequest struct {
	GlusterBricksService *GlusterBricksService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *GlusterBricksServiceListRequest) Header(key, value string) *GlusterBricksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBricksServiceListRequest) Query(key, value string) *GlusterBricksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBricksServiceListRequest) Follow(follow string) *GlusterBricksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *GlusterBricksServiceListRequest) Max(max int64) *GlusterBricksServiceListRequest {
	p.max = &max
	return p
}

func (p *GlusterBricksServiceListRequest) Send() (*GlusterBricksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterBricksService.connection.URL(), p.GlusterBricksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBricksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBricksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBricksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBricksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBricksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterBrickReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &GlusterBricksServiceListResponse{bricks: result}, nil
}

func (p *GlusterBricksServiceListRequest) MustSend() *GlusterBricksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists the bricks of a gluster volume.
// For example, to list bricks of gluster volume `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// Provides an output as below:
// [source,xml]
// ----
// <bricks>
//   <brick id="234">
//     <name>host1:/rhgs/data/brick1</name>
//     <brick_dir>/rhgs/data/brick1</brick_dir>
//     <server_id>111</server_id>
//     <status>up</status>
//   </brick>
//   <brick id="233">
//     <name>host2:/rhgs/data/brick1</name>
//     <brick_dir>/rhgs/data/brick1</brick_dir>
//     <server_id>222</server_id>
//     <status>up</status>
//   </brick>
// </bricks>
// ----
// The order of the returned list is based on the brick order provided at gluster volume creation.
//
type GlusterBricksServiceListResponse struct {
	bricks *GlusterBrickSlice
}

func (p *GlusterBricksServiceListResponse) Bricks() (*GlusterBrickSlice, bool) {
	if p.bricks != nil {
		return p.bricks, true
	}
	return nil, false
}

func (p *GlusterBricksServiceListResponse) MustBricks() *GlusterBrickSlice {
	if p.bricks == nil {
		panic("bricks in response does not exist")
	}
	return p.bricks
}

//
// Lists the bricks of a gluster volume.
// For example, to list bricks of gluster volume `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// Provides an output as below:
// [source,xml]
// ----
// <bricks>
//   <brick id="234">
//     <name>host1:/rhgs/data/brick1</name>
//     <brick_dir>/rhgs/data/brick1</brick_dir>
//     <server_id>111</server_id>
//     <status>up</status>
//   </brick>
//   <brick id="233">
//     <name>host2:/rhgs/data/brick1</name>
//     <brick_dir>/rhgs/data/brick1</brick_dir>
//     <server_id>222</server_id>
//     <status>up</status>
//   </brick>
// </bricks>
// ----
// The order of the returned list is based on the brick order provided at gluster volume creation.
//
func (p *GlusterBricksService) List() *GlusterBricksServiceListRequest {
	return &GlusterBricksServiceListRequest{GlusterBricksService: p}
}

//
// Start migration of data prior to removing bricks.
// Removing bricks is a two-step process, where the data on bricks to be removed, is first migrated to remaining
// bricks. Once migration is completed the removal of bricks is confirmed via the API
// <<services/gluster_bricks/methods/remove, remove>>. If at any point, the action needs to be cancelled
// <<services/gluster_bricks/methods/stop_migrate, stopmigrate>> has to be called.
// For instance, to delete a brick from a gluster volume with id `123`, send a request:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/migrate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <bricks>
//     <brick>
//       <name>host1:/rhgs/brick1</name>
//     </brick>
//   </bricks>
// </action>
// ----
// The migration process can be tracked from the job id returned from the API using
// <<services/job/methods/get, job>> and steps in job using <<services/step/methods/get, step>>
//
type GlusterBricksServiceMigrateRequest struct {
	GlusterBricksService *GlusterBricksService
	header               map[string]string
	query                map[string]string
	async                *bool
	bricks               *GlusterBrickSlice
}

func (p *GlusterBricksServiceMigrateRequest) Header(key, value string) *GlusterBricksServiceMigrateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBricksServiceMigrateRequest) Query(key, value string) *GlusterBricksServiceMigrateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBricksServiceMigrateRequest) Async(async bool) *GlusterBricksServiceMigrateRequest {
	p.async = &async
	return p
}

func (p *GlusterBricksServiceMigrateRequest) Bricks(bricks *GlusterBrickSlice) *GlusterBricksServiceMigrateRequest {
	p.bricks = bricks
	return p
}

func (p *GlusterBricksServiceMigrateRequest) BricksOfAny(anys ...*GlusterBrick) *GlusterBricksServiceMigrateRequest {
	if p.bricks == nil {
		p.bricks = new(GlusterBrickSlice)
	}
	p.bricks.slice = append(p.bricks.slice, anys...)
	return p
}

func (p *GlusterBricksServiceMigrateRequest) Send() (*GlusterBricksServiceMigrateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/migrate", p.GlusterBricksService.connection.URL(), p.GlusterBricksService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Bricks(p.bricks)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBricksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBricksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBricksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBricksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBricksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterBricksServiceMigrateResponse), nil
}

func (p *GlusterBricksServiceMigrateRequest) MustSend() *GlusterBricksServiceMigrateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Start migration of data prior to removing bricks.
// Removing bricks is a two-step process, where the data on bricks to be removed, is first migrated to remaining
// bricks. Once migration is completed the removal of bricks is confirmed via the API
// <<services/gluster_bricks/methods/remove, remove>>. If at any point, the action needs to be cancelled
// <<services/gluster_bricks/methods/stop_migrate, stopmigrate>> has to be called.
// For instance, to delete a brick from a gluster volume with id `123`, send a request:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/migrate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <bricks>
//     <brick>
//       <name>host1:/rhgs/brick1</name>
//     </brick>
//   </bricks>
// </action>
// ----
// The migration process can be tracked from the job id returned from the API using
// <<services/job/methods/get, job>> and steps in job using <<services/step/methods/get, step>>
//
type GlusterBricksServiceMigrateResponse struct {
}

//
// Start migration of data prior to removing bricks.
// Removing bricks is a two-step process, where the data on bricks to be removed, is first migrated to remaining
// bricks. Once migration is completed the removal of bricks is confirmed via the API
// <<services/gluster_bricks/methods/remove, remove>>. If at any point, the action needs to be cancelled
// <<services/gluster_bricks/methods/stop_migrate, stopmigrate>> has to be called.
// For instance, to delete a brick from a gluster volume with id `123`, send a request:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/migrate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <bricks>
//     <brick>
//       <name>host1:/rhgs/brick1</name>
//     </brick>
//   </bricks>
// </action>
// ----
// The migration process can be tracked from the job id returned from the API using
// <<services/job/methods/get, job>> and steps in job using <<services/step/methods/get, step>>
//
func (p *GlusterBricksService) Migrate() *GlusterBricksServiceMigrateRequest {
	return &GlusterBricksServiceMigrateRequest{GlusterBricksService: p}
}

//
// Removes bricks from gluster volume.
// The recommended way to remove bricks without data loss is to first migrate the data using
// <<services/gluster_bricks/methods/stop_migrate, stopmigrate>> and then removing them. If migrate was not called on
// bricks prior to remove, the bricks are removed without data migration which may lead to data loss.
// For example, to delete the bricks from gluster volume `123`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <name>host:brick_directory</name>
//   </brick>
// </bricks>
// ----
//
type GlusterBricksServiceRemoveRequest struct {
	GlusterBricksService *GlusterBricksService
	header               map[string]string
	query                map[string]string
	async                *bool
	bricks               *GlusterBrickSlice
	replicaCount         *int64
}

func (p *GlusterBricksServiceRemoveRequest) Header(key, value string) *GlusterBricksServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBricksServiceRemoveRequest) Query(key, value string) *GlusterBricksServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBricksServiceRemoveRequest) Async(async bool) *GlusterBricksServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *GlusterBricksServiceRemoveRequest) Bricks(bricks *GlusterBrickSlice) *GlusterBricksServiceRemoveRequest {
	p.bricks = bricks
	return p
}

func (p *GlusterBricksServiceRemoveRequest) BricksOfAny(anys ...*GlusterBrick) *GlusterBricksServiceRemoveRequest {
	if p.bricks == nil {
		p.bricks = new(GlusterBrickSlice)
	}
	p.bricks.slice = append(p.bricks.slice, anys...)
	return p
}

func (p *GlusterBricksServiceRemoveRequest) ReplicaCount(replicaCount int64) *GlusterBricksServiceRemoveRequest {
	p.replicaCount = &replicaCount
	return p
}

func (p *GlusterBricksServiceRemoveRequest) Send() (*GlusterBricksServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterBricksService.connection.URL(), p.GlusterBricksService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.replicaCount != nil {
		values["replica_count"] = []string{fmt.Sprintf("%v", *p.replicaCount)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBricksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBricksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBricksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBricksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBricksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(GlusterBricksServiceRemoveResponse), nil
}

func (p *GlusterBricksServiceRemoveRequest) MustSend() *GlusterBricksServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes bricks from gluster volume.
// The recommended way to remove bricks without data loss is to first migrate the data using
// <<services/gluster_bricks/methods/stop_migrate, stopmigrate>> and then removing them. If migrate was not called on
// bricks prior to remove, the bricks are removed without data migration which may lead to data loss.
// For example, to delete the bricks from gluster volume `123`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <name>host:brick_directory</name>
//   </brick>
// </bricks>
// ----
//
type GlusterBricksServiceRemoveResponse struct {
}

//
// Removes bricks from gluster volume.
// The recommended way to remove bricks without data loss is to first migrate the data using
// <<services/gluster_bricks/methods/stop_migrate, stopmigrate>> and then removing them. If migrate was not called on
// bricks prior to remove, the bricks are removed without data migration which may lead to data loss.
// For example, to delete the bricks from gluster volume `123`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <name>host:brick_directory</name>
//   </brick>
// </bricks>
// ----
//
func (p *GlusterBricksService) Remove() *GlusterBricksServiceRemoveRequest {
	return &GlusterBricksServiceRemoveRequest{GlusterBricksService: p}
}

//
// Stops migration of data from bricks for a remove brick operation.
// To cancel data migration that was started as part of the 2-step remove brick process in case the user wishes to
// continue using the bricks. The bricks that were marked for removal will function as normal bricks post this
// operation.
// For example, to stop migration of data from the bricks of gluster volume `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/stopmigrate
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <name>host:brick_directory</name>
//   </brick>
// </bricks>
// ----
//
type GlusterBricksServiceStopMigrateRequest struct {
	GlusterBricksService *GlusterBricksService
	header               map[string]string
	query                map[string]string
	async                *bool
	bricks               *GlusterBrickSlice
}

func (p *GlusterBricksServiceStopMigrateRequest) Header(key, value string) *GlusterBricksServiceStopMigrateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBricksServiceStopMigrateRequest) Query(key, value string) *GlusterBricksServiceStopMigrateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBricksServiceStopMigrateRequest) Async(async bool) *GlusterBricksServiceStopMigrateRequest {
	p.async = &async
	return p
}

func (p *GlusterBricksServiceStopMigrateRequest) Bricks(bricks *GlusterBrickSlice) *GlusterBricksServiceStopMigrateRequest {
	p.bricks = bricks
	return p
}

func (p *GlusterBricksServiceStopMigrateRequest) BricksOfAny(anys ...*GlusterBrick) *GlusterBricksServiceStopMigrateRequest {
	if p.bricks == nil {
		p.bricks = new(GlusterBrickSlice)
	}
	p.bricks.slice = append(p.bricks.slice, anys...)
	return p
}

func (p *GlusterBricksServiceStopMigrateRequest) Send() (*GlusterBricksServiceStopMigrateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/stopmigrate", p.GlusterBricksService.connection.URL(), p.GlusterBricksService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Bricks(p.bricks)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBricksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBricksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBricksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBricksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBricksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterBricksServiceStopMigrateResponse), nil
}

func (p *GlusterBricksServiceStopMigrateRequest) MustSend() *GlusterBricksServiceStopMigrateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Stops migration of data from bricks for a remove brick operation.
// To cancel data migration that was started as part of the 2-step remove brick process in case the user wishes to
// continue using the bricks. The bricks that were marked for removal will function as normal bricks post this
// operation.
// For example, to stop migration of data from the bricks of gluster volume `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/stopmigrate
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <name>host:brick_directory</name>
//   </brick>
// </bricks>
// ----
//
type GlusterBricksServiceStopMigrateResponse struct {
}

//
// Stops migration of data from bricks for a remove brick operation.
// To cancel data migration that was started as part of the 2-step remove brick process in case the user wishes to
// continue using the bricks. The bricks that were marked for removal will function as normal bricks post this
// operation.
// For example, to stop migration of data from the bricks of gluster volume `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/stopmigrate
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <name>host:brick_directory</name>
//   </brick>
// </bricks>
// ----
//
func (p *GlusterBricksService) StopMigrate() *GlusterBricksServiceStopMigrateRequest {
	return &GlusterBricksServiceStopMigrateRequest{GlusterBricksService: p}
}

//
// Returns a reference to the service managing a single gluster brick.
//
func (op *GlusterBricksService) BrickService(id string) *GlusterBrickService {
	return NewGlusterBrickService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GlusterBricksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.BrickService(path), nil
	}
	return op.BrickService(path[:index]).Service(path[index+1:])
}

func (op *GlusterBricksService) String() string {
	return fmt.Sprintf("GlusterBricksService:%s", op.path)
}

//
// Manages the set of snapshots of a storage domain or virtual machine.
//
type SnapshotsService struct {
	BaseService
}

func NewSnapshotsService(connection *Connection, path string) *SnapshotsService {
	var result SnapshotsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a virtual machine snapshot.
// For example, to create a new snapshot for virtual machine `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/vms/123/snapshots
// ----
// With a request body like this:
// [source,xml]
// ----
// <snapshot>
//   <description>My snapshot</description>
// </snapshot>
// ----
// For including only a sub-set of disks in the snapshots, add `disk_attachments` element to the
// request body. Note that disks which are not specified in `disk_attachments` element will not be a
// part of the snapshot. If an empty `disk_attachments` element is passed, the snapshot will include
// only the virtual machine configuration. If no `disk_attachments` element is passed, then all
// the disks will be included in the snapshot.
// For each disk, `image_id` element can be specified for setting the new active image id.
// This is used in order to restore a chain of images from backup. I.e. when restoring
// a disk with snapshots, the relevant `image_id` should be specified for each snapshot
// (so the identifiers of the disk snapshots are identical to the backup).
// [source,xml]
// ----
// <snapshot>
//   <description>My snapshot</description>
//   <disk_attachments>
//     <disk_attachment>
//       <disk id="123">
//         <image_id>456</image_id>
//       </disk>
//     </disk_attachment>
//   </disk_attachments>
// </snapshot>
// ----
// [IMPORTANT]
// ====
// When a snapshot is created the default value for the <<types/snapshot/attributes/persist_memorystate,
// persist_memorystate>> attribute is `true`. That means that the content of the memory of the virtual
// machine will be included in the snapshot, and it also means that the virtual machine will be paused
// for a longer time. That can negatively affect applications that are very sensitive to timing (NTP
// servers, for example). In those cases make sure that you set the attribute to `false`:
// [source,xml]
// ----
// <snapshot>
//   <description>My snapshot</description>
//   <persist_memorystate>false</persist_memorystate>
// </snapshot>
// ----
// ====
//
type SnapshotsServiceAddRequest struct {
	SnapshotsService *SnapshotsService
	header           map[string]string
	query            map[string]string
	snapshot         *Snapshot
}

func (p *SnapshotsServiceAddRequest) Header(key, value string) *SnapshotsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SnapshotsServiceAddRequest) Query(key, value string) *SnapshotsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SnapshotsServiceAddRequest) Snapshot(snapshot *Snapshot) *SnapshotsServiceAddRequest {
	p.snapshot = snapshot
	return p
}

func (p *SnapshotsServiceAddRequest) Send() (*SnapshotsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SnapshotsService.connection.URL(), p.SnapshotsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLSnapshotWriteOne(writer, p.snapshot, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SnapshotsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SnapshotsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SnapshotsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SnapshotsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SnapshotsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSnapshotReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SnapshotsServiceAddResponse{snapshot: result}, nil
}

func (p *SnapshotsServiceAddRequest) MustSend() *SnapshotsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a virtual machine snapshot.
// For example, to create a new snapshot for virtual machine `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/vms/123/snapshots
// ----
// With a request body like this:
// [source,xml]
// ----
// <snapshot>
//   <description>My snapshot</description>
// </snapshot>
// ----
// For including only a sub-set of disks in the snapshots, add `disk_attachments` element to the
// request body. Note that disks which are not specified in `disk_attachments` element will not be a
// part of the snapshot. If an empty `disk_attachments` element is passed, the snapshot will include
// only the virtual machine configuration. If no `disk_attachments` element is passed, then all
// the disks will be included in the snapshot.
// For each disk, `image_id` element can be specified for setting the new active image id.
// This is used in order to restore a chain of images from backup. I.e. when restoring
// a disk with snapshots, the relevant `image_id` should be specified for each snapshot
// (so the identifiers of the disk snapshots are identical to the backup).
// [source,xml]
// ----
// <snapshot>
//   <description>My snapshot</description>
//   <disk_attachments>
//     <disk_attachment>
//       <disk id="123">
//         <image_id>456</image_id>
//       </disk>
//     </disk_attachment>
//   </disk_attachments>
// </snapshot>
// ----
// [IMPORTANT]
// ====
// When a snapshot is created the default value for the <<types/snapshot/attributes/persist_memorystate,
// persist_memorystate>> attribute is `true`. That means that the content of the memory of the virtual
// machine will be included in the snapshot, and it also means that the virtual machine will be paused
// for a longer time. That can negatively affect applications that are very sensitive to timing (NTP
// servers, for example). In those cases make sure that you set the attribute to `false`:
// [source,xml]
// ----
// <snapshot>
//   <description>My snapshot</description>
//   <persist_memorystate>false</persist_memorystate>
// </snapshot>
// ----
// ====
//
type SnapshotsServiceAddResponse struct {
	snapshot *Snapshot
}

func (p *SnapshotsServiceAddResponse) Snapshot() (*Snapshot, bool) {
	if p.snapshot != nil {
		return p.snapshot, true
	}
	return nil, false
}

func (p *SnapshotsServiceAddResponse) MustSnapshot() *Snapshot {
	if p.snapshot == nil {
		panic("snapshot in response does not exist")
	}
	return p.snapshot
}

//
// Creates a virtual machine snapshot.
// For example, to create a new snapshot for virtual machine `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/vms/123/snapshots
// ----
// With a request body like this:
// [source,xml]
// ----
// <snapshot>
//   <description>My snapshot</description>
// </snapshot>
// ----
// For including only a sub-set of disks in the snapshots, add `disk_attachments` element to the
// request body. Note that disks which are not specified in `disk_attachments` element will not be a
// part of the snapshot. If an empty `disk_attachments` element is passed, the snapshot will include
// only the virtual machine configuration. If no `disk_attachments` element is passed, then all
// the disks will be included in the snapshot.
// For each disk, `image_id` element can be specified for setting the new active image id.
// This is used in order to restore a chain of images from backup. I.e. when restoring
// a disk with snapshots, the relevant `image_id` should be specified for each snapshot
// (so the identifiers of the disk snapshots are identical to the backup).
// [source,xml]
// ----
// <snapshot>
//   <description>My snapshot</description>
//   <disk_attachments>
//     <disk_attachment>
//       <disk id="123">
//         <image_id>456</image_id>
//       </disk>
//     </disk_attachment>
//   </disk_attachments>
// </snapshot>
// ----
// [IMPORTANT]
// ====
// When a snapshot is created the default value for the <<types/snapshot/attributes/persist_memorystate,
// persist_memorystate>> attribute is `true`. That means that the content of the memory of the virtual
// machine will be included in the snapshot, and it also means that the virtual machine will be paused
// for a longer time. That can negatively affect applications that are very sensitive to timing (NTP
// servers, for example). In those cases make sure that you set the attribute to `false`:
// [source,xml]
// ----
// <snapshot>
//   <description>My snapshot</description>
//   <persist_memorystate>false</persist_memorystate>
// </snapshot>
// ----
// ====
//
func (p *SnapshotsService) Add() *SnapshotsServiceAddRequest {
	return &SnapshotsServiceAddRequest{SnapshotsService: p}
}

//
// Returns the list of snapshots of the storage domain or virtual machine.
// The order of the returned list of snapshots isn't guaranteed.
//
type SnapshotsServiceListRequest struct {
	SnapshotsService *SnapshotsService
	header           map[string]string
	query            map[string]string
	allContent       *bool
	follow           *string
	max              *int64
}

func (p *SnapshotsServiceListRequest) Header(key, value string) *SnapshotsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SnapshotsServiceListRequest) Query(key, value string) *SnapshotsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SnapshotsServiceListRequest) AllContent(allContent bool) *SnapshotsServiceListRequest {
	p.allContent = &allContent
	return p
}

func (p *SnapshotsServiceListRequest) Follow(follow string) *SnapshotsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *SnapshotsServiceListRequest) Max(max int64) *SnapshotsServiceListRequest {
	p.max = &max
	return p
}

func (p *SnapshotsServiceListRequest) Send() (*SnapshotsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SnapshotsService.connection.URL(), p.SnapshotsService.path)
	values := make(url.Values)
	if p.allContent != nil {
		values["all_content"] = []string{fmt.Sprintf("%v", *p.allContent)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SnapshotsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SnapshotsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SnapshotsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SnapshotsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SnapshotsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSnapshotReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &SnapshotsServiceListResponse{snapshots: result}, nil
}

func (p *SnapshotsServiceListRequest) MustSend() *SnapshotsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of snapshots of the storage domain or virtual machine.
// The order of the returned list of snapshots isn't guaranteed.
//
type SnapshotsServiceListResponse struct {
	snapshots *SnapshotSlice
}

func (p *SnapshotsServiceListResponse) Snapshots() (*SnapshotSlice, bool) {
	if p.snapshots != nil {
		return p.snapshots, true
	}
	return nil, false
}

func (p *SnapshotsServiceListResponse) MustSnapshots() *SnapshotSlice {
	if p.snapshots == nil {
		panic("snapshots in response does not exist")
	}
	return p.snapshots
}

//
// Returns the list of snapshots of the storage domain or virtual machine.
// The order of the returned list of snapshots isn't guaranteed.
//
func (p *SnapshotsService) List() *SnapshotsServiceListRequest {
	return &SnapshotsServiceListRequest{SnapshotsService: p}
}

//
//
func (op *SnapshotsService) SnapshotService(id string) *SnapshotService {
	return NewSnapshotService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SnapshotsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.SnapshotService(path), nil
	}
	return op.SnapshotService(path[:index]).Service(path[index+1:])
}

func (op *SnapshotsService) String() string {
	return fmt.Sprintf("SnapshotsService:%s", op.path)
}

//
// This service manages hostgroups.
//
type ExternalHostGroupsService struct {
	BaseService
}

func NewExternalHostGroupsService(connection *Connection, path string) *ExternalHostGroupsService {
	var result ExternalHostGroupsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get host groups list from external host provider.
// Host group is a term of host providers - the host group includes provision details. This API returns all possible
// hostgroups exposed by the external provider.
// For example, to get the details of all host groups of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/hostgroups
// ....
// The response will be like this:
// [source,xml]
// ----
// <external_host_groups>
//   <external_host_group href="/ovirt-engine/api/externalhostproviders/123/hostgroups/234" id="234">
//     <name>rhel7</name>
//     <architecture_name>x86_64</architecture_name>
//     <domain_name>example.com</domain_name>
//     <operating_system_name>RedHat 7.3</operating_system_name>
//     <subnet_name>sat0</subnet_name>
//     <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//   </external_host_group>
//   ...
// </external_host_groups>
// ----
// The order of the returned list of host groups isn't guaranteed.
//
type ExternalHostGroupsServiceListRequest struct {
	ExternalHostGroupsService *ExternalHostGroupsService
	header                    map[string]string
	query                     map[string]string
	follow                    *string
	max                       *int64
}

func (p *ExternalHostGroupsServiceListRequest) Header(key, value string) *ExternalHostGroupsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostGroupsServiceListRequest) Query(key, value string) *ExternalHostGroupsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostGroupsServiceListRequest) Follow(follow string) *ExternalHostGroupsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ExternalHostGroupsServiceListRequest) Max(max int64) *ExternalHostGroupsServiceListRequest {
	p.max = &max
	return p
}

func (p *ExternalHostGroupsServiceListRequest) Send() (*ExternalHostGroupsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostGroupsService.connection.URL(), p.ExternalHostGroupsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostGroupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostGroupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostGroupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostGroupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostGroupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostGroupReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ExternalHostGroupsServiceListResponse{groups: result}, nil
}

func (p *ExternalHostGroupsServiceListRequest) MustSend() *ExternalHostGroupsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get host groups list from external host provider.
// Host group is a term of host providers - the host group includes provision details. This API returns all possible
// hostgroups exposed by the external provider.
// For example, to get the details of all host groups of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/hostgroups
// ....
// The response will be like this:
// [source,xml]
// ----
// <external_host_groups>
//   <external_host_group href="/ovirt-engine/api/externalhostproviders/123/hostgroups/234" id="234">
//     <name>rhel7</name>
//     <architecture_name>x86_64</architecture_name>
//     <domain_name>example.com</domain_name>
//     <operating_system_name>RedHat 7.3</operating_system_name>
//     <subnet_name>sat0</subnet_name>
//     <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//   </external_host_group>
//   ...
// </external_host_groups>
// ----
// The order of the returned list of host groups isn't guaranteed.
//
type ExternalHostGroupsServiceListResponse struct {
	groups *ExternalHostGroupSlice
}

func (p *ExternalHostGroupsServiceListResponse) Groups() (*ExternalHostGroupSlice, bool) {
	if p.groups != nil {
		return p.groups, true
	}
	return nil, false
}

func (p *ExternalHostGroupsServiceListResponse) MustGroups() *ExternalHostGroupSlice {
	if p.groups == nil {
		panic("groups in response does not exist")
	}
	return p.groups
}

//
// Get host groups list from external host provider.
// Host group is a term of host providers - the host group includes provision details. This API returns all possible
// hostgroups exposed by the external provider.
// For example, to get the details of all host groups of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/hostgroups
// ....
// The response will be like this:
// [source,xml]
// ----
// <external_host_groups>
//   <external_host_group href="/ovirt-engine/api/externalhostproviders/123/hostgroups/234" id="234">
//     <name>rhel7</name>
//     <architecture_name>x86_64</architecture_name>
//     <domain_name>example.com</domain_name>
//     <operating_system_name>RedHat 7.3</operating_system_name>
//     <subnet_name>sat0</subnet_name>
//     <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//   </external_host_group>
//   ...
// </external_host_groups>
// ----
// The order of the returned list of host groups isn't guaranteed.
//
func (p *ExternalHostGroupsService) List() *ExternalHostGroupsServiceListRequest {
	return &ExternalHostGroupsServiceListRequest{ExternalHostGroupsService: p}
}

//
// This service manages hostgroup instance.
//
func (op *ExternalHostGroupsService) GroupService(id string) *ExternalHostGroupService {
	return NewExternalHostGroupService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalHostGroupsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.GroupService(path), nil
	}
	return op.GroupService(path[:index]).Service(path[index+1:])
}

func (op *ExternalHostGroupsService) String() string {
	return fmt.Sprintf("ExternalHostGroupsService:%s", op.path)
}

//
// Manages a single external compute resource.
// Compute resource is a term of host external provider. The external provider also needs to know to where the
// provisioned host needs to register. The login details of the engine are saved as a compute resource  in the external
// provider side.
//
type ExternalComputeResourceService struct {
	BaseService
}

func NewExternalComputeResourceService(connection *Connection, path string) *ExternalComputeResourceService {
	var result ExternalComputeResourceService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves external compute resource details.
// For example, to get the details of compute resource `234` of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/computeresources/234
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_compute_resource href="/ovirt-engine/api/externalhostproviders/123/computeresources/234" id="234">
//   <name>hostname</name>
//   <provider>oVirt</provider>
//   <url>https://hostname/api</url>
//   <user>admin@internal</user>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_compute_resource>
// ----
//
type ExternalComputeResourceServiceGetRequest struct {
	ExternalComputeResourceService *ExternalComputeResourceService
	header                         map[string]string
	query                          map[string]string
	follow                         *string
}

func (p *ExternalComputeResourceServiceGetRequest) Header(key, value string) *ExternalComputeResourceServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalComputeResourceServiceGetRequest) Query(key, value string) *ExternalComputeResourceServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalComputeResourceServiceGetRequest) Follow(follow string) *ExternalComputeResourceServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalComputeResourceServiceGetRequest) Send() (*ExternalComputeResourceServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalComputeResourceService.connection.URL(), p.ExternalComputeResourceService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalComputeResourceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalComputeResourceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalComputeResourceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalComputeResourceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalComputeResourceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalComputeResourceReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalComputeResourceServiceGetResponse{resource: result}, nil
}

func (p *ExternalComputeResourceServiceGetRequest) MustSend() *ExternalComputeResourceServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves external compute resource details.
// For example, to get the details of compute resource `234` of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/computeresources/234
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_compute_resource href="/ovirt-engine/api/externalhostproviders/123/computeresources/234" id="234">
//   <name>hostname</name>
//   <provider>oVirt</provider>
//   <url>https://hostname/api</url>
//   <user>admin@internal</user>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_compute_resource>
// ----
//
type ExternalComputeResourceServiceGetResponse struct {
	resource *ExternalComputeResource
}

func (p *ExternalComputeResourceServiceGetResponse) Resource() (*ExternalComputeResource, bool) {
	if p.resource != nil {
		return p.resource, true
	}
	return nil, false
}

func (p *ExternalComputeResourceServiceGetResponse) MustResource() *ExternalComputeResource {
	if p.resource == nil {
		panic("resource in response does not exist")
	}
	return p.resource
}

//
// Retrieves external compute resource details.
// For example, to get the details of compute resource `234` of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/computeresources/234
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_compute_resource href="/ovirt-engine/api/externalhostproviders/123/computeresources/234" id="234">
//   <name>hostname</name>
//   <provider>oVirt</provider>
//   <url>https://hostname/api</url>
//   <user>admin@internal</user>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_compute_resource>
// ----
//
func (p *ExternalComputeResourceService) Get() *ExternalComputeResourceServiceGetRequest {
	return &ExternalComputeResourceServiceGetRequest{ExternalComputeResourceService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalComputeResourceService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalComputeResourceService) String() string {
	return fmt.Sprintf("ExternalComputeResourceService:%s", op.path)
}

//
//
type ExternalHostsService struct {
	BaseService
}

func NewExternalHostsService(connection *Connection, path string) *ExternalHostsService {
	var result ExternalHostsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Return the list of external hosts.
// The order of the returned list of hosts isn't guaranteed.
//
type ExternalHostsServiceListRequest struct {
	ExternalHostsService *ExternalHostsService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *ExternalHostsServiceListRequest) Header(key, value string) *ExternalHostsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostsServiceListRequest) Query(key, value string) *ExternalHostsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostsServiceListRequest) Follow(follow string) *ExternalHostsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ExternalHostsServiceListRequest) Max(max int64) *ExternalHostsServiceListRequest {
	p.max = &max
	return p
}

func (p *ExternalHostsServiceListRequest) Send() (*ExternalHostsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostsService.connection.URL(), p.ExternalHostsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ExternalHostsServiceListResponse{hosts: result}, nil
}

func (p *ExternalHostsServiceListRequest) MustSend() *ExternalHostsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Return the list of external hosts.
// The order of the returned list of hosts isn't guaranteed.
//
type ExternalHostsServiceListResponse struct {
	hosts *ExternalHostSlice
}

func (p *ExternalHostsServiceListResponse) Hosts() (*ExternalHostSlice, bool) {
	if p.hosts != nil {
		return p.hosts, true
	}
	return nil, false
}

func (p *ExternalHostsServiceListResponse) MustHosts() *ExternalHostSlice {
	if p.hosts == nil {
		panic("hosts in response does not exist")
	}
	return p.hosts
}

//
// Return the list of external hosts.
// The order of the returned list of hosts isn't guaranteed.
//
func (p *ExternalHostsService) List() *ExternalHostsServiceListRequest {
	return &ExternalHostsServiceListRequest{ExternalHostsService: p}
}

//
//
func (op *ExternalHostsService) HostService(id string) *ExternalHostService {
	return NewExternalHostService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalHostsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.HostService(path), nil
	}
	return op.HostService(path[:index]).Service(path[index+1:])
}

func (op *ExternalHostsService) String() string {
	return fmt.Sprintf("ExternalHostsService:%s", op.path)
}

//
//
type ExternalHostService struct {
	BaseService
}

func NewExternalHostService(connection *Connection, path string) *ExternalHostService {
	var result ExternalHostService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type ExternalHostServiceGetRequest struct {
	ExternalHostService *ExternalHostService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *ExternalHostServiceGetRequest) Header(key, value string) *ExternalHostServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostServiceGetRequest) Query(key, value string) *ExternalHostServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostServiceGetRequest) Follow(follow string) *ExternalHostServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalHostServiceGetRequest) Send() (*ExternalHostServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostService.connection.URL(), p.ExternalHostService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalHostServiceGetResponse{host: result}, nil
}

func (p *ExternalHostServiceGetRequest) MustSend() *ExternalHostServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type ExternalHostServiceGetResponse struct {
	host *ExternalHost
}

func (p *ExternalHostServiceGetResponse) Host() (*ExternalHost, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

func (p *ExternalHostServiceGetResponse) MustHost() *ExternalHost {
	if p.host == nil {
		panic("host in response does not exist")
	}
	return p.host
}

//
//
func (p *ExternalHostService) Get() *ExternalHostServiceGetRequest {
	return &ExternalHostServiceGetRequest{ExternalHostService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalHostService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalHostService) String() string {
	return fmt.Sprintf("ExternalHostService:%s", op.path)
}

//
// Manages a collection of external compute resources.
// Compute resource is a term of host external provider. The external provider also needs to know to where the
// provisioned host needs to register. The login details of the engine is saved as a compute resource in the external
// provider side.
//
type ExternalComputeResourcesService struct {
	BaseService
}

func NewExternalComputeResourcesService(connection *Connection, path string) *ExternalComputeResourcesService {
	var result ExternalComputeResourcesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves a list of external compute resources.
// For example, to retrieve the compute resources of external host provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/computeresources
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_compute_resources>
//   <external_compute_resource href="/ovirt-engine/api/externalhostproviders/123/computeresources/234" id="234">
//     <name>hostname</name>
//     <provider>oVirt</provider>
//     <url>https://address/api</url>
//     <user>admin@internal</user>
//     <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//    </external_compute_resource>
//    ...
// </external_compute_resources>
// ----
// The order of the returned list of compute resources isn't guaranteed.
//
type ExternalComputeResourcesServiceListRequest struct {
	ExternalComputeResourcesService *ExternalComputeResourcesService
	header                          map[string]string
	query                           map[string]string
	follow                          *string
	max                             *int64
}

func (p *ExternalComputeResourcesServiceListRequest) Header(key, value string) *ExternalComputeResourcesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalComputeResourcesServiceListRequest) Query(key, value string) *ExternalComputeResourcesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalComputeResourcesServiceListRequest) Follow(follow string) *ExternalComputeResourcesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ExternalComputeResourcesServiceListRequest) Max(max int64) *ExternalComputeResourcesServiceListRequest {
	p.max = &max
	return p
}

func (p *ExternalComputeResourcesServiceListRequest) Send() (*ExternalComputeResourcesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalComputeResourcesService.connection.URL(), p.ExternalComputeResourcesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalComputeResourcesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalComputeResourcesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalComputeResourcesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalComputeResourcesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalComputeResourcesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalComputeResourceReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ExternalComputeResourcesServiceListResponse{resources: result}, nil
}

func (p *ExternalComputeResourcesServiceListRequest) MustSend() *ExternalComputeResourcesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves a list of external compute resources.
// For example, to retrieve the compute resources of external host provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/computeresources
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_compute_resources>
//   <external_compute_resource href="/ovirt-engine/api/externalhostproviders/123/computeresources/234" id="234">
//     <name>hostname</name>
//     <provider>oVirt</provider>
//     <url>https://address/api</url>
//     <user>admin@internal</user>
//     <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//    </external_compute_resource>
//    ...
// </external_compute_resources>
// ----
// The order of the returned list of compute resources isn't guaranteed.
//
type ExternalComputeResourcesServiceListResponse struct {
	resources *ExternalComputeResourceSlice
}

func (p *ExternalComputeResourcesServiceListResponse) Resources() (*ExternalComputeResourceSlice, bool) {
	if p.resources != nil {
		return p.resources, true
	}
	return nil, false
}

func (p *ExternalComputeResourcesServiceListResponse) MustResources() *ExternalComputeResourceSlice {
	if p.resources == nil {
		panic("resources in response does not exist")
	}
	return p.resources
}

//
// Retrieves a list of external compute resources.
// For example, to retrieve the compute resources of external host provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/computeresources
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_compute_resources>
//   <external_compute_resource href="/ovirt-engine/api/externalhostproviders/123/computeresources/234" id="234">
//     <name>hostname</name>
//     <provider>oVirt</provider>
//     <url>https://address/api</url>
//     <user>admin@internal</user>
//     <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//    </external_compute_resource>
//    ...
// </external_compute_resources>
// ----
// The order of the returned list of compute resources isn't guaranteed.
//
func (p *ExternalComputeResourcesService) List() *ExternalComputeResourcesServiceListRequest {
	return &ExternalComputeResourcesServiceListRequest{ExternalComputeResourcesService: p}
}

//
// This service manages compute resource instance
//
func (op *ExternalComputeResourcesService) ResourceService(id string) *ExternalComputeResourceService {
	return NewExternalComputeResourceService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalComputeResourcesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ResourceService(path), nil
	}
	return op.ResourceService(path[:index]).Service(path[index+1:])
}

func (op *ExternalComputeResourcesService) String() string {
	return fmt.Sprintf("ExternalComputeResourcesService:%s", op.path)
}

//
// A service to manage Katello errata.
// The information is retrieved from Katello.
//
type KatelloErrataService struct {
	BaseService
}

func NewKatelloErrataService(connection *Connection, path string) *KatelloErrataService {
	var result KatelloErrataService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the representation of the Katello errata.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_errata>
//   <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//     <name>RHBA-2013:XYZ</name>
//     <description>The description of the erratum</description>
//     <title>some bug fix update</title>
//     <type>bugfix</type>
//     <issued>2013-11-20T02:00:00.000+02:00</issued>
//     <solution>Few guidelines regarding the solution</solution>
//     <summary>Updated packages that fix one bug are now available for XYZ</summary>
//     <packages>
//       <package>
//         <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//       </package>
//       ...
//     </packages>
//   </katello_erratum>
//   ...
// </katello_errata>
// ----
// The order of the returned list of erratum isn't guaranteed.
//
type KatelloErrataServiceListRequest struct {
	KatelloErrataService *KatelloErrataService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *KatelloErrataServiceListRequest) Header(key, value string) *KatelloErrataServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *KatelloErrataServiceListRequest) Query(key, value string) *KatelloErrataServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *KatelloErrataServiceListRequest) Follow(follow string) *KatelloErrataServiceListRequest {
	p.follow = &follow
	return p
}

func (p *KatelloErrataServiceListRequest) Max(max int64) *KatelloErrataServiceListRequest {
	p.max = &max
	return p
}

func (p *KatelloErrataServiceListRequest) Send() (*KatelloErrataServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.KatelloErrataService.connection.URL(), p.KatelloErrataService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.KatelloErrataService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.KatelloErrataService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.KatelloErrataService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.KatelloErrataService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.KatelloErrataService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLKatelloErratumReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &KatelloErrataServiceListResponse{errata: result}, nil
}

func (p *KatelloErrataServiceListRequest) MustSend() *KatelloErrataServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the representation of the Katello errata.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_errata>
//   <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//     <name>RHBA-2013:XYZ</name>
//     <description>The description of the erratum</description>
//     <title>some bug fix update</title>
//     <type>bugfix</type>
//     <issued>2013-11-20T02:00:00.000+02:00</issued>
//     <solution>Few guidelines regarding the solution</solution>
//     <summary>Updated packages that fix one bug are now available for XYZ</summary>
//     <packages>
//       <package>
//         <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//       </package>
//       ...
//     </packages>
//   </katello_erratum>
//   ...
// </katello_errata>
// ----
// The order of the returned list of erratum isn't guaranteed.
//
type KatelloErrataServiceListResponse struct {
	errata *KatelloErratumSlice
}

func (p *KatelloErrataServiceListResponse) Errata() (*KatelloErratumSlice, bool) {
	if p.errata != nil {
		return p.errata, true
	}
	return nil, false
}

func (p *KatelloErrataServiceListResponse) MustErrata() *KatelloErratumSlice {
	if p.errata == nil {
		panic("errata in response does not exist")
	}
	return p.errata
}

//
// Retrieves the representation of the Katello errata.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_errata>
//   <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//     <name>RHBA-2013:XYZ</name>
//     <description>The description of the erratum</description>
//     <title>some bug fix update</title>
//     <type>bugfix</type>
//     <issued>2013-11-20T02:00:00.000+02:00</issued>
//     <solution>Few guidelines regarding the solution</solution>
//     <summary>Updated packages that fix one bug are now available for XYZ</summary>
//     <packages>
//       <package>
//         <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//       </package>
//       ...
//     </packages>
//   </katello_erratum>
//   ...
// </katello_errata>
// ----
// The order of the returned list of erratum isn't guaranteed.
//
func (p *KatelloErrataService) List() *KatelloErrataServiceListRequest {
	return &KatelloErrataServiceListRequest{KatelloErrataService: p}
}

//
// Reference to the Katello erratum service.
// Use this service to view the erratum by its id.
//
func (op *KatelloErrataService) KatelloErratumService(id string) *KatelloErratumService {
	return NewKatelloErratumService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *KatelloErrataService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.KatelloErratumService(path), nil
	}
	return op.KatelloErratumService(path[:index]).Service(path[index+1:])
}

func (op *KatelloErrataService) String() string {
	return fmt.Sprintf("KatelloErrataService:%s", op.path)
}

//
// This service manages a single discovered host.
//
type ExternalDiscoveredHostService struct {
	BaseService
}

func NewExternalDiscoveredHostService(connection *Connection, path string) *ExternalDiscoveredHostService {
	var result ExternalDiscoveredHostService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get discovered host info.
// Retrieves information about an host that is managed in external provider management system, such as Foreman. The
// information includes hostname, address, subnet, base image and more.
// For example, to get the details of host `234` from provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/discoveredhosts/234
// ....
// The result will be like this:
// [source,xml]
// ----
// <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/234" id="234">
//  <name>mac001a4ad04040</name>
//  <ip>10.34.67.43</ip>
//  <last_report>2017-04-24 11:05:41 UTC</last_report>
//  <mac>00:1a:4a:d0:40:40</mac>
//  <subnet_name>sat0</subnet_name>
//  <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_discovered_host>
// ----
//
type ExternalDiscoveredHostServiceGetRequest struct {
	ExternalDiscoveredHostService *ExternalDiscoveredHostService
	header                        map[string]string
	query                         map[string]string
	follow                        *string
}

func (p *ExternalDiscoveredHostServiceGetRequest) Header(key, value string) *ExternalDiscoveredHostServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalDiscoveredHostServiceGetRequest) Query(key, value string) *ExternalDiscoveredHostServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalDiscoveredHostServiceGetRequest) Follow(follow string) *ExternalDiscoveredHostServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalDiscoveredHostServiceGetRequest) Send() (*ExternalDiscoveredHostServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalDiscoveredHostService.connection.URL(), p.ExternalDiscoveredHostService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalDiscoveredHostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalDiscoveredHostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalDiscoveredHostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalDiscoveredHostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalDiscoveredHostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalDiscoveredHostReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalDiscoveredHostServiceGetResponse{host: result}, nil
}

func (p *ExternalDiscoveredHostServiceGetRequest) MustSend() *ExternalDiscoveredHostServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get discovered host info.
// Retrieves information about an host that is managed in external provider management system, such as Foreman. The
// information includes hostname, address, subnet, base image and more.
// For example, to get the details of host `234` from provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/discoveredhosts/234
// ....
// The result will be like this:
// [source,xml]
// ----
// <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/234" id="234">
//  <name>mac001a4ad04040</name>
//  <ip>10.34.67.43</ip>
//  <last_report>2017-04-24 11:05:41 UTC</last_report>
//  <mac>00:1a:4a:d0:40:40</mac>
//  <subnet_name>sat0</subnet_name>
//  <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_discovered_host>
// ----
//
type ExternalDiscoveredHostServiceGetResponse struct {
	host *ExternalDiscoveredHost
}

func (p *ExternalDiscoveredHostServiceGetResponse) Host() (*ExternalDiscoveredHost, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

func (p *ExternalDiscoveredHostServiceGetResponse) MustHost() *ExternalDiscoveredHost {
	if p.host == nil {
		panic("host in response does not exist")
	}
	return p.host
}

//
// Get discovered host info.
// Retrieves information about an host that is managed in external provider management system, such as Foreman. The
// information includes hostname, address, subnet, base image and more.
// For example, to get the details of host `234` from provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/discoveredhosts/234
// ....
// The result will be like this:
// [source,xml]
// ----
// <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/234" id="234">
//  <name>mac001a4ad04040</name>
//  <ip>10.34.67.43</ip>
//  <last_report>2017-04-24 11:05:41 UTC</last_report>
//  <mac>00:1a:4a:d0:40:40</mac>
//  <subnet_name>sat0</subnet_name>
//  <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_discovered_host>
// ----
//
func (p *ExternalDiscoveredHostService) Get() *ExternalDiscoveredHostServiceGetRequest {
	return &ExternalDiscoveredHostServiceGetRequest{ExternalDiscoveredHostService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalDiscoveredHostService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalDiscoveredHostService) String() string {
	return fmt.Sprintf("ExternalDiscoveredHostService:%s", op.path)
}

//
// Represents an external host provider, such as Foreman or Satellite.
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite for more details on Red Hat Satellite.
//
type ExternalHostProviderService struct {
	BaseService
}

func NewExternalHostProviderService(connection *Connection, path string) *ExternalHostProviderService {
	var result ExternalHostProviderService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get external host provider information
// Host provider, Foreman or Satellite, can be set as an external provider in ovirt. To see details about specific
// host providers attached to ovirt use this API.
// For example, to get the details of host provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123
// ....
// The response will be like this:
// [source,xml]
// ----
// <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123">
//   <name>mysatellite</name>
//   <requires_authentication>true</requires_authentication>
//   <url>https://mysatellite.example.com</url>
//   <username>admin</username>
// </external_host_provider>
// ----
//
type ExternalHostProviderServiceGetRequest struct {
	ExternalHostProviderService *ExternalHostProviderService
	header                      map[string]string
	query                       map[string]string
	follow                      *string
}

func (p *ExternalHostProviderServiceGetRequest) Header(key, value string) *ExternalHostProviderServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProviderServiceGetRequest) Query(key, value string) *ExternalHostProviderServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProviderServiceGetRequest) Follow(follow string) *ExternalHostProviderServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalHostProviderServiceGetRequest) Send() (*ExternalHostProviderServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostProviderService.connection.URL(), p.ExternalHostProviderService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalHostProviderServiceGetResponse{provider: result}, nil
}

func (p *ExternalHostProviderServiceGetRequest) MustSend() *ExternalHostProviderServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get external host provider information
// Host provider, Foreman or Satellite, can be set as an external provider in ovirt. To see details about specific
// host providers attached to ovirt use this API.
// For example, to get the details of host provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123
// ....
// The response will be like this:
// [source,xml]
// ----
// <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123">
//   <name>mysatellite</name>
//   <requires_authentication>true</requires_authentication>
//   <url>https://mysatellite.example.com</url>
//   <username>admin</username>
// </external_host_provider>
// ----
//
type ExternalHostProviderServiceGetResponse struct {
	provider *ExternalHostProvider
}

func (p *ExternalHostProviderServiceGetResponse) Provider() (*ExternalHostProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *ExternalHostProviderServiceGetResponse) MustProvider() *ExternalHostProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Get external host provider information
// Host provider, Foreman or Satellite, can be set as an external provider in ovirt. To see details about specific
// host providers attached to ovirt use this API.
// For example, to get the details of host provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123
// ....
// The response will be like this:
// [source,xml]
// ----
// <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123">
//   <name>mysatellite</name>
//   <requires_authentication>true</requires_authentication>
//   <url>https://mysatellite.example.com</url>
//   <username>admin</username>
// </external_host_provider>
// ----
//
func (p *ExternalHostProviderService) Get() *ExternalHostProviderServiceGetRequest {
	return &ExternalHostProviderServiceGetRequest{ExternalHostProviderService: p}
}

//
// Import the SSL certificates of the external host provider.
//
type ExternalHostProviderServiceImportCertificatesRequest struct {
	ExternalHostProviderService *ExternalHostProviderService
	header                      map[string]string
	query                       map[string]string
	certificates                *CertificateSlice
}

func (p *ExternalHostProviderServiceImportCertificatesRequest) Header(key, value string) *ExternalHostProviderServiceImportCertificatesRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProviderServiceImportCertificatesRequest) Query(key, value string) *ExternalHostProviderServiceImportCertificatesRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProviderServiceImportCertificatesRequest) Certificates(certificates *CertificateSlice) *ExternalHostProviderServiceImportCertificatesRequest {
	p.certificates = certificates
	return p
}

func (p *ExternalHostProviderServiceImportCertificatesRequest) CertificatesOfAny(anys ...*Certificate) *ExternalHostProviderServiceImportCertificatesRequest {
	if p.certificates == nil {
		p.certificates = new(CertificateSlice)
	}
	p.certificates.slice = append(p.certificates.slice, anys...)
	return p
}

func (p *ExternalHostProviderServiceImportCertificatesRequest) Send() (*ExternalHostProviderServiceImportCertificatesResponse, error) {
	rawURL := fmt.Sprintf("%s%s/importcertificates", p.ExternalHostProviderService.connection.URL(), p.ExternalHostProviderService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Certificates(p.certificates)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(ExternalHostProviderServiceImportCertificatesResponse), nil
}

func (p *ExternalHostProviderServiceImportCertificatesRequest) MustSend() *ExternalHostProviderServiceImportCertificatesResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Import the SSL certificates of the external host provider.
//
type ExternalHostProviderServiceImportCertificatesResponse struct {
}

//
// Import the SSL certificates of the external host provider.
//
func (p *ExternalHostProviderService) ImportCertificates() *ExternalHostProviderServiceImportCertificatesRequest {
	return &ExternalHostProviderServiceImportCertificatesRequest{ExternalHostProviderService: p}
}

//
//
type ExternalHostProviderServiceRemoveRequest struct {
	ExternalHostProviderService *ExternalHostProviderService
	header                      map[string]string
	query                       map[string]string
	async                       *bool
}

func (p *ExternalHostProviderServiceRemoveRequest) Header(key, value string) *ExternalHostProviderServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProviderServiceRemoveRequest) Query(key, value string) *ExternalHostProviderServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProviderServiceRemoveRequest) Async(async bool) *ExternalHostProviderServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *ExternalHostProviderServiceRemoveRequest) Send() (*ExternalHostProviderServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostProviderService.connection.URL(), p.ExternalHostProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(ExternalHostProviderServiceRemoveResponse), nil
}

func (p *ExternalHostProviderServiceRemoveRequest) MustSend() *ExternalHostProviderServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type ExternalHostProviderServiceRemoveResponse struct {
}

//
//
func (p *ExternalHostProviderService) Remove() *ExternalHostProviderServiceRemoveRequest {
	return &ExternalHostProviderServiceRemoveRequest{ExternalHostProviderService: p}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type ExternalHostProviderServiceTestConnectivityRequest struct {
	ExternalHostProviderService *ExternalHostProviderService
	header                      map[string]string
	query                       map[string]string
	async                       *bool
}

func (p *ExternalHostProviderServiceTestConnectivityRequest) Header(key, value string) *ExternalHostProviderServiceTestConnectivityRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProviderServiceTestConnectivityRequest) Query(key, value string) *ExternalHostProviderServiceTestConnectivityRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProviderServiceTestConnectivityRequest) Async(async bool) *ExternalHostProviderServiceTestConnectivityRequest {
	p.async = &async
	return p
}

func (p *ExternalHostProviderServiceTestConnectivityRequest) Send() (*ExternalHostProviderServiceTestConnectivityResponse, error) {
	rawURL := fmt.Sprintf("%s%s/testconnectivity", p.ExternalHostProviderService.connection.URL(), p.ExternalHostProviderService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(ExternalHostProviderServiceTestConnectivityResponse), nil
}

func (p *ExternalHostProviderServiceTestConnectivityRequest) MustSend() *ExternalHostProviderServiceTestConnectivityResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type ExternalHostProviderServiceTestConnectivityResponse struct {
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
func (p *ExternalHostProviderService) TestConnectivity() *ExternalHostProviderServiceTestConnectivityRequest {
	return &ExternalHostProviderServiceTestConnectivityRequest{ExternalHostProviderService: p}
}

//
// Update the specified external host provider in the system.
//
type ExternalHostProviderServiceUpdateRequest struct {
	ExternalHostProviderService *ExternalHostProviderService
	header                      map[string]string
	query                       map[string]string
	async                       *bool
	provider                    *ExternalHostProvider
}

func (p *ExternalHostProviderServiceUpdateRequest) Header(key, value string) *ExternalHostProviderServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProviderServiceUpdateRequest) Query(key, value string) *ExternalHostProviderServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProviderServiceUpdateRequest) Async(async bool) *ExternalHostProviderServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *ExternalHostProviderServiceUpdateRequest) Provider(provider *ExternalHostProvider) *ExternalHostProviderServiceUpdateRequest {
	p.provider = provider
	return p
}

func (p *ExternalHostProviderServiceUpdateRequest) Send() (*ExternalHostProviderServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostProviderService.connection.URL(), p.ExternalHostProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLExternalHostProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalHostProviderServiceUpdateResponse{provider: result}, nil
}

func (p *ExternalHostProviderServiceUpdateRequest) MustSend() *ExternalHostProviderServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified external host provider in the system.
//
type ExternalHostProviderServiceUpdateResponse struct {
	provider *ExternalHostProvider
}

func (p *ExternalHostProviderServiceUpdateResponse) Provider() (*ExternalHostProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *ExternalHostProviderServiceUpdateResponse) MustProvider() *ExternalHostProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Update the specified external host provider in the system.
//
func (p *ExternalHostProviderService) Update() *ExternalHostProviderServiceUpdateRequest {
	return &ExternalHostProviderServiceUpdateRequest{ExternalHostProviderService: p}
}

//
// A service to view certificates for this external provider.
//
func (op *ExternalHostProviderService) CertificatesService() *ExternalProviderCertificatesService {
	return NewExternalProviderCertificatesService(op.connection, fmt.Sprintf("%s/certificates", op.path))
}

//
//
func (op *ExternalHostProviderService) ComputeResourcesService() *ExternalComputeResourcesService {
	return NewExternalComputeResourcesService(op.connection, fmt.Sprintf("%s/computeresources", op.path))
}

//
//
func (op *ExternalHostProviderService) DiscoveredHostsService() *ExternalDiscoveredHostsService {
	return NewExternalDiscoveredHostsService(op.connection, fmt.Sprintf("%s/discoveredhosts", op.path))
}

//
//
func (op *ExternalHostProviderService) HostGroupsService() *ExternalHostGroupsService {
	return NewExternalHostGroupsService(op.connection, fmt.Sprintf("%s/hostgroups", op.path))
}

//
//
func (op *ExternalHostProviderService) HostsService() *ExternalHostsService {
	return NewExternalHostsService(op.connection, fmt.Sprintf("%s/hosts", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalHostProviderService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "certificates" {
		return op.CertificatesService(), nil
	}
	if strings.HasPrefix(path, "certificates/") {
		return op.CertificatesService().Service(path[13:])
	}
	if path == "computeresources" {
		return op.ComputeResourcesService(), nil
	}
	if strings.HasPrefix(path, "computeresources/") {
		return op.ComputeResourcesService().Service(path[17:])
	}
	if path == "discoveredhosts" {
		return op.DiscoveredHostsService(), nil
	}
	if strings.HasPrefix(path, "discoveredhosts/") {
		return op.DiscoveredHostsService().Service(path[16:])
	}
	if path == "hostgroups" {
		return op.HostGroupsService(), nil
	}
	if strings.HasPrefix(path, "hostgroups/") {
		return op.HostGroupsService().Service(path[11:])
	}
	if path == "hosts" {
		return op.HostsService(), nil
	}
	if strings.HasPrefix(path, "hosts/") {
		return op.HostsService().Service(path[6:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalHostProviderService) String() string {
	return fmt.Sprintf("ExternalHostProviderService:%s", op.path)
}

//
// A service to manage a Katello erratum.
//
type KatelloErratumService struct {
	BaseService
}

func NewKatelloErratumService(connection *Connection, path string) *KatelloErratumService {
	var result KatelloErratumService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves a Katello erratum.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata/123
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//   <name>RHBA-2013:XYZ</name>
//   <description>The description of the erratum</description>
//   <title>some bug fix update</title>
//   <type>bugfix</type>
//   <issued>2013-11-20T02:00:00.000+02:00</issued>
//   <solution>Few guidelines regarding the solution</solution>
//   <summary>Updated packages that fix one bug are now available for XYZ</summary>
//   <packages>
//     <package>
//       <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//     </package>
//     ...
//   </packages>
// </katello_erratum>
// ----
//
type KatelloErratumServiceGetRequest struct {
	KatelloErratumService *KatelloErratumService
	header                map[string]string
	query                 map[string]string
	follow                *string
}

func (p *KatelloErratumServiceGetRequest) Header(key, value string) *KatelloErratumServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *KatelloErratumServiceGetRequest) Query(key, value string) *KatelloErratumServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *KatelloErratumServiceGetRequest) Follow(follow string) *KatelloErratumServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *KatelloErratumServiceGetRequest) Send() (*KatelloErratumServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.KatelloErratumService.connection.URL(), p.KatelloErratumService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.KatelloErratumService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.KatelloErratumService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.KatelloErratumService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.KatelloErratumService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.KatelloErratumService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLKatelloErratumReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &KatelloErratumServiceGetResponse{erratum: result}, nil
}

func (p *KatelloErratumServiceGetRequest) MustSend() *KatelloErratumServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves a Katello erratum.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata/123
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//   <name>RHBA-2013:XYZ</name>
//   <description>The description of the erratum</description>
//   <title>some bug fix update</title>
//   <type>bugfix</type>
//   <issued>2013-11-20T02:00:00.000+02:00</issued>
//   <solution>Few guidelines regarding the solution</solution>
//   <summary>Updated packages that fix one bug are now available for XYZ</summary>
//   <packages>
//     <package>
//       <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//     </package>
//     ...
//   </packages>
// </katello_erratum>
// ----
//
type KatelloErratumServiceGetResponse struct {
	erratum *KatelloErratum
}

func (p *KatelloErratumServiceGetResponse) Erratum() (*KatelloErratum, bool) {
	if p.erratum != nil {
		return p.erratum, true
	}
	return nil, false
}

func (p *KatelloErratumServiceGetResponse) MustErratum() *KatelloErratum {
	if p.erratum == nil {
		panic("erratum in response does not exist")
	}
	return p.erratum
}

//
// Retrieves a Katello erratum.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata/123
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//   <name>RHBA-2013:XYZ</name>
//   <description>The description of the erratum</description>
//   <title>some bug fix update</title>
//   <type>bugfix</type>
//   <issued>2013-11-20T02:00:00.000+02:00</issued>
//   <solution>Few guidelines regarding the solution</solution>
//   <summary>Updated packages that fix one bug are now available for XYZ</summary>
//   <packages>
//     <package>
//       <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//     </package>
//     ...
//   </packages>
// </katello_erratum>
// ----
//
func (p *KatelloErratumService) Get() *KatelloErratumServiceGetRequest {
	return &KatelloErratumServiceGetRequest{KatelloErratumService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *KatelloErratumService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *KatelloErratumService) String() string {
	return fmt.Sprintf("KatelloErratumService:%s", op.path)
}

//
//
type ExternalHostProvidersService struct {
	BaseService
}

func NewExternalHostProvidersService(connection *Connection, path string) *ExternalHostProvidersService {
	var result ExternalHostProvidersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new external host provider to the system.
//
type ExternalHostProvidersServiceAddRequest struct {
	ExternalHostProvidersService *ExternalHostProvidersService
	header                       map[string]string
	query                        map[string]string
	provider                     *ExternalHostProvider
}

func (p *ExternalHostProvidersServiceAddRequest) Header(key, value string) *ExternalHostProvidersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProvidersServiceAddRequest) Query(key, value string) *ExternalHostProvidersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProvidersServiceAddRequest) Provider(provider *ExternalHostProvider) *ExternalHostProvidersServiceAddRequest {
	p.provider = provider
	return p
}

func (p *ExternalHostProvidersServiceAddRequest) Send() (*ExternalHostProvidersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostProvidersService.connection.URL(), p.ExternalHostProvidersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLExternalHostProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalHostProvidersServiceAddResponse{provider: result}, nil
}

func (p *ExternalHostProvidersServiceAddRequest) MustSend() *ExternalHostProvidersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new external host provider to the system.
//
type ExternalHostProvidersServiceAddResponse struct {
	provider *ExternalHostProvider
}

func (p *ExternalHostProvidersServiceAddResponse) Provider() (*ExternalHostProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *ExternalHostProvidersServiceAddResponse) MustProvider() *ExternalHostProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Add a new external host provider to the system.
//
func (p *ExternalHostProvidersService) Add() *ExternalHostProvidersServiceAddRequest {
	return &ExternalHostProvidersServiceAddRequest{ExternalHostProvidersService: p}
}

//
// Returns the list of external host providers.
// The order of the returned list of host providers isn't guaranteed.
//
type ExternalHostProvidersServiceListRequest struct {
	ExternalHostProvidersService *ExternalHostProvidersService
	header                       map[string]string
	query                        map[string]string
	follow                       *string
	max                          *int64
	search                       *string
}

func (p *ExternalHostProvidersServiceListRequest) Header(key, value string) *ExternalHostProvidersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProvidersServiceListRequest) Query(key, value string) *ExternalHostProvidersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProvidersServiceListRequest) Follow(follow string) *ExternalHostProvidersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ExternalHostProvidersServiceListRequest) Max(max int64) *ExternalHostProvidersServiceListRequest {
	p.max = &max
	return p
}

func (p *ExternalHostProvidersServiceListRequest) Search(search string) *ExternalHostProvidersServiceListRequest {
	p.search = &search
	return p
}

func (p *ExternalHostProvidersServiceListRequest) Send() (*ExternalHostProvidersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostProvidersService.connection.URL(), p.ExternalHostProvidersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostProviderReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ExternalHostProvidersServiceListResponse{providers: result}, nil
}

func (p *ExternalHostProvidersServiceListRequest) MustSend() *ExternalHostProvidersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of external host providers.
// The order of the returned list of host providers isn't guaranteed.
//
type ExternalHostProvidersServiceListResponse struct {
	providers *ExternalHostProviderSlice
}

func (p *ExternalHostProvidersServiceListResponse) Providers() (*ExternalHostProviderSlice, bool) {
	if p.providers != nil {
		return p.providers, true
	}
	return nil, false
}

func (p *ExternalHostProvidersServiceListResponse) MustProviders() *ExternalHostProviderSlice {
	if p.providers == nil {
		panic("providers in response does not exist")
	}
	return p.providers
}

//
// Returns the list of external host providers.
// The order of the returned list of host providers isn't guaranteed.
//
func (p *ExternalHostProvidersService) List() *ExternalHostProvidersServiceListRequest {
	return &ExternalHostProvidersServiceListRequest{ExternalHostProvidersService: p}
}

//
//
func (op *ExternalHostProvidersService) ProviderService(id string) *ExternalHostProviderService {
	return NewExternalHostProviderService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalHostProvidersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProviderService(path), nil
	}
	return op.ProviderService(path[:index]).Service(path[index+1:])
}

func (op *ExternalHostProvidersService) String() string {
	return fmt.Sprintf("ExternalHostProvidersService:%s", op.path)
}

//
// This service manages external discovered hosts.
//
type ExternalDiscoveredHostsService struct {
	BaseService
}

func NewExternalDiscoveredHostsService(connection *Connection, path string) *ExternalDiscoveredHostsService {
	var result ExternalDiscoveredHostsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get list of discovered hosts' information.
// Discovered hosts are fetched from third-party providers such as Foreman.
// To list all discovered hosts for provider `123` send the following:
// [source]
// ----
// GET /ovirt-engine/api/externalhostproviders/123/discoveredhost
// ----
// [source,xml]
// ----
// <external_discovered_hosts>
//  <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/456" id="456">
//   <name>mac001a4ad04031</name>
//   <ip>10.34.67.42</ip>
//   <last_report>2017-04-24 11:05:41 UTC</last_report>
//   <mac>00:1a:4a:d0:40:31</mac>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//  </external_discovered_host>
//  <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/789" id="789">
//   <name>mac001a4ad04040</name>
//   <ip>10.34.67.43</ip>
//   <last_report>2017-04-24 11:05:41 UTC</last_report>
//   <mac>00:1a:4a:d0:40:40</mac>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//  </external_discovered_host>
//  ...
// </external_discovered_hosts>
// ----
// The order of the returned list of hosts isn't guaranteed.
//
type ExternalDiscoveredHostsServiceListRequest struct {
	ExternalDiscoveredHostsService *ExternalDiscoveredHostsService
	header                         map[string]string
	query                          map[string]string
	follow                         *string
	max                            *int64
}

func (p *ExternalDiscoveredHostsServiceListRequest) Header(key, value string) *ExternalDiscoveredHostsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalDiscoveredHostsServiceListRequest) Query(key, value string) *ExternalDiscoveredHostsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalDiscoveredHostsServiceListRequest) Follow(follow string) *ExternalDiscoveredHostsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ExternalDiscoveredHostsServiceListRequest) Max(max int64) *ExternalDiscoveredHostsServiceListRequest {
	p.max = &max
	return p
}

func (p *ExternalDiscoveredHostsServiceListRequest) Send() (*ExternalDiscoveredHostsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalDiscoveredHostsService.connection.URL(), p.ExternalDiscoveredHostsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalDiscoveredHostsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalDiscoveredHostsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalDiscoveredHostsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalDiscoveredHostsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalDiscoveredHostsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalDiscoveredHostReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ExternalDiscoveredHostsServiceListResponse{hosts: result}, nil
}

func (p *ExternalDiscoveredHostsServiceListRequest) MustSend() *ExternalDiscoveredHostsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get list of discovered hosts' information.
// Discovered hosts are fetched from third-party providers such as Foreman.
// To list all discovered hosts for provider `123` send the following:
// [source]
// ----
// GET /ovirt-engine/api/externalhostproviders/123/discoveredhost
// ----
// [source,xml]
// ----
// <external_discovered_hosts>
//  <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/456" id="456">
//   <name>mac001a4ad04031</name>
//   <ip>10.34.67.42</ip>
//   <last_report>2017-04-24 11:05:41 UTC</last_report>
//   <mac>00:1a:4a:d0:40:31</mac>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//  </external_discovered_host>
//  <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/789" id="789">
//   <name>mac001a4ad04040</name>
//   <ip>10.34.67.43</ip>
//   <last_report>2017-04-24 11:05:41 UTC</last_report>
//   <mac>00:1a:4a:d0:40:40</mac>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//  </external_discovered_host>
//  ...
// </external_discovered_hosts>
// ----
// The order of the returned list of hosts isn't guaranteed.
//
type ExternalDiscoveredHostsServiceListResponse struct {
	hosts *ExternalDiscoveredHostSlice
}

func (p *ExternalDiscoveredHostsServiceListResponse) Hosts() (*ExternalDiscoveredHostSlice, bool) {
	if p.hosts != nil {
		return p.hosts, true
	}
	return nil, false
}

func (p *ExternalDiscoveredHostsServiceListResponse) MustHosts() *ExternalDiscoveredHostSlice {
	if p.hosts == nil {
		panic("hosts in response does not exist")
	}
	return p.hosts
}

//
// Get list of discovered hosts' information.
// Discovered hosts are fetched from third-party providers such as Foreman.
// To list all discovered hosts for provider `123` send the following:
// [source]
// ----
// GET /ovirt-engine/api/externalhostproviders/123/discoveredhost
// ----
// [source,xml]
// ----
// <external_discovered_hosts>
//  <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/456" id="456">
//   <name>mac001a4ad04031</name>
//   <ip>10.34.67.42</ip>
//   <last_report>2017-04-24 11:05:41 UTC</last_report>
//   <mac>00:1a:4a:d0:40:31</mac>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//  </external_discovered_host>
//  <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/789" id="789">
//   <name>mac001a4ad04040</name>
//   <ip>10.34.67.43</ip>
//   <last_report>2017-04-24 11:05:41 UTC</last_report>
//   <mac>00:1a:4a:d0:40:40</mac>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//  </external_discovered_host>
//  ...
// </external_discovered_hosts>
// ----
// The order of the returned list of hosts isn't guaranteed.
//
func (p *ExternalDiscoveredHostsService) List() *ExternalDiscoveredHostsServiceListRequest {
	return &ExternalDiscoveredHostsServiceListRequest{ExternalDiscoveredHostsService: p}
}

//
//
func (op *ExternalDiscoveredHostsService) HostService(id string) *ExternalDiscoveredHostService {
	return NewExternalDiscoveredHostService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalDiscoveredHostsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.HostService(path), nil
	}
	return op.HostService(path[:index]).Service(path[index+1:])
}

func (op *ExternalDiscoveredHostsService) String() string {
	return fmt.Sprintf("ExternalDiscoveredHostsService:%s", op.path)
}

//
// A service to manage Katello errata assigned to the engine.
// The information is retrieved from Katello.
//
type EngineKatelloErrataService struct {
	BaseService
}

func NewEngineKatelloErrataService(connection *Connection, path string) *EngineKatelloErrataService {
	var result EngineKatelloErrataService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the representation of the Katello errata.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_errata>
//   <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//     <name>RHBA-2013:XYZ</name>
//     <description>The description of the erratum</description>
//     <title>some bug fix update</title>
//     <type>bugfix</type>
//     <issued>2013-11-20T02:00:00.000+02:00</issued>
//     <solution>Few guidelines regarding the solution</solution>
//     <summary>Updated packages that fix one bug are now available for XYZ</summary>
//     <packages>
//       <package>
//         <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//       </package>
//       ...
//     </packages>
//   </katello_erratum>
//   ...
// </katello_errata>
// ----
// The order of the returned list of erratum isn't guaranteed.
//
type EngineKatelloErrataServiceListRequest struct {
	EngineKatelloErrataService *EngineKatelloErrataService
	header                     map[string]string
	query                      map[string]string
	follow                     *string
	max                        *int64
}

func (p *EngineKatelloErrataServiceListRequest) Header(key, value string) *EngineKatelloErrataServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EngineKatelloErrataServiceListRequest) Query(key, value string) *EngineKatelloErrataServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EngineKatelloErrataServiceListRequest) Follow(follow string) *EngineKatelloErrataServiceListRequest {
	p.follow = &follow
	return p
}

func (p *EngineKatelloErrataServiceListRequest) Max(max int64) *EngineKatelloErrataServiceListRequest {
	p.max = &max
	return p
}

func (p *EngineKatelloErrataServiceListRequest) Send() (*EngineKatelloErrataServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EngineKatelloErrataService.connection.URL(), p.EngineKatelloErrataService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EngineKatelloErrataService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EngineKatelloErrataService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EngineKatelloErrataService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EngineKatelloErrataService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EngineKatelloErrataService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLKatelloErratumReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &EngineKatelloErrataServiceListResponse{errata: result}, nil
}

func (p *EngineKatelloErrataServiceListRequest) MustSend() *EngineKatelloErrataServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the representation of the Katello errata.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_errata>
//   <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//     <name>RHBA-2013:XYZ</name>
//     <description>The description of the erratum</description>
//     <title>some bug fix update</title>
//     <type>bugfix</type>
//     <issued>2013-11-20T02:00:00.000+02:00</issued>
//     <solution>Few guidelines regarding the solution</solution>
//     <summary>Updated packages that fix one bug are now available for XYZ</summary>
//     <packages>
//       <package>
//         <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//       </package>
//       ...
//     </packages>
//   </katello_erratum>
//   ...
// </katello_errata>
// ----
// The order of the returned list of erratum isn't guaranteed.
//
type EngineKatelloErrataServiceListResponse struct {
	errata *KatelloErratumSlice
}

func (p *EngineKatelloErrataServiceListResponse) Errata() (*KatelloErratumSlice, bool) {
	if p.errata != nil {
		return p.errata, true
	}
	return nil, false
}

func (p *EngineKatelloErrataServiceListResponse) MustErrata() *KatelloErratumSlice {
	if p.errata == nil {
		panic("errata in response does not exist")
	}
	return p.errata
}

//
// Retrieves the representation of the Katello errata.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_errata>
//   <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//     <name>RHBA-2013:XYZ</name>
//     <description>The description of the erratum</description>
//     <title>some bug fix update</title>
//     <type>bugfix</type>
//     <issued>2013-11-20T02:00:00.000+02:00</issued>
//     <solution>Few guidelines regarding the solution</solution>
//     <summary>Updated packages that fix one bug are now available for XYZ</summary>
//     <packages>
//       <package>
//         <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//       </package>
//       ...
//     </packages>
//   </katello_erratum>
//   ...
// </katello_errata>
// ----
// The order of the returned list of erratum isn't guaranteed.
//
func (p *EngineKatelloErrataService) List() *EngineKatelloErrataServiceListRequest {
	return &EngineKatelloErrataServiceListRequest{EngineKatelloErrataService: p}
}

//
// Reference to the Katello erratum service.
// Use this service to view the erratum by its id.
//
func (op *EngineKatelloErrataService) KatelloErratumService(id string) *KatelloErratumService {
	return NewKatelloErratumService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *EngineKatelloErrataService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.KatelloErratumService(path), nil
	}
	return op.KatelloErratumService(path[:index]).Service(path[index+1:])
}

func (op *EngineKatelloErrataService) String() string {
	return fmt.Sprintf("EngineKatelloErrataService:%s", op.path)
}

//
// This service manages a single host group information.
// Host group is a term of host provider - the host group includes provision details that are applied to new discovered
// host. Information such as subnet, operating system, domain, etc.
//
type ExternalHostGroupService struct {
	BaseService
}

func NewExternalHostGroupService(connection *Connection, path string) *ExternalHostGroupService {
	var result ExternalHostGroupService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get host group information.
// For example, to get the details of hostgroup `234` of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/hostgroups/234
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_host_group href="/ovirt-engine/api/externalhostproviders/123/hostgroups/234" id="234">
//   <name>rhel7</name>
//   <architecture_name>x86_64</architecture_name>
//   <domain_name>s.com</domain_name>
//   <operating_system_name>RedHat 7.3</operating_system_name>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_host_group>
// ----
//
type ExternalHostGroupServiceGetRequest struct {
	ExternalHostGroupService *ExternalHostGroupService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
}

func (p *ExternalHostGroupServiceGetRequest) Header(key, value string) *ExternalHostGroupServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostGroupServiceGetRequest) Query(key, value string) *ExternalHostGroupServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostGroupServiceGetRequest) Follow(follow string) *ExternalHostGroupServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalHostGroupServiceGetRequest) Send() (*ExternalHostGroupServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostGroupService.connection.URL(), p.ExternalHostGroupService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostGroupService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostGroupService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostGroupService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostGroupService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostGroupService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostGroupReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalHostGroupServiceGetResponse{group: result}, nil
}

func (p *ExternalHostGroupServiceGetRequest) MustSend() *ExternalHostGroupServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get host group information.
// For example, to get the details of hostgroup `234` of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/hostgroups/234
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_host_group href="/ovirt-engine/api/externalhostproviders/123/hostgroups/234" id="234">
//   <name>rhel7</name>
//   <architecture_name>x86_64</architecture_name>
//   <domain_name>s.com</domain_name>
//   <operating_system_name>RedHat 7.3</operating_system_name>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_host_group>
// ----
//
type ExternalHostGroupServiceGetResponse struct {
	group *ExternalHostGroup
}

func (p *ExternalHostGroupServiceGetResponse) Group() (*ExternalHostGroup, bool) {
	if p.group != nil {
		return p.group, true
	}
	return nil, false
}

func (p *ExternalHostGroupServiceGetResponse) MustGroup() *ExternalHostGroup {
	if p.group == nil {
		panic("group in response does not exist")
	}
	return p.group
}

//
// Get host group information.
// For example, to get the details of hostgroup `234` of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/hostgroups/234
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_host_group href="/ovirt-engine/api/externalhostproviders/123/hostgroups/234" id="234">
//   <name>rhel7</name>
//   <architecture_name>x86_64</architecture_name>
//   <domain_name>s.com</domain_name>
//   <operating_system_name>RedHat 7.3</operating_system_name>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_host_group>
// ----
//
func (p *ExternalHostGroupService) Get() *ExternalHostGroupServiceGetRequest {
	return &ExternalHostGroupServiceGetRequest{ExternalHostGroupService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalHostGroupService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalHostGroupService) String() string {
	return fmt.Sprintf("ExternalHostGroupService:%s", op.path)
}

//
// Manages the set of quota limits configured for a cluster.
//
type QuotaClusterLimitsService struct {
	BaseService
}

func NewQuotaClusterLimitsService(connection *Connection, path string) *QuotaClusterLimitsService {
	var result QuotaClusterLimitsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a cluster limit to a specified Quota.
//
type QuotaClusterLimitsServiceAddRequest struct {
	QuotaClusterLimitsService *QuotaClusterLimitsService
	header                    map[string]string
	query                     map[string]string
	limit                     *QuotaClusterLimit
}

func (p *QuotaClusterLimitsServiceAddRequest) Header(key, value string) *QuotaClusterLimitsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *QuotaClusterLimitsServiceAddRequest) Query(key, value string) *QuotaClusterLimitsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *QuotaClusterLimitsServiceAddRequest) Limit(limit *QuotaClusterLimit) *QuotaClusterLimitsServiceAddRequest {
	p.limit = limit
	return p
}

func (p *QuotaClusterLimitsServiceAddRequest) Send() (*QuotaClusterLimitsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.QuotaClusterLimitsService.connection.URL(), p.QuotaClusterLimitsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLQuotaClusterLimitWriteOne(writer, p.limit, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.QuotaClusterLimitsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.QuotaClusterLimitsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.QuotaClusterLimitsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.QuotaClusterLimitsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.QuotaClusterLimitsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLQuotaClusterLimitReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &QuotaClusterLimitsServiceAddResponse{limit: result}, nil
}

func (p *QuotaClusterLimitsServiceAddRequest) MustSend() *QuotaClusterLimitsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a cluster limit to a specified Quota.
//
type QuotaClusterLimitsServiceAddResponse struct {
	limit *QuotaClusterLimit
}

func (p *QuotaClusterLimitsServiceAddResponse) Limit() (*QuotaClusterLimit, bool) {
	if p.limit != nil {
		return p.limit, true
	}
	return nil, false
}

func (p *QuotaClusterLimitsServiceAddResponse) MustLimit() *QuotaClusterLimit {
	if p.limit == nil {
		panic("limit in response does not exist")
	}
	return p.limit
}

//
// Add a cluster limit to a specified Quota.
//
func (p *QuotaClusterLimitsService) Add() *QuotaClusterLimitsServiceAddRequest {
	return &QuotaClusterLimitsServiceAddRequest{QuotaClusterLimitsService: p}
}

//
// Returns the set of quota limits configured for the cluster.
// The returned list of quota limits isn't guaranteed.
//
type QuotaClusterLimitsServiceListRequest struct {
	QuotaClusterLimitsService *QuotaClusterLimitsService
	header                    map[string]string
	query                     map[string]string
	follow                    *string
	max                       *int64
}

func (p *QuotaClusterLimitsServiceListRequest) Header(key, value string) *QuotaClusterLimitsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *QuotaClusterLimitsServiceListRequest) Query(key, value string) *QuotaClusterLimitsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *QuotaClusterLimitsServiceListRequest) Follow(follow string) *QuotaClusterLimitsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *QuotaClusterLimitsServiceListRequest) Max(max int64) *QuotaClusterLimitsServiceListRequest {
	p.max = &max
	return p
}

func (p *QuotaClusterLimitsServiceListRequest) Send() (*QuotaClusterLimitsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.QuotaClusterLimitsService.connection.URL(), p.QuotaClusterLimitsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.QuotaClusterLimitsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.QuotaClusterLimitsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.QuotaClusterLimitsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.QuotaClusterLimitsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.QuotaClusterLimitsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLQuotaClusterLimitReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &QuotaClusterLimitsServiceListResponse{limits: result}, nil
}

func (p *QuotaClusterLimitsServiceListRequest) MustSend() *QuotaClusterLimitsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the set of quota limits configured for the cluster.
// The returned list of quota limits isn't guaranteed.
//
type QuotaClusterLimitsServiceListResponse struct {
	limits *QuotaClusterLimitSlice
}

func (p *QuotaClusterLimitsServiceListResponse) Limits() (*QuotaClusterLimitSlice, bool) {
	if p.limits != nil {
		return p.limits, true
	}
	return nil, false
}

func (p *QuotaClusterLimitsServiceListResponse) MustLimits() *QuotaClusterLimitSlice {
	if p.limits == nil {
		panic("limits in response does not exist")
	}
	return p.limits
}

//
// Returns the set of quota limits configured for the cluster.
// The returned list of quota limits isn't guaranteed.
//
func (p *QuotaClusterLimitsService) List() *QuotaClusterLimitsServiceListRequest {
	return &QuotaClusterLimitsServiceListRequest{QuotaClusterLimitsService: p}
}

//
//
func (op *QuotaClusterLimitsService) LimitService(id string) *QuotaClusterLimitService {
	return NewQuotaClusterLimitService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *QuotaClusterLimitsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.LimitService(path), nil
	}
	return op.LimitService(path[:index]).Service(path[index+1:])
}

func (op *QuotaClusterLimitsService) String() string {
	return fmt.Sprintf("QuotaClusterLimitsService:%s", op.path)
}

//
// This service doesn't add any new methods, it is just a placeholder for the annotation that specifies the path of the
// resource that manages the permissions assigned to the system object.
//
type SystemPermissionsService struct {
	BaseService
}

func NewSystemPermissionsService(connection *Connection, path string) *SystemPermissionsService {
	var result SystemPermissionsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Assign a new permission to a user or group for specific entity.
// For example, to assign the `UserVmManager` role to the virtual machine with id `123` to the user with id `456`
// send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserVmManager</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// To assign the `SuperUser` role to the system to the user with id `456` send a request like this:
// ....
// POST /ovirt-engine/api/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>SuperUser</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// If you want to assign permission to the group instead of the user please replace the `user` element with the
// `group` element with proper `id` of the group. For example to assign the `UserRole` role to the cluster with
// id `123` to the group with id `789` send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserRole</name>
//   </role>
//   <group id="789"/>
// </permission>
// ----
//
type SystemPermissionsServiceAddRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddRequest) Header(key, value string) *SystemPermissionsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddRequest) Query(key, value string) *SystemPermissionsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddRequest) Permission(permission *Permission) *SystemPermissionsServiceAddRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddRequest) Send() (*SystemPermissionsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLPermissionWriteOne(writer, p.permission, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLPermissionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SystemPermissionsServiceAddResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddRequest) MustSend() *SystemPermissionsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Assign a new permission to a user or group for specific entity.
// For example, to assign the `UserVmManager` role to the virtual machine with id `123` to the user with id `456`
// send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserVmManager</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// To assign the `SuperUser` role to the system to the user with id `456` send a request like this:
// ....
// POST /ovirt-engine/api/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>SuperUser</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// If you want to assign permission to the group instead of the user please replace the `user` element with the
// `group` element with proper `id` of the group. For example to assign the `UserRole` role to the cluster with
// id `123` to the group with id `789` send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserRole</name>
//   </role>
//   <group id="789"/>
// </permission>
// ----
//
type SystemPermissionsServiceAddResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Assign a new permission to a user or group for specific entity.
// For example, to assign the `UserVmManager` role to the virtual machine with id `123` to the user with id `456`
// send a request like this:
// ....
// POST /ovirt-engine/api/vms/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserVmManager</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// To assign the `SuperUser` role to the system to the user with id `456` send a request like this:
// ....
// POST /ovirt-engine/api/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>SuperUser</name>
//   </role>
//   <user id="456"/>
// </permission>
// ----
// If you want to assign permission to the group instead of the user please replace the `user` element with the
// `group` element with proper `id` of the group. For example to assign the `UserRole` role to the cluster with
// id `123` to the group with id `789` send a request like this:
// ....
// POST /ovirt-engine/api/clusters/123/permissions
// ....
// With a request body like this:
// [source,xml]
// ----
// <permission>
//   <role>
//     <name>UserRole</name>
//   </role>
//   <group id="789"/>
// </permission>
// ----
//
func (p *SystemPermissionsService) Add() *SystemPermissionsServiceAddRequest {
	return &SystemPermissionsServiceAddRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the cluster to the group in the system.
//
type SystemPermissionsServiceAddClusterPermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddClusterPermissionRequest) Header(key, value string) *SystemPermissionsServiceAddClusterPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddClusterPermissionRequest) Query(key, value string) *SystemPermissionsServiceAddClusterPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddClusterPermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddClusterPermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddClusterPermissionRequest) Send() (*SystemPermissionsServiceAddClusterPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/clusterpermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddClusterPermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddClusterPermissionRequest) MustSend() *SystemPermissionsServiceAddClusterPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the cluster to the group in the system.
//
type SystemPermissionsServiceAddClusterPermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddClusterPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddClusterPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the cluster to the group in the system.
//
func (p *SystemPermissionsService) AddClusterPermission() *SystemPermissionsServiceAddClusterPermissionRequest {
	return &SystemPermissionsServiceAddClusterPermissionRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the data center to the group in the system.
//
type SystemPermissionsServiceAddDataCenterPermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddDataCenterPermissionRequest) Header(key, value string) *SystemPermissionsServiceAddDataCenterPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddDataCenterPermissionRequest) Query(key, value string) *SystemPermissionsServiceAddDataCenterPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddDataCenterPermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddDataCenterPermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddDataCenterPermissionRequest) Send() (*SystemPermissionsServiceAddDataCenterPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/datacenterpermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddDataCenterPermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddDataCenterPermissionRequest) MustSend() *SystemPermissionsServiceAddDataCenterPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the data center to the group in the system.
//
type SystemPermissionsServiceAddDataCenterPermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddDataCenterPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddDataCenterPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the data center to the group in the system.
//
func (p *SystemPermissionsService) AddDataCenterPermission() *SystemPermissionsServiceAddDataCenterPermissionRequest {
	return &SystemPermissionsServiceAddDataCenterPermissionRequest{SystemPermissionsService: p}
}

//
// Add a new group level permission for a given virtual machine.
//
type SystemPermissionsServiceAddGroupLevelRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddGroupLevelRequest) Header(key, value string) *SystemPermissionsServiceAddGroupLevelRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddGroupLevelRequest) Query(key, value string) *SystemPermissionsServiceAddGroupLevelRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddGroupLevelRequest) Permission(permission *Permission) *SystemPermissionsServiceAddGroupLevelRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddGroupLevelRequest) Send() (*SystemPermissionsServiceAddGroupLevelResponse, error) {
	rawURL := fmt.Sprintf("%s%s/grouplevel", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddGroupLevelResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddGroupLevelRequest) MustSend() *SystemPermissionsServiceAddGroupLevelResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new group level permission for a given virtual machine.
//
type SystemPermissionsServiceAddGroupLevelResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddGroupLevelResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddGroupLevelResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new group level permission for a given virtual machine.
//
func (p *SystemPermissionsService) AddGroupLevel() *SystemPermissionsServiceAddGroupLevelRequest {
	return &SystemPermissionsServiceAddGroupLevelRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the host to the group in the system.
//
type SystemPermissionsServiceAddHostPermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddHostPermissionRequest) Header(key, value string) *SystemPermissionsServiceAddHostPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddHostPermissionRequest) Query(key, value string) *SystemPermissionsServiceAddHostPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddHostPermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddHostPermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddHostPermissionRequest) Send() (*SystemPermissionsServiceAddHostPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/hostpermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddHostPermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddHostPermissionRequest) MustSend() *SystemPermissionsServiceAddHostPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the host to the group in the system.
//
type SystemPermissionsServiceAddHostPermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddHostPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddHostPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the host to the group in the system.
//
func (p *SystemPermissionsService) AddHostPermission() *SystemPermissionsServiceAddHostPermissionRequest {
	return &SystemPermissionsServiceAddHostPermissionRequest{SystemPermissionsService: p}
}

//
// List all the permissions of the specific entity.
// For example to list all the permissions of the cluster with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/clusters/123/permissions
// ....
// [source,xml]
// ----
// <permissions>
//   <permission id="456">
//     <cluster id="123"/>
//     <role id="789"/>
//     <user id="451"/>
//   </permission>
//   <permission id="654">
//     <cluster id="123"/>
//     <role id="789"/>
//     <group id="127"/>
//   </permission>
// </permissions>
// ----
// The order of the returned permissions isn't guaranteed.
//
type SystemPermissionsServiceListRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
}

func (p *SystemPermissionsServiceListRequest) Header(key, value string) *SystemPermissionsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceListRequest) Query(key, value string) *SystemPermissionsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceListRequest) Follow(follow string) *SystemPermissionsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *SystemPermissionsServiceListRequest) Send() (*SystemPermissionsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLPermissionReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &SystemPermissionsServiceListResponse{permissions: result}, nil
}

func (p *SystemPermissionsServiceListRequest) MustSend() *SystemPermissionsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all the permissions of the specific entity.
// For example to list all the permissions of the cluster with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/clusters/123/permissions
// ....
// [source,xml]
// ----
// <permissions>
//   <permission id="456">
//     <cluster id="123"/>
//     <role id="789"/>
//     <user id="451"/>
//   </permission>
//   <permission id="654">
//     <cluster id="123"/>
//     <role id="789"/>
//     <group id="127"/>
//   </permission>
// </permissions>
// ----
// The order of the returned permissions isn't guaranteed.
//
type SystemPermissionsServiceListResponse struct {
	permissions *PermissionSlice
}

func (p *SystemPermissionsServiceListResponse) Permissions() (*PermissionSlice, bool) {
	if p.permissions != nil {
		return p.permissions, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceListResponse) MustPermissions() *PermissionSlice {
	if p.permissions == nil {
		panic("permissions in response does not exist")
	}
	return p.permissions
}

//
// List all the permissions of the specific entity.
// For example to list all the permissions of the cluster with id `123` send a request like this:
// ....
// GET /ovirt-engine/api/clusters/123/permissions
// ....
// [source,xml]
// ----
// <permissions>
//   <permission id="456">
//     <cluster id="123"/>
//     <role id="789"/>
//     <user id="451"/>
//   </permission>
//   <permission id="654">
//     <cluster id="123"/>
//     <role id="789"/>
//     <group id="127"/>
//   </permission>
// </permissions>
// ----
// The order of the returned permissions isn't guaranteed.
//
func (p *SystemPermissionsService) List() *SystemPermissionsServiceListRequest {
	return &SystemPermissionsServiceListRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the storage domain to the group in the system.
//
type SystemPermissionsServiceAddStorageDomainPermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionRequest) Header(key, value string) *SystemPermissionsServiceAddStorageDomainPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionRequest) Query(key, value string) *SystemPermissionsServiceAddStorageDomainPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddStorageDomainPermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionRequest) Send() (*SystemPermissionsServiceAddStorageDomainPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/storagedomainpermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddStorageDomainPermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionRequest) MustSend() *SystemPermissionsServiceAddStorageDomainPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the storage domain to the group in the system.
//
type SystemPermissionsServiceAddStorageDomainPermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddStorageDomainPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the storage domain to the group in the system.
//
func (p *SystemPermissionsService) AddStorageDomainPermission() *SystemPermissionsServiceAddStorageDomainPermissionRequest {
	return &SystemPermissionsServiceAddStorageDomainPermissionRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the template to the group in the system.
//
type SystemPermissionsServiceAddTemplatePermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddTemplatePermissionRequest) Header(key, value string) *SystemPermissionsServiceAddTemplatePermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddTemplatePermissionRequest) Query(key, value string) *SystemPermissionsServiceAddTemplatePermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddTemplatePermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddTemplatePermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddTemplatePermissionRequest) Send() (*SystemPermissionsServiceAddTemplatePermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/templatepermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddTemplatePermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddTemplatePermissionRequest) MustSend() *SystemPermissionsServiceAddTemplatePermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the template to the group in the system.
//
type SystemPermissionsServiceAddTemplatePermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddTemplatePermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddTemplatePermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the template to the group in the system.
//
func (p *SystemPermissionsService) AddTemplatePermission() *SystemPermissionsServiceAddTemplatePermissionRequest {
	return &SystemPermissionsServiceAddTemplatePermissionRequest{SystemPermissionsService: p}
}

//
// Add a new user level permission for a given virtual machine.
//
type SystemPermissionsServiceAddUserLevelRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddUserLevelRequest) Header(key, value string) *SystemPermissionsServiceAddUserLevelRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddUserLevelRequest) Query(key, value string) *SystemPermissionsServiceAddUserLevelRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddUserLevelRequest) Permission(permission *Permission) *SystemPermissionsServiceAddUserLevelRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddUserLevelRequest) Send() (*SystemPermissionsServiceAddUserLevelResponse, error) {
	rawURL := fmt.Sprintf("%s%s/userlevel", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddUserLevelResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddUserLevelRequest) MustSend() *SystemPermissionsServiceAddUserLevelResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new user level permission for a given virtual machine.
//
type SystemPermissionsServiceAddUserLevelResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddUserLevelResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddUserLevelResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new user level permission for a given virtual machine.
//
func (p *SystemPermissionsService) AddUserLevel() *SystemPermissionsServiceAddUserLevelRequest {
	return &SystemPermissionsServiceAddUserLevelRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the vm to the group in the system.
//
type SystemPermissionsServiceAddVmPermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddVmPermissionRequest) Header(key, value string) *SystemPermissionsServiceAddVmPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddVmPermissionRequest) Query(key, value string) *SystemPermissionsServiceAddVmPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddVmPermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddVmPermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddVmPermissionRequest) Send() (*SystemPermissionsServiceAddVmPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/vmpermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddVmPermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddVmPermissionRequest) MustSend() *SystemPermissionsServiceAddVmPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the vm to the group in the system.
//
type SystemPermissionsServiceAddVmPermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddVmPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddVmPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the vm to the group in the system.
//
func (p *SystemPermissionsService) AddVmPermission() *SystemPermissionsServiceAddVmPermissionRequest {
	return &SystemPermissionsServiceAddVmPermissionRequest{SystemPermissionsService: p}
}

//
// Add a new permission on the vm pool to the group in the system.
//
type SystemPermissionsServiceAddVmPoolPermissionRequest struct {
	SystemPermissionsService *SystemPermissionsService
	header                   map[string]string
	query                    map[string]string
	permission               *Permission
}

func (p *SystemPermissionsServiceAddVmPoolPermissionRequest) Header(key, value string) *SystemPermissionsServiceAddVmPoolPermissionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SystemPermissionsServiceAddVmPoolPermissionRequest) Query(key, value string) *SystemPermissionsServiceAddVmPoolPermissionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SystemPermissionsServiceAddVmPoolPermissionRequest) Permission(permission *Permission) *SystemPermissionsServiceAddVmPoolPermissionRequest {
	p.permission = permission
	return p
}

func (p *SystemPermissionsServiceAddVmPoolPermissionRequest) Send() (*SystemPermissionsServiceAddVmPoolPermissionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/vmpoolpermission", p.SystemPermissionsService.connection.URL(), p.SystemPermissionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Permission(p.permission)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SystemPermissionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SystemPermissionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SystemPermissionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SystemPermissionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SystemPermissionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustPermission()
	return &SystemPermissionsServiceAddVmPoolPermissionResponse{permission: result}, nil
}

func (p *SystemPermissionsServiceAddVmPoolPermissionRequest) MustSend() *SystemPermissionsServiceAddVmPoolPermissionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new permission on the vm pool to the group in the system.
//
type SystemPermissionsServiceAddVmPoolPermissionResponse struct {
	permission *Permission
}

func (p *SystemPermissionsServiceAddVmPoolPermissionResponse) Permission() (*Permission, bool) {
	if p.permission != nil {
		return p.permission, true
	}
	return nil, false
}

func (p *SystemPermissionsServiceAddVmPoolPermissionResponse) MustPermission() *Permission {
	if p.permission == nil {
		panic("permission in response does not exist")
	}
	return p.permission
}

//
// Add a new permission on the vm pool to the group in the system.
//
func (p *SystemPermissionsService) AddVmPoolPermission() *SystemPermissionsServiceAddVmPoolPermissionRequest {
	return &SystemPermissionsServiceAddVmPoolPermissionRequest{SystemPermissionsService: p}
}

//
// Sub-resource locator method, returns individual permission resource on which the remainder of the URI is
// dispatched.
//
func (op *SystemPermissionsService) PermissionService(id string) *PermissionService {
	return NewPermissionService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SystemPermissionsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.PermissionService(path), nil
	}
	return op.PermissionService(path[:index]).Service(path[index+1:])
}

func (op *SystemPermissionsService) String() string {
	return fmt.Sprintf("SystemPermissionsService:%s", op.path)
}

//
// Manages the set of disks available in a storage domain.
//
type StorageDomainContentDisksService struct {
	BaseService
}

func NewStorageDomainContentDisksService(connection *Connection, path string) *StorageDomainContentDisksService {
	var result StorageDomainContentDisksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of disks available in the storage domain.
// The order of the returned list of disks is guaranteed only if the `sortby` clause is included in
// the `search` parameter.
//
type StorageDomainContentDisksServiceListRequest struct {
	StorageDomainContentDisksService *StorageDomainContentDisksService
	header                           map[string]string
	query                            map[string]string
	caseSensitive                    *bool
	follow                           *string
	max                              *int64
	search                           *string
}

func (p *StorageDomainContentDisksServiceListRequest) Header(key, value string) *StorageDomainContentDisksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainContentDisksServiceListRequest) Query(key, value string) *StorageDomainContentDisksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainContentDisksServiceListRequest) CaseSensitive(caseSensitive bool) *StorageDomainContentDisksServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *StorageDomainContentDisksServiceListRequest) Follow(follow string) *StorageDomainContentDisksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *StorageDomainContentDisksServiceListRequest) Max(max int64) *StorageDomainContentDisksServiceListRequest {
	p.max = &max
	return p
}

func (p *StorageDomainContentDisksServiceListRequest) Search(search string) *StorageDomainContentDisksServiceListRequest {
	p.search = &search
	return p
}

func (p *StorageDomainContentDisksServiceListRequest) Send() (*StorageDomainContentDisksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainContentDisksService.connection.URL(), p.StorageDomainContentDisksService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainContentDisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainContentDisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainContentDisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainContentDisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainContentDisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &StorageDomainContentDisksServiceListResponse{disks: result}, nil
}

func (p *StorageDomainContentDisksServiceListRequest) MustSend() *StorageDomainContentDisksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of disks available in the storage domain.
// The order of the returned list of disks is guaranteed only if the `sortby` clause is included in
// the `search` parameter.
//
type StorageDomainContentDisksServiceListResponse struct {
	disks *DiskSlice
}

func (p *StorageDomainContentDisksServiceListResponse) Disks() (*DiskSlice, bool) {
	if p.disks != nil {
		return p.disks, true
	}
	return nil, false
}

func (p *StorageDomainContentDisksServiceListResponse) MustDisks() *DiskSlice {
	if p.disks == nil {
		panic("disks in response does not exist")
	}
	return p.disks
}

//
// Returns the list of disks available in the storage domain.
// The order of the returned list of disks is guaranteed only if the `sortby` clause is included in
// the `search` parameter.
//
func (p *StorageDomainContentDisksService) List() *StorageDomainContentDisksServiceListRequest {
	return &StorageDomainContentDisksServiceListRequest{StorageDomainContentDisksService: p}
}

//
//
func (op *StorageDomainContentDisksService) DiskService(id string) *StorageDomainContentDiskService {
	return NewStorageDomainContentDiskService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageDomainContentDisksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DiskService(path), nil
	}
	return op.DiskService(path[:index]).Service(path[index+1:])
}

func (op *StorageDomainContentDisksService) String() string {
	return fmt.Sprintf("StorageDomainContentDisksService:%s", op.path)
}

//
// Provides information about the cluster features that are supported by a cluster level.
//
type ClusterFeaturesService struct {
	BaseService
}

func NewClusterFeaturesService(connection *Connection, path string) *ClusterFeaturesService {
	var result ClusterFeaturesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Lists the cluster features supported by the cluster level.
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels/4.1/clusterfeatures
// ----
// This will return a list of cluster features supported by the cluster level:
// [source,xml]
// ----
// <cluster_features>
//   <cluster_feature id="123">
//      <name>test_feature</name>
//   </cluster_feature>
//   ...
// </cluster_features>
// ----
//
type ClusterFeaturesServiceListRequest struct {
	ClusterFeaturesService *ClusterFeaturesService
	header                 map[string]string
	query                  map[string]string
	follow                 *string
}

func (p *ClusterFeaturesServiceListRequest) Header(key, value string) *ClusterFeaturesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClusterFeaturesServiceListRequest) Query(key, value string) *ClusterFeaturesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClusterFeaturesServiceListRequest) Follow(follow string) *ClusterFeaturesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ClusterFeaturesServiceListRequest) Send() (*ClusterFeaturesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClusterFeaturesService.connection.URL(), p.ClusterFeaturesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClusterFeaturesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClusterFeaturesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClusterFeaturesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClusterFeaturesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClusterFeaturesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterFeatureReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ClusterFeaturesServiceListResponse{features: result}, nil
}

func (p *ClusterFeaturesServiceListRequest) MustSend() *ClusterFeaturesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists the cluster features supported by the cluster level.
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels/4.1/clusterfeatures
// ----
// This will return a list of cluster features supported by the cluster level:
// [source,xml]
// ----
// <cluster_features>
//   <cluster_feature id="123">
//      <name>test_feature</name>
//   </cluster_feature>
//   ...
// </cluster_features>
// ----
//
type ClusterFeaturesServiceListResponse struct {
	features *ClusterFeatureSlice
}

func (p *ClusterFeaturesServiceListResponse) Features() (*ClusterFeatureSlice, bool) {
	if p.features != nil {
		return p.features, true
	}
	return nil, false
}

func (p *ClusterFeaturesServiceListResponse) MustFeatures() *ClusterFeatureSlice {
	if p.features == nil {
		panic("features in response does not exist")
	}
	return p.features
}

//
// Lists the cluster features supported by the cluster level.
// [source]
// ----
// GET /ovirt-engine/api/clusterlevels/4.1/clusterfeatures
// ----
// This will return a list of cluster features supported by the cluster level:
// [source,xml]
// ----
// <cluster_features>
//   <cluster_feature id="123">
//      <name>test_feature</name>
//   </cluster_feature>
//   ...
// </cluster_features>
// ----
//
func (p *ClusterFeaturesService) List() *ClusterFeaturesServiceListRequest {
	return &ClusterFeaturesServiceListRequest{ClusterFeaturesService: p}
}

//
// Reference to the service that provides information about a specific feature.
//
func (op *ClusterFeaturesService) FeatureService(id string) *ClusterFeatureService {
	return NewClusterFeatureService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClusterFeaturesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.FeatureService(path), nil
	}
	return op.FeatureService(path[:index]).Service(path[index+1:])
}

func (op *ClusterFeaturesService) String() string {
	return fmt.Sprintf("ClusterFeaturesService:%s", op.path)
}

//
//
type SchedulingPolicyUnitService struct {
	BaseService
}

func NewSchedulingPolicyUnitService(connection *Connection, path string) *SchedulingPolicyUnitService {
	var result SchedulingPolicyUnitService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type SchedulingPolicyUnitServiceGetRequest struct {
	SchedulingPolicyUnitService *SchedulingPolicyUnitService
	header                      map[string]string
	query                       map[string]string
	filter                      *bool
	follow                      *string
}

func (p *SchedulingPolicyUnitServiceGetRequest) Header(key, value string) *SchedulingPolicyUnitServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPolicyUnitServiceGetRequest) Query(key, value string) *SchedulingPolicyUnitServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPolicyUnitServiceGetRequest) Filter(filter bool) *SchedulingPolicyUnitServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *SchedulingPolicyUnitServiceGetRequest) Follow(follow string) *SchedulingPolicyUnitServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *SchedulingPolicyUnitServiceGetRequest) Send() (*SchedulingPolicyUnitServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPolicyUnitService.connection.URL(), p.SchedulingPolicyUnitService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPolicyUnitService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPolicyUnitService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPolicyUnitService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPolicyUnitService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPolicyUnitService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSchedulingPolicyUnitReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SchedulingPolicyUnitServiceGetResponse{unit: result}, nil
}

func (p *SchedulingPolicyUnitServiceGetRequest) MustSend() *SchedulingPolicyUnitServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SchedulingPolicyUnitServiceGetResponse struct {
	unit *SchedulingPolicyUnit
}

func (p *SchedulingPolicyUnitServiceGetResponse) Unit() (*SchedulingPolicyUnit, bool) {
	if p.unit != nil {
		return p.unit, true
	}
	return nil, false
}

func (p *SchedulingPolicyUnitServiceGetResponse) MustUnit() *SchedulingPolicyUnit {
	if p.unit == nil {
		panic("unit in response does not exist")
	}
	return p.unit
}

//
//
func (p *SchedulingPolicyUnitService) Get() *SchedulingPolicyUnitServiceGetRequest {
	return &SchedulingPolicyUnitServiceGetRequest{SchedulingPolicyUnitService: p}
}

//
//
type SchedulingPolicyUnitServiceRemoveRequest struct {
	SchedulingPolicyUnitService *SchedulingPolicyUnitService
	header                      map[string]string
	query                       map[string]string
	async                       *bool
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) Header(key, value string) *SchedulingPolicyUnitServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) Query(key, value string) *SchedulingPolicyUnitServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) Async(async bool) *SchedulingPolicyUnitServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) Send() (*SchedulingPolicyUnitServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPolicyUnitService.connection.URL(), p.SchedulingPolicyUnitService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPolicyUnitService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPolicyUnitService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPolicyUnitService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPolicyUnitService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPolicyUnitService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(SchedulingPolicyUnitServiceRemoveResponse), nil
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) MustSend() *SchedulingPolicyUnitServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SchedulingPolicyUnitServiceRemoveResponse struct {
}

//
//
func (p *SchedulingPolicyUnitService) Remove() *SchedulingPolicyUnitServiceRemoveRequest {
	return &SchedulingPolicyUnitServiceRemoveRequest{SchedulingPolicyUnitService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SchedulingPolicyUnitService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *SchedulingPolicyUnitService) String() string {
	return fmt.Sprintf("SchedulingPolicyUnitService:%s", op.path)
}

//
//
type StorageServerConnectionsService struct {
	BaseService
}

func NewStorageServerConnectionsService(connection *Connection, path string) *StorageServerConnectionsService {
	var result StorageServerConnectionsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new storage connection.
// For example, to create a new storage connection for the NFS server `mynfs.example.com` and NFS share
// `/export/mydata` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageconnections
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_connection>
//   <type>nfs</type>
//   <address>mynfs.example.com</address>
//   <path>/export/mydata</path>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_connection>
// ----
//
type StorageServerConnectionsServiceAddRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	connection                      *StorageConnection
}

func (p *StorageServerConnectionsServiceAddRequest) Header(key, value string) *StorageServerConnectionsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddRequest) Query(key, value string) *StorageServerConnectionsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddRequest) Connection(connection *StorageConnection) *StorageServerConnectionsServiceAddRequest {
	p.connection = connection
	return p
}

func (p *StorageServerConnectionsServiceAddRequest) Send() (*StorageServerConnectionsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLStorageConnectionWriteOne(writer, p.connection, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageConnectionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageServerConnectionsServiceAddResponse{connection: result}, nil
}

func (p *StorageServerConnectionsServiceAddRequest) MustSend() *StorageServerConnectionsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new storage connection.
// For example, to create a new storage connection for the NFS server `mynfs.example.com` and NFS share
// `/export/mydata` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageconnections
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_connection>
//   <type>nfs</type>
//   <address>mynfs.example.com</address>
//   <path>/export/mydata</path>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_connection>
// ----
//
type StorageServerConnectionsServiceAddResponse struct {
	connection *StorageConnection
}

func (p *StorageServerConnectionsServiceAddResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceAddResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
// Creates a new storage connection.
// For example, to create a new storage connection for the NFS server `mynfs.example.com` and NFS share
// `/export/mydata` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageconnections
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_connection>
//   <type>nfs</type>
//   <address>mynfs.example.com</address>
//   <path>/export/mydata</path>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_connection>
// ----
//
func (p *StorageServerConnectionsService) Add() *StorageServerConnectionsServiceAddRequest {
	return &StorageServerConnectionsServiceAddRequest{StorageServerConnectionsService: p}
}

//
// Add a Glusterfs storage connection to the system.
//
type StorageServerConnectionsServiceAddGlusterfsRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	connection                      *StorageConnection
}

func (p *StorageServerConnectionsServiceAddGlusterfsRequest) Header(key, value string) *StorageServerConnectionsServiceAddGlusterfsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddGlusterfsRequest) Query(key, value string) *StorageServerConnectionsServiceAddGlusterfsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddGlusterfsRequest) Connection(connection *StorageConnection) *StorageServerConnectionsServiceAddGlusterfsRequest {
	p.connection = connection
	return p
}

func (p *StorageServerConnectionsServiceAddGlusterfsRequest) Send() (*StorageServerConnectionsServiceAddGlusterfsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/glusterfs", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Connection(p.connection)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustConnection()
	return &StorageServerConnectionsServiceAddGlusterfsResponse{connection: result}, nil
}

func (p *StorageServerConnectionsServiceAddGlusterfsRequest) MustSend() *StorageServerConnectionsServiceAddGlusterfsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a Glusterfs storage connection to the system.
//
type StorageServerConnectionsServiceAddGlusterfsResponse struct {
	connection *StorageConnection
}

func (p *StorageServerConnectionsServiceAddGlusterfsResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceAddGlusterfsResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
// Add a Glusterfs storage connection to the system.
//
func (p *StorageServerConnectionsService) AddGlusterfs() *StorageServerConnectionsServiceAddGlusterfsRequest {
	return &StorageServerConnectionsServiceAddGlusterfsRequest{StorageServerConnectionsService: p}
}

//
// Add a iSCSI storage connection to the system.
//
type StorageServerConnectionsServiceAddIscsiRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	connection                      *StorageConnection
}

func (p *StorageServerConnectionsServiceAddIscsiRequest) Header(key, value string) *StorageServerConnectionsServiceAddIscsiRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddIscsiRequest) Query(key, value string) *StorageServerConnectionsServiceAddIscsiRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddIscsiRequest) Connection(connection *StorageConnection) *StorageServerConnectionsServiceAddIscsiRequest {
	p.connection = connection
	return p
}

func (p *StorageServerConnectionsServiceAddIscsiRequest) Send() (*StorageServerConnectionsServiceAddIscsiResponse, error) {
	rawURL := fmt.Sprintf("%s%s/iscsi", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Connection(p.connection)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustConnection()
	return &StorageServerConnectionsServiceAddIscsiResponse{connection: result}, nil
}

func (p *StorageServerConnectionsServiceAddIscsiRequest) MustSend() *StorageServerConnectionsServiceAddIscsiResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a iSCSI storage connection to the system.
//
type StorageServerConnectionsServiceAddIscsiResponse struct {
	connection *StorageConnection
}

func (p *StorageServerConnectionsServiceAddIscsiResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceAddIscsiResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
// Add a iSCSI storage connection to the system.
//
func (p *StorageServerConnectionsService) AddIscsi() *StorageServerConnectionsServiceAddIscsiRequest {
	return &StorageServerConnectionsServiceAddIscsiRequest{StorageServerConnectionsService: p}
}

//
// Returns the list of storage connections.
// The order of the returned list of connections isn't guaranteed.
//
type StorageServerConnectionsServiceListRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	follow                          *string
	max                             *int64
}

func (p *StorageServerConnectionsServiceListRequest) Header(key, value string) *StorageServerConnectionsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceListRequest) Query(key, value string) *StorageServerConnectionsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceListRequest) Follow(follow string) *StorageServerConnectionsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *StorageServerConnectionsServiceListRequest) Max(max int64) *StorageServerConnectionsServiceListRequest {
	p.max = &max
	return p
}

func (p *StorageServerConnectionsServiceListRequest) Send() (*StorageServerConnectionsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageConnectionReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &StorageServerConnectionsServiceListResponse{connections: result}, nil
}

func (p *StorageServerConnectionsServiceListRequest) MustSend() *StorageServerConnectionsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of storage connections.
// The order of the returned list of connections isn't guaranteed.
//
type StorageServerConnectionsServiceListResponse struct {
	connections *StorageConnectionSlice
}

func (p *StorageServerConnectionsServiceListResponse) Connections() (*StorageConnectionSlice, bool) {
	if p.connections != nil {
		return p.connections, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceListResponse) MustConnections() *StorageConnectionSlice {
	if p.connections == nil {
		panic("connections in response does not exist")
	}
	return p.connections
}

//
// Returns the list of storage connections.
// The order of the returned list of connections isn't guaranteed.
//
func (p *StorageServerConnectionsService) List() *StorageServerConnectionsServiceListRequest {
	return &StorageServerConnectionsServiceListRequest{StorageServerConnectionsService: p}
}

//
// Add a local storage connection to the system.
//
type StorageServerConnectionsServiceAddLocalRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	connection                      *StorageConnection
}

func (p *StorageServerConnectionsServiceAddLocalRequest) Header(key, value string) *StorageServerConnectionsServiceAddLocalRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddLocalRequest) Query(key, value string) *StorageServerConnectionsServiceAddLocalRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddLocalRequest) Connection(connection *StorageConnection) *StorageServerConnectionsServiceAddLocalRequest {
	p.connection = connection
	return p
}

func (p *StorageServerConnectionsServiceAddLocalRequest) Send() (*StorageServerConnectionsServiceAddLocalResponse, error) {
	rawURL := fmt.Sprintf("%s%s/local", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Connection(p.connection)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustConnection()
	return &StorageServerConnectionsServiceAddLocalResponse{connection: result}, nil
}

func (p *StorageServerConnectionsServiceAddLocalRequest) MustSend() *StorageServerConnectionsServiceAddLocalResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a local storage connection to the system.
//
type StorageServerConnectionsServiceAddLocalResponse struct {
	connection *StorageConnection
}

func (p *StorageServerConnectionsServiceAddLocalResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceAddLocalResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
// Add a local storage connection to the system.
//
func (p *StorageServerConnectionsService) AddLocal() *StorageServerConnectionsServiceAddLocalRequest {
	return &StorageServerConnectionsServiceAddLocalRequest{StorageServerConnectionsService: p}
}

//
// Add a nfs storage connection to the system.
//
type StorageServerConnectionsServiceAddNfsRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	connection                      *StorageConnection
}

func (p *StorageServerConnectionsServiceAddNfsRequest) Header(key, value string) *StorageServerConnectionsServiceAddNfsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddNfsRequest) Query(key, value string) *StorageServerConnectionsServiceAddNfsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddNfsRequest) Connection(connection *StorageConnection) *StorageServerConnectionsServiceAddNfsRequest {
	p.connection = connection
	return p
}

func (p *StorageServerConnectionsServiceAddNfsRequest) Send() (*StorageServerConnectionsServiceAddNfsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/nfs", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Connection(p.connection)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustConnection()
	return &StorageServerConnectionsServiceAddNfsResponse{connection: result}, nil
}

func (p *StorageServerConnectionsServiceAddNfsRequest) MustSend() *StorageServerConnectionsServiceAddNfsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a nfs storage connection to the system.
//
type StorageServerConnectionsServiceAddNfsResponse struct {
	connection *StorageConnection
}

func (p *StorageServerConnectionsServiceAddNfsResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceAddNfsResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
// Add a nfs storage connection to the system.
//
func (p *StorageServerConnectionsService) AddNfs() *StorageServerConnectionsServiceAddNfsRequest {
	return &StorageServerConnectionsServiceAddNfsRequest{StorageServerConnectionsService: p}
}

//
// Add a vfs storage connection to the system.
//
type StorageServerConnectionsServiceAddVfsRequest struct {
	StorageServerConnectionsService *StorageServerConnectionsService
	header                          map[string]string
	query                           map[string]string
	connection                      *StorageConnection
}

func (p *StorageServerConnectionsServiceAddVfsRequest) Header(key, value string) *StorageServerConnectionsServiceAddVfsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddVfsRequest) Query(key, value string) *StorageServerConnectionsServiceAddVfsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServerConnectionsServiceAddVfsRequest) Connection(connection *StorageConnection) *StorageServerConnectionsServiceAddVfsRequest {
	p.connection = connection
	return p
}

func (p *StorageServerConnectionsServiceAddVfsRequest) Send() (*StorageServerConnectionsServiceAddVfsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/vfs", p.StorageServerConnectionsService.connection.URL(), p.StorageServerConnectionsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Connection(p.connection)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustConnection()
	return &StorageServerConnectionsServiceAddVfsResponse{connection: result}, nil
}

func (p *StorageServerConnectionsServiceAddVfsRequest) MustSend() *StorageServerConnectionsServiceAddVfsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a vfs storage connection to the system.
//
type StorageServerConnectionsServiceAddVfsResponse struct {
	connection *StorageConnection
}

func (p *StorageServerConnectionsServiceAddVfsResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageServerConnectionsServiceAddVfsResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
// Add a vfs storage connection to the system.
//
func (p *StorageServerConnectionsService) AddVfs() *StorageServerConnectionsServiceAddVfsRequest {
	return &StorageServerConnectionsServiceAddVfsRequest{StorageServerConnectionsService: p}
}

//
//
func (op *StorageServerConnectionsService) StorageConnectionService(id string) *StorageServerConnectionService {
	return NewStorageServerConnectionService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageServerConnectionsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.StorageConnectionService(path), nil
	}
	return op.StorageConnectionService(path[:index]).Service(path[index+1:])
}

func (op *StorageServerConnectionsService) String() string {
	return fmt.Sprintf("StorageServerConnectionsService:%s", op.path)
}

//
// Manages the collection of disks available in the system.
//
type DisksService struct {
	BaseService
}

func NewDisksService(connection *Connection, path string) *DisksService {
	var result DisksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a new floating disk.
// There are three types of disks that can be added - disk image, direct LUN and
//  https://wiki.openstack.org/wiki/Cinder[Cinder] disk.
// *Adding a new image disk:*
// When creating a new floating image <<types/disk,Disk>>, the API requires the `storage_domain`, `provisioned_size`
// and `format` attributes.
// Note that block storage domains (i.e., storage domains with the <<types/storage_type, storage type>> of iSCSI or
// FCP) don't support the combination of the raw `format` with `sparse=true`, so `sparse=false` must be stated
// explicitly.
// To create a new floating image disk with specified `provisioned_size`, `format` and `name` on a storage domain
// with an id `123`, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
// *Adding a new direct LUN disk:*
// When adding a new floating direct LUN via the API, there are two flavors that can be used:
// . With a `host` element - in this case, the host is used for sanity checks (e.g., that the LUN is visible) and
// to retrieve basic information about the LUN (e.g., size and serial).
// . Without a `host` element - in this case, the operation is a database-only operation, and the storage is never
// accessed.
// To create a new floating direct LUN disk with a `host` element with an id `123`, specified `alias`, `type` and
// `logical_unit` with an id `456` (that has the attributes `address`, `port` and `target`),
// send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <alias>mylun</alias>
//   <lun_storage>
//     <host id="123"/>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="456">
//         <address>10.35.10.20</address>
//         <port>3260</port>
//         <target>iqn.2017-01.com.myhost:444</target>
//       </logical_unit>
//     </logical_units>
//   </lun_storage>
// </disk>
// ----
// To create a new floating direct LUN disk without using a host, remove the `host` element.
// *Adding a new Cinder disk:*
// To create a new floating Cinder disk, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <openstack_volume_type>
//     <name>myceph</name>
//   </openstack_volume_type>
//   <storage_domains>
//     <storage_domain>
//       <name>cinderDomain</name>
//     </storage_domain>
//   </storage_domains>
//   <provisioned_size>1073741824</provisioned_size>
//   <interface>virtio</interface>
//   <format>raw</format>
// </disk>
// ----
// *Adding a floating disks in order to upload disk snapshots:*
// Since version 4.2 of the engine it is possible to upload disks with
// snapshots. This request should be used to create the base image of the
// images chain (The consecutive disk snapshots (images), should be created
// using `disk-attachments` element when creating a snapshot).
// The disk has to be created with the same disk identifier and image identifier
// of the uploaded image. I.e. the identifiers should be saved as part of the
// backup process. The image identifier can be also fetched using the
// `qemu-img info` command. For example, if the disk image is stored into
// a file named `b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img`:
// [source,shell]
// ----
// $ qemu-img info b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img
// image: b548366b-fb51-4b41-97be-733c887fe305
// file format: qcow2
// virtual size: 1.0G (1073741824 bytes)
// disk size: 196K
// cluster_size: 65536
// backing file: ad58716a-1fe9-481f-815e-664de1df04eb
// backing file format: raw
// ----
// To create a disk with with the disk identifier and image identifier obtained
// with the `qemu-img info` command shown above, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk id="b7a4c6c5-443b-47c5-967f-6abc79675e8b">
//   <image_id>b548366b-fb51-4b41-97be-733c887fe305</image_id>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
//
type DisksServiceAddRequest struct {
	DisksService *DisksService
	header       map[string]string
	query        map[string]string
	disk         *Disk
}

func (p *DisksServiceAddRequest) Header(key, value string) *DisksServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DisksServiceAddRequest) Query(key, value string) *DisksServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DisksServiceAddRequest) Disk(disk *Disk) *DisksServiceAddRequest {
	p.disk = disk
	return p
}

func (p *DisksServiceAddRequest) Send() (*DisksServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DisksService.connection.URL(), p.DisksService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskWriteOne(writer, p.disk, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DisksServiceAddResponse{disk: result}, nil
}

func (p *DisksServiceAddRequest) MustSend() *DisksServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a new floating disk.
// There are three types of disks that can be added - disk image, direct LUN and
//  https://wiki.openstack.org/wiki/Cinder[Cinder] disk.
// *Adding a new image disk:*
// When creating a new floating image <<types/disk,Disk>>, the API requires the `storage_domain`, `provisioned_size`
// and `format` attributes.
// Note that block storage domains (i.e., storage domains with the <<types/storage_type, storage type>> of iSCSI or
// FCP) don't support the combination of the raw `format` with `sparse=true`, so `sparse=false` must be stated
// explicitly.
// To create a new floating image disk with specified `provisioned_size`, `format` and `name` on a storage domain
// with an id `123`, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
// *Adding a new direct LUN disk:*
// When adding a new floating direct LUN via the API, there are two flavors that can be used:
// . With a `host` element - in this case, the host is used for sanity checks (e.g., that the LUN is visible) and
// to retrieve basic information about the LUN (e.g., size and serial).
// . Without a `host` element - in this case, the operation is a database-only operation, and the storage is never
// accessed.
// To create a new floating direct LUN disk with a `host` element with an id `123`, specified `alias`, `type` and
// `logical_unit` with an id `456` (that has the attributes `address`, `port` and `target`),
// send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <alias>mylun</alias>
//   <lun_storage>
//     <host id="123"/>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="456">
//         <address>10.35.10.20</address>
//         <port>3260</port>
//         <target>iqn.2017-01.com.myhost:444</target>
//       </logical_unit>
//     </logical_units>
//   </lun_storage>
// </disk>
// ----
// To create a new floating direct LUN disk without using a host, remove the `host` element.
// *Adding a new Cinder disk:*
// To create a new floating Cinder disk, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <openstack_volume_type>
//     <name>myceph</name>
//   </openstack_volume_type>
//   <storage_domains>
//     <storage_domain>
//       <name>cinderDomain</name>
//     </storage_domain>
//   </storage_domains>
//   <provisioned_size>1073741824</provisioned_size>
//   <interface>virtio</interface>
//   <format>raw</format>
// </disk>
// ----
// *Adding a floating disks in order to upload disk snapshots:*
// Since version 4.2 of the engine it is possible to upload disks with
// snapshots. This request should be used to create the base image of the
// images chain (The consecutive disk snapshots (images), should be created
// using `disk-attachments` element when creating a snapshot).
// The disk has to be created with the same disk identifier and image identifier
// of the uploaded image. I.e. the identifiers should be saved as part of the
// backup process. The image identifier can be also fetched using the
// `qemu-img info` command. For example, if the disk image is stored into
// a file named `b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img`:
// [source,shell]
// ----
// $ qemu-img info b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img
// image: b548366b-fb51-4b41-97be-733c887fe305
// file format: qcow2
// virtual size: 1.0G (1073741824 bytes)
// disk size: 196K
// cluster_size: 65536
// backing file: ad58716a-1fe9-481f-815e-664de1df04eb
// backing file format: raw
// ----
// To create a disk with with the disk identifier and image identifier obtained
// with the `qemu-img info` command shown above, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk id="b7a4c6c5-443b-47c5-967f-6abc79675e8b">
//   <image_id>b548366b-fb51-4b41-97be-733c887fe305</image_id>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
//
type DisksServiceAddResponse struct {
	disk *Disk
}

func (p *DisksServiceAddResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *DisksServiceAddResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Adds a new floating disk.
// There are three types of disks that can be added - disk image, direct LUN and
//  https://wiki.openstack.org/wiki/Cinder[Cinder] disk.
// *Adding a new image disk:*
// When creating a new floating image <<types/disk,Disk>>, the API requires the `storage_domain`, `provisioned_size`
// and `format` attributes.
// Note that block storage domains (i.e., storage domains with the <<types/storage_type, storage type>> of iSCSI or
// FCP) don't support the combination of the raw `format` with `sparse=true`, so `sparse=false` must be stated
// explicitly.
// To create a new floating image disk with specified `provisioned_size`, `format` and `name` on a storage domain
// with an id `123`, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
// *Adding a new direct LUN disk:*
// When adding a new floating direct LUN via the API, there are two flavors that can be used:
// . With a `host` element - in this case, the host is used for sanity checks (e.g., that the LUN is visible) and
// to retrieve basic information about the LUN (e.g., size and serial).
// . Without a `host` element - in this case, the operation is a database-only operation, and the storage is never
// accessed.
// To create a new floating direct LUN disk with a `host` element with an id `123`, specified `alias`, `type` and
// `logical_unit` with an id `456` (that has the attributes `address`, `port` and `target`),
// send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <alias>mylun</alias>
//   <lun_storage>
//     <host id="123"/>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="456">
//         <address>10.35.10.20</address>
//         <port>3260</port>
//         <target>iqn.2017-01.com.myhost:444</target>
//       </logical_unit>
//     </logical_units>
//   </lun_storage>
// </disk>
// ----
// To create a new floating direct LUN disk without using a host, remove the `host` element.
// *Adding a new Cinder disk:*
// To create a new floating Cinder disk, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <openstack_volume_type>
//     <name>myceph</name>
//   </openstack_volume_type>
//   <storage_domains>
//     <storage_domain>
//       <name>cinderDomain</name>
//     </storage_domain>
//   </storage_domains>
//   <provisioned_size>1073741824</provisioned_size>
//   <interface>virtio</interface>
//   <format>raw</format>
// </disk>
// ----
// *Adding a floating disks in order to upload disk snapshots:*
// Since version 4.2 of the engine it is possible to upload disks with
// snapshots. This request should be used to create the base image of the
// images chain (The consecutive disk snapshots (images), should be created
// using `disk-attachments` element when creating a snapshot).
// The disk has to be created with the same disk identifier and image identifier
// of the uploaded image. I.e. the identifiers should be saved as part of the
// backup process. The image identifier can be also fetched using the
// `qemu-img info` command. For example, if the disk image is stored into
// a file named `b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img`:
// [source,shell]
// ----
// $ qemu-img info b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img
// image: b548366b-fb51-4b41-97be-733c887fe305
// file format: qcow2
// virtual size: 1.0G (1073741824 bytes)
// disk size: 196K
// cluster_size: 65536
// backing file: ad58716a-1fe9-481f-815e-664de1df04eb
// backing file format: raw
// ----
// To create a disk with with the disk identifier and image identifier obtained
// with the `qemu-img info` command shown above, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk id="b7a4c6c5-443b-47c5-967f-6abc79675e8b">
//   <image_id>b548366b-fb51-4b41-97be-733c887fe305</image_id>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
//
func (p *DisksService) Add() *DisksServiceAddRequest {
	return &DisksServiceAddRequest{DisksService: p}
}

//
// Get list of disks.
// [source]
// ----
// GET /ovirt-engine/api/disks
// ----
// You will get a XML response which will look like this one:
// [source,xml]
// ----
// <disks>
//   <disk id="123">
//     <actions>...</actions>
//     <name>MyDisk</name>
//     <description>MyDisk description</description>
//     <link href="/ovirt-engine/api/disks/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/disks/123/statistics" rel="statistics"/>
//     <actual_size>5345845248</actual_size>
//     <alias>MyDisk alias</alias>
//     ...
//     <status>ok</status>
//     <storage_type>image</storage_type>
//     <wipe_after_delete>false</wipe_after_delete>
//     <disk_profile id="123"/>
//     <quota id="123"/>
//     <storage_domains>...</storage_domains>
//   </disk>
//   ...
// </disks>
// ----
// The order of the returned list of disks is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type DisksServiceListRequest struct {
	DisksService  *DisksService
	header        map[string]string
	query         map[string]string
	caseSensitive *bool
	follow        *string
	max           *int64
	search        *string
}

func (p *DisksServiceListRequest) Header(key, value string) *DisksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DisksServiceListRequest) Query(key, value string) *DisksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DisksServiceListRequest) CaseSensitive(caseSensitive bool) *DisksServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *DisksServiceListRequest) Follow(follow string) *DisksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DisksServiceListRequest) Max(max int64) *DisksServiceListRequest {
	p.max = &max
	return p
}

func (p *DisksServiceListRequest) Search(search string) *DisksServiceListRequest {
	p.search = &search
	return p
}

func (p *DisksServiceListRequest) Send() (*DisksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DisksService.connection.URL(), p.DisksService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DisksServiceListResponse{disks: result}, nil
}

func (p *DisksServiceListRequest) MustSend() *DisksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get list of disks.
// [source]
// ----
// GET /ovirt-engine/api/disks
// ----
// You will get a XML response which will look like this one:
// [source,xml]
// ----
// <disks>
//   <disk id="123">
//     <actions>...</actions>
//     <name>MyDisk</name>
//     <description>MyDisk description</description>
//     <link href="/ovirt-engine/api/disks/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/disks/123/statistics" rel="statistics"/>
//     <actual_size>5345845248</actual_size>
//     <alias>MyDisk alias</alias>
//     ...
//     <status>ok</status>
//     <storage_type>image</storage_type>
//     <wipe_after_delete>false</wipe_after_delete>
//     <disk_profile id="123"/>
//     <quota id="123"/>
//     <storage_domains>...</storage_domains>
//   </disk>
//   ...
// </disks>
// ----
// The order of the returned list of disks is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type DisksServiceListResponse struct {
	disks *DiskSlice
}

func (p *DisksServiceListResponse) Disks() (*DiskSlice, bool) {
	if p.disks != nil {
		return p.disks, true
	}
	return nil, false
}

func (p *DisksServiceListResponse) MustDisks() *DiskSlice {
	if p.disks == nil {
		panic("disks in response does not exist")
	}
	return p.disks
}

//
// Get list of disks.
// [source]
// ----
// GET /ovirt-engine/api/disks
// ----
// You will get a XML response which will look like this one:
// [source,xml]
// ----
// <disks>
//   <disk id="123">
//     <actions>...</actions>
//     <name>MyDisk</name>
//     <description>MyDisk description</description>
//     <link href="/ovirt-engine/api/disks/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/disks/123/statistics" rel="statistics"/>
//     <actual_size>5345845248</actual_size>
//     <alias>MyDisk alias</alias>
//     ...
//     <status>ok</status>
//     <storage_type>image</storage_type>
//     <wipe_after_delete>false</wipe_after_delete>
//     <disk_profile id="123"/>
//     <quota id="123"/>
//     <storage_domains>...</storage_domains>
//   </disk>
//   ...
// </disks>
// ----
// The order of the returned list of disks is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
func (p *DisksService) List() *DisksServiceListRequest {
	return &DisksServiceListRequest{DisksService: p}
}

//
// Add a new lun disk to the storage domain.
//
type DisksServiceAddLunRequest struct {
	DisksService *DisksService
	header       map[string]string
	query        map[string]string
	disk         *Disk
}

func (p *DisksServiceAddLunRequest) Header(key, value string) *DisksServiceAddLunRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DisksServiceAddLunRequest) Query(key, value string) *DisksServiceAddLunRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DisksServiceAddLunRequest) Disk(disk *Disk) *DisksServiceAddLunRequest {
	p.disk = disk
	return p
}

func (p *DisksServiceAddLunRequest) Send() (*DisksServiceAddLunResponse, error) {
	rawURL := fmt.Sprintf("%s%s/lun", p.DisksService.connection.URL(), p.DisksService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Disk(p.disk)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustDisk()
	return &DisksServiceAddLunResponse{disk: result}, nil
}

func (p *DisksServiceAddLunRequest) MustSend() *DisksServiceAddLunResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new lun disk to the storage domain.
//
type DisksServiceAddLunResponse struct {
	disk *Disk
}

func (p *DisksServiceAddLunResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *DisksServiceAddLunResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Add a new lun disk to the storage domain.
//
func (p *DisksService) AddLun() *DisksServiceAddLunRequest {
	return &DisksServiceAddLunRequest{DisksService: p}
}

//
// Add a new disk to the storage domain with the specified size allocating space from the storage domain.
//
type DisksServiceAddOnStorageDomainRequest struct {
	DisksService *DisksService
	header       map[string]string
	query        map[string]string
	disk         *Disk
}

func (p *DisksServiceAddOnStorageDomainRequest) Header(key, value string) *DisksServiceAddOnStorageDomainRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DisksServiceAddOnStorageDomainRequest) Query(key, value string) *DisksServiceAddOnStorageDomainRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DisksServiceAddOnStorageDomainRequest) Disk(disk *Disk) *DisksServiceAddOnStorageDomainRequest {
	p.disk = disk
	return p
}

func (p *DisksServiceAddOnStorageDomainRequest) Send() (*DisksServiceAddOnStorageDomainResponse, error) {
	rawURL := fmt.Sprintf("%s%s/onstoragedomain", p.DisksService.connection.URL(), p.DisksService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Disk(p.disk)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustDisk()
	return &DisksServiceAddOnStorageDomainResponse{disk: result}, nil
}

func (p *DisksServiceAddOnStorageDomainRequest) MustSend() *DisksServiceAddOnStorageDomainResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new disk to the storage domain with the specified size allocating space from the storage domain.
//
type DisksServiceAddOnStorageDomainResponse struct {
	disk *Disk
}

func (p *DisksServiceAddOnStorageDomainResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *DisksServiceAddOnStorageDomainResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Add a new disk to the storage domain with the specified size allocating space from the storage domain.
//
func (p *DisksService) AddOnStorageDomain() *DisksServiceAddOnStorageDomainRequest {
	return &DisksServiceAddOnStorageDomainRequest{DisksService: p}
}

//
// Reference to a service managing a specific disk.
//
func (op *DisksService) DiskService(id string) *DiskService {
	return NewDiskService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DisksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DiskService(path), nil
	}
	return op.DiskService(path[:index]).Service(path[index+1:])
}

func (op *DisksService) String() string {
	return fmt.Sprintf("DisksService:%s", op.path)
}

//
//
type VirtualFunctionAllowedNetworkService struct {
	BaseService
}

func NewVirtualFunctionAllowedNetworkService(connection *Connection, path string) *VirtualFunctionAllowedNetworkService {
	var result VirtualFunctionAllowedNetworkService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type VirtualFunctionAllowedNetworkServiceGetRequest struct {
	VirtualFunctionAllowedNetworkService *VirtualFunctionAllowedNetworkService
	header                               map[string]string
	query                                map[string]string
	follow                               *string
}

func (p *VirtualFunctionAllowedNetworkServiceGetRequest) Header(key, value string) *VirtualFunctionAllowedNetworkServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VirtualFunctionAllowedNetworkServiceGetRequest) Query(key, value string) *VirtualFunctionAllowedNetworkServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VirtualFunctionAllowedNetworkServiceGetRequest) Follow(follow string) *VirtualFunctionAllowedNetworkServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *VirtualFunctionAllowedNetworkServiceGetRequest) Send() (*VirtualFunctionAllowedNetworkServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VirtualFunctionAllowedNetworkService.connection.URL(), p.VirtualFunctionAllowedNetworkService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VirtualFunctionAllowedNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VirtualFunctionAllowedNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VirtualFunctionAllowedNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VirtualFunctionAllowedNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VirtualFunctionAllowedNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VirtualFunctionAllowedNetworkServiceGetResponse{network: result}, nil
}

func (p *VirtualFunctionAllowedNetworkServiceGetRequest) MustSend() *VirtualFunctionAllowedNetworkServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VirtualFunctionAllowedNetworkServiceGetResponse struct {
	network *Network
}

func (p *VirtualFunctionAllowedNetworkServiceGetResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *VirtualFunctionAllowedNetworkServiceGetResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
//
func (p *VirtualFunctionAllowedNetworkService) Get() *VirtualFunctionAllowedNetworkServiceGetRequest {
	return &VirtualFunctionAllowedNetworkServiceGetRequest{VirtualFunctionAllowedNetworkService: p}
}

//
//
type VirtualFunctionAllowedNetworkServiceRemoveRequest struct {
	VirtualFunctionAllowedNetworkService *VirtualFunctionAllowedNetworkService
	header                               map[string]string
	query                                map[string]string
	async                                *bool
}

func (p *VirtualFunctionAllowedNetworkServiceRemoveRequest) Header(key, value string) *VirtualFunctionAllowedNetworkServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VirtualFunctionAllowedNetworkServiceRemoveRequest) Query(key, value string) *VirtualFunctionAllowedNetworkServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VirtualFunctionAllowedNetworkServiceRemoveRequest) Async(async bool) *VirtualFunctionAllowedNetworkServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *VirtualFunctionAllowedNetworkServiceRemoveRequest) Send() (*VirtualFunctionAllowedNetworkServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VirtualFunctionAllowedNetworkService.connection.URL(), p.VirtualFunctionAllowedNetworkService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VirtualFunctionAllowedNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VirtualFunctionAllowedNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VirtualFunctionAllowedNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VirtualFunctionAllowedNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VirtualFunctionAllowedNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(VirtualFunctionAllowedNetworkServiceRemoveResponse), nil
}

func (p *VirtualFunctionAllowedNetworkServiceRemoveRequest) MustSend() *VirtualFunctionAllowedNetworkServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VirtualFunctionAllowedNetworkServiceRemoveResponse struct {
}

//
//
func (p *VirtualFunctionAllowedNetworkService) Remove() *VirtualFunctionAllowedNetworkServiceRemoveRequest {
	return &VirtualFunctionAllowedNetworkServiceRemoveRequest{VirtualFunctionAllowedNetworkService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VirtualFunctionAllowedNetworkService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *VirtualFunctionAllowedNetworkService) String() string {
	return fmt.Sprintf("VirtualFunctionAllowedNetworkService:%s", op.path)
}

//
// Manages the set of scheduling policy units available in the system.
//
type SchedulingPolicyUnitsService struct {
	BaseService
}

func NewSchedulingPolicyUnitsService(connection *Connection, path string) *SchedulingPolicyUnitsService {
	var result SchedulingPolicyUnitsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of scheduling policy units available in the system.
// The order of the returned list of scheduling policy units isn't guaranteed.
//
type SchedulingPolicyUnitsServiceListRequest struct {
	SchedulingPolicyUnitsService *SchedulingPolicyUnitsService
	header                       map[string]string
	query                        map[string]string
	filter                       *bool
	follow                       *string
	max                          *int64
}

func (p *SchedulingPolicyUnitsServiceListRequest) Header(key, value string) *SchedulingPolicyUnitsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPolicyUnitsServiceListRequest) Query(key, value string) *SchedulingPolicyUnitsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPolicyUnitsServiceListRequest) Filter(filter bool) *SchedulingPolicyUnitsServiceListRequest {
	p.filter = &filter
	return p
}

func (p *SchedulingPolicyUnitsServiceListRequest) Follow(follow string) *SchedulingPolicyUnitsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *SchedulingPolicyUnitsServiceListRequest) Max(max int64) *SchedulingPolicyUnitsServiceListRequest {
	p.max = &max
	return p
}

func (p *SchedulingPolicyUnitsServiceListRequest) Send() (*SchedulingPolicyUnitsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPolicyUnitsService.connection.URL(), p.SchedulingPolicyUnitsService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPolicyUnitsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPolicyUnitsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPolicyUnitsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPolicyUnitsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPolicyUnitsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSchedulingPolicyUnitReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &SchedulingPolicyUnitsServiceListResponse{units: result}, nil
}

func (p *SchedulingPolicyUnitsServiceListRequest) MustSend() *SchedulingPolicyUnitsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of scheduling policy units available in the system.
// The order of the returned list of scheduling policy units isn't guaranteed.
//
type SchedulingPolicyUnitsServiceListResponse struct {
	units *SchedulingPolicyUnitSlice
}

func (p *SchedulingPolicyUnitsServiceListResponse) Units() (*SchedulingPolicyUnitSlice, bool) {
	if p.units != nil {
		return p.units, true
	}
	return nil, false
}

func (p *SchedulingPolicyUnitsServiceListResponse) MustUnits() *SchedulingPolicyUnitSlice {
	if p.units == nil {
		panic("units in response does not exist")
	}
	return p.units
}

//
// Returns the list of scheduling policy units available in the system.
// The order of the returned list of scheduling policy units isn't guaranteed.
//
func (p *SchedulingPolicyUnitsService) List() *SchedulingPolicyUnitsServiceListRequest {
	return &SchedulingPolicyUnitsServiceListRequest{SchedulingPolicyUnitsService: p}
}

//
//
func (op *SchedulingPolicyUnitsService) UnitService(id string) *SchedulingPolicyUnitService {
	return NewSchedulingPolicyUnitService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SchedulingPolicyUnitsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.UnitService(path), nil
	}
	return op.UnitService(path[:index]).Service(path[index+1:])
}

func (op *SchedulingPolicyUnitsService) String() string {
	return fmt.Sprintf("SchedulingPolicyUnitsService:%s", op.path)
}

//
// Manages a single disk available in a storage domain attached to a data center.
// IMPORTANT: Since version 4.2 of the engine this service is intended only to list disks available in the storage
// domain, and to register unregistered disks. All the other operations, like copying a disk, moving a disk, etc, have
// been deprecated and will be removed in the future. To perform those operations use the <<services/disks, service
// that manages all the disks of the system>>, or the <<services/disk, service that manages an specific disk>>.
//
type AttachedStorageDomainDiskService struct {
	BaseService
}

func NewAttachedStorageDomainDiskService(connection *Connection, path string) *AttachedStorageDomainDiskService {
	var result AttachedStorageDomainDiskService
	result.connection = connection
	result.path = path
	return &result
}

//
// Copies a disk to the specified storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To copy a disk use the <<services/disk/methods/copy, copy>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceCopyRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	disk                             *Disk
	storageDomain                    *StorageDomain
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) Header(key, value string) *AttachedStorageDomainDiskServiceCopyRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) Query(key, value string) *AttachedStorageDomainDiskServiceCopyRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) Disk(disk *Disk) *AttachedStorageDomainDiskServiceCopyRequest {
	p.disk = disk
	return p
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) StorageDomain(storageDomain *StorageDomain) *AttachedStorageDomainDiskServiceCopyRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) Send() (*AttachedStorageDomainDiskServiceCopyResponse, error) {
	rawURL := fmt.Sprintf("%s%s/copy", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Disk(p.disk)
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceCopyResponse), nil
}

func (p *AttachedStorageDomainDiskServiceCopyRequest) MustSend() *AttachedStorageDomainDiskServiceCopyResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Copies a disk to the specified storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To copy a disk use the <<services/disk/methods/copy, copy>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceCopyResponse struct {
}

//
// Copies a disk to the specified storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To copy a disk use the <<services/disk/methods/copy, copy>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Copy() *AttachedStorageDomainDiskServiceCopyRequest {
	return &AttachedStorageDomainDiskServiceCopyRequest{AttachedStorageDomainDiskService: p}
}

//
// Exports a disk to an export storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To export a disk use the <<services/disk/methods/export, export>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceExportRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	storageDomain                    *StorageDomain
}

func (p *AttachedStorageDomainDiskServiceExportRequest) Header(key, value string) *AttachedStorageDomainDiskServiceExportRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceExportRequest) Query(key, value string) *AttachedStorageDomainDiskServiceExportRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceExportRequest) StorageDomain(storageDomain *StorageDomain) *AttachedStorageDomainDiskServiceExportRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *AttachedStorageDomainDiskServiceExportRequest) Send() (*AttachedStorageDomainDiskServiceExportResponse, error) {
	rawURL := fmt.Sprintf("%s%s/export", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceExportResponse), nil
}

func (p *AttachedStorageDomainDiskServiceExportRequest) MustSend() *AttachedStorageDomainDiskServiceExportResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Exports a disk to an export storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To export a disk use the <<services/disk/methods/export, export>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceExportResponse struct {
}

//
// Exports a disk to an export storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To export a disk use the <<services/disk/methods/export, export>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Export() *AttachedStorageDomainDiskServiceExportRequest {
	return &AttachedStorageDomainDiskServiceExportRequest{AttachedStorageDomainDiskService: p}
}

//
// Retrieves the description of the disk.
//
type AttachedStorageDomainDiskServiceGetRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	follow                           *string
}

func (p *AttachedStorageDomainDiskServiceGetRequest) Header(key, value string) *AttachedStorageDomainDiskServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceGetRequest) Query(key, value string) *AttachedStorageDomainDiskServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceGetRequest) Follow(follow string) *AttachedStorageDomainDiskServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AttachedStorageDomainDiskServiceGetRequest) Send() (*AttachedStorageDomainDiskServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainDiskServiceGetResponse{disk: result}, nil
}

func (p *AttachedStorageDomainDiskServiceGetRequest) MustSend() *AttachedStorageDomainDiskServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the description of the disk.
//
type AttachedStorageDomainDiskServiceGetResponse struct {
	disk *Disk
}

func (p *AttachedStorageDomainDiskServiceGetResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *AttachedStorageDomainDiskServiceGetResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Retrieves the description of the disk.
//
func (p *AttachedStorageDomainDiskService) Get() *AttachedStorageDomainDiskServiceGetRequest {
	return &AttachedStorageDomainDiskServiceGetRequest{AttachedStorageDomainDiskService: p}
}

//
// Moves a disk to another storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To move a disk use the <<services/disk/methods/move, move>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceMoveRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	async                            *bool
	filter                           *bool
	storageDomain                    *StorageDomain
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Header(key, value string) *AttachedStorageDomainDiskServiceMoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Query(key, value string) *AttachedStorageDomainDiskServiceMoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Async(async bool) *AttachedStorageDomainDiskServiceMoveRequest {
	p.async = &async
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Filter(filter bool) *AttachedStorageDomainDiskServiceMoveRequest {
	p.filter = &filter
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) StorageDomain(storageDomain *StorageDomain) *AttachedStorageDomainDiskServiceMoveRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) Send() (*AttachedStorageDomainDiskServiceMoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s/move", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.filter != nil {
		actionBuilder.Filter(*p.filter)
	}
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceMoveResponse), nil
}

func (p *AttachedStorageDomainDiskServiceMoveRequest) MustSend() *AttachedStorageDomainDiskServiceMoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Moves a disk to another storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To move a disk use the <<services/disk/methods/move, move>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceMoveResponse struct {
}

//
// Moves a disk to another storage domain.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To move a disk use the <<services/disk/methods/move, move>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Move() *AttachedStorageDomainDiskServiceMoveRequest {
	return &AttachedStorageDomainDiskServiceMoveRequest{AttachedStorageDomainDiskService: p}
}

//
// Registers an unregistered disk.
//
type AttachedStorageDomainDiskServiceRegisterRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
}

func (p *AttachedStorageDomainDiskServiceRegisterRequest) Header(key, value string) *AttachedStorageDomainDiskServiceRegisterRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceRegisterRequest) Query(key, value string) *AttachedStorageDomainDiskServiceRegisterRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceRegisterRequest) Send() (*AttachedStorageDomainDiskServiceRegisterResponse, error) {
	rawURL := fmt.Sprintf("%s%s/register", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceRegisterResponse), nil
}

func (p *AttachedStorageDomainDiskServiceRegisterRequest) MustSend() *AttachedStorageDomainDiskServiceRegisterResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Registers an unregistered disk.
//
type AttachedStorageDomainDiskServiceRegisterResponse struct {
}

//
// Registers an unregistered disk.
//
func (p *AttachedStorageDomainDiskService) Register() *AttachedStorageDomainDiskServiceRegisterRequest {
	return &AttachedStorageDomainDiskServiceRegisterRequest{AttachedStorageDomainDiskService: p}
}

//
// Removes a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceRemoveRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
}

func (p *AttachedStorageDomainDiskServiceRemoveRequest) Header(key, value string) *AttachedStorageDomainDiskServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceRemoveRequest) Query(key, value string) *AttachedStorageDomainDiskServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceRemoveRequest) Send() (*AttachedStorageDomainDiskServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AttachedStorageDomainDiskServiceRemoveResponse), nil
}

func (p *AttachedStorageDomainDiskServiceRemoveRequest) MustSend() *AttachedStorageDomainDiskServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceRemoveResponse struct {
}

//
// Removes a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Remove() *AttachedStorageDomainDiskServiceRemoveRequest {
	return &AttachedStorageDomainDiskServiceRemoveRequest{AttachedStorageDomainDiskService: p}
}

//
// Sparsify the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceSparsifyRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
}

func (p *AttachedStorageDomainDiskServiceSparsifyRequest) Header(key, value string) *AttachedStorageDomainDiskServiceSparsifyRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceSparsifyRequest) Query(key, value string) *AttachedStorageDomainDiskServiceSparsifyRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceSparsifyRequest) Send() (*AttachedStorageDomainDiskServiceSparsifyResponse, error) {
	rawURL := fmt.Sprintf("%s%s/sparsify", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	actionBuilder := NewActionBuilder()
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainDiskServiceSparsifyResponse), nil
}

func (p *AttachedStorageDomainDiskServiceSparsifyRequest) MustSend() *AttachedStorageDomainDiskServiceSparsifyResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Sparsify the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceSparsifyResponse struct {
}

//
// Sparsify the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To remove a disk use the <<services/disk/methods/remove, remove>>
// operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Sparsify() *AttachedStorageDomainDiskServiceSparsifyRequest {
	return &AttachedStorageDomainDiskServiceSparsifyRequest{AttachedStorageDomainDiskService: p}
}

//
// Updates the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To update a disk use the
// <<services/disk/methods/update, update>> operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceUpdateRequest struct {
	AttachedStorageDomainDiskService *AttachedStorageDomainDiskService
	header                           map[string]string
	query                            map[string]string
	disk                             *Disk
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) Header(key, value string) *AttachedStorageDomainDiskServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) Query(key, value string) *AttachedStorageDomainDiskServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) Disk(disk *Disk) *AttachedStorageDomainDiskServiceUpdateRequest {
	p.disk = disk
	return p
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) Send() (*AttachedStorageDomainDiskServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainDiskService.connection.URL(), p.AttachedStorageDomainDiskService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskWriteOne(writer, p.disk, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainDiskServiceUpdateResponse{disk: result}, nil
}

func (p *AttachedStorageDomainDiskServiceUpdateRequest) MustSend() *AttachedStorageDomainDiskServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To update a disk use the
// <<services/disk/methods/update, update>> operation of the service that manages that disk.
//
type AttachedStorageDomainDiskServiceUpdateResponse struct {
	disk *Disk
}

func (p *AttachedStorageDomainDiskServiceUpdateResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *AttachedStorageDomainDiskServiceUpdateResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Updates the disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To update a disk use the
// <<services/disk/methods/update, update>> operation of the service that manages that disk.
//
func (p *AttachedStorageDomainDiskService) Update() *AttachedStorageDomainDiskServiceUpdateRequest {
	return &AttachedStorageDomainDiskServiceUpdateRequest{AttachedStorageDomainDiskService: p}
}

//
// Reference to the service that manages the permissions assigned to the disk.
//
func (op *AttachedStorageDomainDiskService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
//
func (op *AttachedStorageDomainDiskService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AttachedStorageDomainDiskService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AttachedStorageDomainDiskService) String() string {
	return fmt.Sprintf("AttachedStorageDomainDiskService:%s", op.path)
}

//
// Describes how an external network provider is provisioned by the system on the host.
//
type ExternalNetworkProviderConfigurationService struct {
	BaseService
}

func NewExternalNetworkProviderConfigurationService(connection *Connection, path string) *ExternalNetworkProviderConfigurationService {
	var result ExternalNetworkProviderConfigurationService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the information about an external network provider on the host.
//
type ExternalNetworkProviderConfigurationServiceGetRequest struct {
	ExternalNetworkProviderConfigurationService *ExternalNetworkProviderConfigurationService
	header                                      map[string]string
	query                                       map[string]string
	follow                                      *string
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) Header(key, value string) *ExternalNetworkProviderConfigurationServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) Query(key, value string) *ExternalNetworkProviderConfigurationServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) Follow(follow string) *ExternalNetworkProviderConfigurationServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) Send() (*ExternalNetworkProviderConfigurationServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalNetworkProviderConfigurationService.connection.URL(), p.ExternalNetworkProviderConfigurationService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalNetworkProviderConfigurationService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalNetworkProviderConfigurationService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalNetworkProviderConfigurationService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalNetworkProviderConfigurationService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalNetworkProviderConfigurationService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalNetworkProviderConfigurationReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalNetworkProviderConfigurationServiceGetResponse{configuration: result}, nil
}

func (p *ExternalNetworkProviderConfigurationServiceGetRequest) MustSend() *ExternalNetworkProviderConfigurationServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the information about an external network provider on the host.
//
type ExternalNetworkProviderConfigurationServiceGetResponse struct {
	configuration *ExternalNetworkProviderConfiguration
}

func (p *ExternalNetworkProviderConfigurationServiceGetResponse) Configuration() (*ExternalNetworkProviderConfiguration, bool) {
	if p.configuration != nil {
		return p.configuration, true
	}
	return nil, false
}

func (p *ExternalNetworkProviderConfigurationServiceGetResponse) MustConfiguration() *ExternalNetworkProviderConfiguration {
	if p.configuration == nil {
		panic("configuration in response does not exist")
	}
	return p.configuration
}

//
// Returns the information about an external network provider on the host.
//
func (p *ExternalNetworkProviderConfigurationService) Get() *ExternalNetworkProviderConfigurationServiceGetRequest {
	return &ExternalNetworkProviderConfigurationServiceGetRequest{ExternalNetworkProviderConfigurationService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalNetworkProviderConfigurationService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalNetworkProviderConfigurationService) String() string {
	return fmt.Sprintf("ExternalNetworkProviderConfigurationService:%s", op.path)
}

//
// Manages the set of scheduling policies available in the system.
//
type SchedulingPoliciesService struct {
	BaseService
}

func NewSchedulingPoliciesService(connection *Connection, path string) *SchedulingPoliciesService {
	var result SchedulingPoliciesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new scheduling policy to the system.
//
type SchedulingPoliciesServiceAddRequest struct {
	SchedulingPoliciesService *SchedulingPoliciesService
	header                    map[string]string
	query                     map[string]string
	policy                    *SchedulingPolicy
}

func (p *SchedulingPoliciesServiceAddRequest) Header(key, value string) *SchedulingPoliciesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPoliciesServiceAddRequest) Query(key, value string) *SchedulingPoliciesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPoliciesServiceAddRequest) Policy(policy *SchedulingPolicy) *SchedulingPoliciesServiceAddRequest {
	p.policy = policy
	return p
}

func (p *SchedulingPoliciesServiceAddRequest) Send() (*SchedulingPoliciesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPoliciesService.connection.URL(), p.SchedulingPoliciesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLSchedulingPolicyWriteOne(writer, p.policy, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPoliciesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPoliciesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPoliciesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPoliciesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPoliciesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSchedulingPolicyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SchedulingPoliciesServiceAddResponse{policy: result}, nil
}

func (p *SchedulingPoliciesServiceAddRequest) MustSend() *SchedulingPoliciesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new scheduling policy to the system.
//
type SchedulingPoliciesServiceAddResponse struct {
	policy *SchedulingPolicy
}

func (p *SchedulingPoliciesServiceAddResponse) Policy() (*SchedulingPolicy, bool) {
	if p.policy != nil {
		return p.policy, true
	}
	return nil, false
}

func (p *SchedulingPoliciesServiceAddResponse) MustPolicy() *SchedulingPolicy {
	if p.policy == nil {
		panic("policy in response does not exist")
	}
	return p.policy
}

//
// Add a new scheduling policy to the system.
//
func (p *SchedulingPoliciesService) Add() *SchedulingPoliciesServiceAddRequest {
	return &SchedulingPoliciesServiceAddRequest{SchedulingPoliciesService: p}
}

//
// Returns the list of scheduling policies available in the system.
// The order of the returned list of scheduling policies isn't guaranteed.
//
type SchedulingPoliciesServiceListRequest struct {
	SchedulingPoliciesService *SchedulingPoliciesService
	header                    map[string]string
	query                     map[string]string
	filter                    *bool
	follow                    *string
	max                       *int64
}

func (p *SchedulingPoliciesServiceListRequest) Header(key, value string) *SchedulingPoliciesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPoliciesServiceListRequest) Query(key, value string) *SchedulingPoliciesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPoliciesServiceListRequest) Filter(filter bool) *SchedulingPoliciesServiceListRequest {
	p.filter = &filter
	return p
}

func (p *SchedulingPoliciesServiceListRequest) Follow(follow string) *SchedulingPoliciesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *SchedulingPoliciesServiceListRequest) Max(max int64) *SchedulingPoliciesServiceListRequest {
	p.max = &max
	return p
}

func (p *SchedulingPoliciesServiceListRequest) Send() (*SchedulingPoliciesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPoliciesService.connection.URL(), p.SchedulingPoliciesService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPoliciesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPoliciesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPoliciesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPoliciesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPoliciesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSchedulingPolicyReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &SchedulingPoliciesServiceListResponse{policies: result}, nil
}

func (p *SchedulingPoliciesServiceListRequest) MustSend() *SchedulingPoliciesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of scheduling policies available in the system.
// The order of the returned list of scheduling policies isn't guaranteed.
//
type SchedulingPoliciesServiceListResponse struct {
	policies *SchedulingPolicySlice
}

func (p *SchedulingPoliciesServiceListResponse) Policies() (*SchedulingPolicySlice, bool) {
	if p.policies != nil {
		return p.policies, true
	}
	return nil, false
}

func (p *SchedulingPoliciesServiceListResponse) MustPolicies() *SchedulingPolicySlice {
	if p.policies == nil {
		panic("policies in response does not exist")
	}
	return p.policies
}

//
// Returns the list of scheduling policies available in the system.
// The order of the returned list of scheduling policies isn't guaranteed.
//
func (p *SchedulingPoliciesService) List() *SchedulingPoliciesServiceListRequest {
	return &SchedulingPoliciesServiceListRequest{SchedulingPoliciesService: p}
}

//
//
func (op *SchedulingPoliciesService) PolicyService(id string) *SchedulingPolicyService {
	return NewSchedulingPolicyService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SchedulingPoliciesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.PolicyService(path), nil
	}
	return op.PolicyService(path[:index]).Service(path[index+1:])
}

func (op *SchedulingPoliciesService) String() string {
	return fmt.Sprintf("SchedulingPoliciesService:%s", op.path)
}

//
//
type ImageService struct {
	BaseService
}

func NewImageService(connection *Connection, path string) *ImageService {
	var result ImageService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type ImageServiceGetRequest struct {
	ImageService *ImageService
	header       map[string]string
	query        map[string]string
	follow       *string
}

func (p *ImageServiceGetRequest) Header(key, value string) *ImageServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ImageServiceGetRequest) Query(key, value string) *ImageServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ImageServiceGetRequest) Follow(follow string) *ImageServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ImageServiceGetRequest) Send() (*ImageServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ImageService.connection.URL(), p.ImageService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ImageService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ImageService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ImageService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ImageService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ImageService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLImageReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ImageServiceGetResponse{image: result}, nil
}

func (p *ImageServiceGetRequest) MustSend() *ImageServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type ImageServiceGetResponse struct {
	image *Image
}

func (p *ImageServiceGetResponse) Image() (*Image, bool) {
	if p.image != nil {
		return p.image, true
	}
	return nil, false
}

func (p *ImageServiceGetResponse) MustImage() *Image {
	if p.image == nil {
		panic("image in response does not exist")
	}
	return p.image
}

//
//
func (p *ImageService) Get() *ImageServiceGetRequest {
	return &ImageServiceGetRequest{ImageService: p}
}

//
// Imports an image.
// If the `import_as_template` parameter is `true` then the image will be imported as a template, otherwise it will
// be imported as a disk.
// When imported as a template, the name of the template can be specified by the optional `template.name`
// parameter. If that parameter is not specified, then the name of the template will be automatically assigned by the
// engine as `GlanceTemplate-x` (where `x` will be seven random hexadecimal characters).
// When imported as a disk, the name of the disk can be specified by the optional `disk.name` parameter. If
// that parameter is not specified, then the name of the disk will be automatically assigned by the engine as
// `GlanceDisk-x` (where `x` will be the seven hexadecimal characters of the image identifier).
// It is recommended to always explicitly specify the template or disk name, to avoid these automatic names
// generated by the engine.
//
type ImageServiceImportRequest struct {
	ImageService     *ImageService
	header           map[string]string
	query            map[string]string
	async            *bool
	cluster          *Cluster
	disk             *Disk
	importAsTemplate *bool
	storageDomain    *StorageDomain
	template         *Template
}

func (p *ImageServiceImportRequest) Header(key, value string) *ImageServiceImportRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ImageServiceImportRequest) Query(key, value string) *ImageServiceImportRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ImageServiceImportRequest) Async(async bool) *ImageServiceImportRequest {
	p.async = &async
	return p
}

func (p *ImageServiceImportRequest) Cluster(cluster *Cluster) *ImageServiceImportRequest {
	p.cluster = cluster
	return p
}

func (p *ImageServiceImportRequest) Disk(disk *Disk) *ImageServiceImportRequest {
	p.disk = disk
	return p
}

func (p *ImageServiceImportRequest) ImportAsTemplate(importAsTemplate bool) *ImageServiceImportRequest {
	p.importAsTemplate = &importAsTemplate
	return p
}

func (p *ImageServiceImportRequest) StorageDomain(storageDomain *StorageDomain) *ImageServiceImportRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *ImageServiceImportRequest) Template(template *Template) *ImageServiceImportRequest {
	p.template = template
	return p
}

func (p *ImageServiceImportRequest) Send() (*ImageServiceImportResponse, error) {
	rawURL := fmt.Sprintf("%s%s/import", p.ImageService.connection.URL(), p.ImageService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Cluster(p.cluster)
	actionBuilder.Disk(p.disk)
	if p.importAsTemplate != nil {
		actionBuilder.ImportAsTemplate(*p.importAsTemplate)
	}
	actionBuilder.StorageDomain(p.storageDomain)
	actionBuilder.Template(p.template)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ImageService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ImageService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ImageService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ImageService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ImageService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(ImageServiceImportResponse), nil
}

func (p *ImageServiceImportRequest) MustSend() *ImageServiceImportResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Imports an image.
// If the `import_as_template` parameter is `true` then the image will be imported as a template, otherwise it will
// be imported as a disk.
// When imported as a template, the name of the template can be specified by the optional `template.name`
// parameter. If that parameter is not specified, then the name of the template will be automatically assigned by the
// engine as `GlanceTemplate-x` (where `x` will be seven random hexadecimal characters).
// When imported as a disk, the name of the disk can be specified by the optional `disk.name` parameter. If
// that parameter is not specified, then the name of the disk will be automatically assigned by the engine as
// `GlanceDisk-x` (where `x` will be the seven hexadecimal characters of the image identifier).
// It is recommended to always explicitly specify the template or disk name, to avoid these automatic names
// generated by the engine.
//
type ImageServiceImportResponse struct {
}

//
// Imports an image.
// If the `import_as_template` parameter is `true` then the image will be imported as a template, otherwise it will
// be imported as a disk.
// When imported as a template, the name of the template can be specified by the optional `template.name`
// parameter. If that parameter is not specified, then the name of the template will be automatically assigned by the
// engine as `GlanceTemplate-x` (where `x` will be seven random hexadecimal characters).
// When imported as a disk, the name of the disk can be specified by the optional `disk.name` parameter. If
// that parameter is not specified, then the name of the disk will be automatically assigned by the engine as
// `GlanceDisk-x` (where `x` will be the seven hexadecimal characters of the image identifier).
// It is recommended to always explicitly specify the template or disk name, to avoid these automatic names
// generated by the engine.
//
func (p *ImageService) Import() *ImageServiceImportRequest {
	return &ImageServiceImportRequest{ImageService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ImageService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ImageService) String() string {
	return fmt.Sprintf("ImageService:%s", op.path)
}

//
// A service to manage a network interface of a host.
//
type HostNicService struct {
	BaseService
}

func NewHostNicService(connection *Connection, path string) *HostNicService {
	var result HostNicService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type HostNicServiceGetRequest struct {
	HostNicService *HostNicService
	header         map[string]string
	query          map[string]string
	allContent     *bool
	follow         *string
}

func (p *HostNicServiceGetRequest) Header(key, value string) *HostNicServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostNicServiceGetRequest) Query(key, value string) *HostNicServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostNicServiceGetRequest) AllContent(allContent bool) *HostNicServiceGetRequest {
	p.allContent = &allContent
	return p
}

func (p *HostNicServiceGetRequest) Follow(follow string) *HostNicServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *HostNicServiceGetRequest) Send() (*HostNicServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostNicService.connection.URL(), p.HostNicService.path)
	values := make(url.Values)
	if p.allContent != nil {
		values["all_content"] = []string{fmt.Sprintf("%v", *p.allContent)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostNicService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostNicService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostNicService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostNicService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostNicService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostNicReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &HostNicServiceGetResponse{nic: result}, nil
}

func (p *HostNicServiceGetRequest) MustSend() *HostNicServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type HostNicServiceGetResponse struct {
	nic *HostNic
}

func (p *HostNicServiceGetResponse) Nic() (*HostNic, bool) {
	if p.nic != nil {
		return p.nic, true
	}
	return nil, false
}

func (p *HostNicServiceGetResponse) MustNic() *HostNic {
	if p.nic == nil {
		panic("nic in response does not exist")
	}
	return p.nic
}

//
//
func (p *HostNicService) Get() *HostNicServiceGetRequest {
	return &HostNicServiceGetRequest{HostNicService: p}
}

//
// The action updates virtual function configuration in case the current resource represents an SR-IOV enabled NIC.
// The input should be consisted of at least one of the following properties:
// - `allNetworksAllowed`
// - `numberOfVirtualFunctions`
// Please see the `HostNicVirtualFunctionsConfiguration` type for the meaning of the properties.
//
type HostNicServiceUpdateVirtualFunctionsConfigurationRequest struct {
	HostNicService                *HostNicService
	header                        map[string]string
	query                         map[string]string
	async                         *bool
	virtualFunctionsConfiguration *HostNicVirtualFunctionsConfiguration
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) Header(key, value string) *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) Query(key, value string) *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) Async(async bool) *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	p.async = &async
	return p
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) VirtualFunctionsConfiguration(virtualFunctionsConfiguration *HostNicVirtualFunctionsConfiguration) *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	p.virtualFunctionsConfiguration = virtualFunctionsConfiguration
	return p
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) Send() (*HostNicServiceUpdateVirtualFunctionsConfigurationResponse, error) {
	rawURL := fmt.Sprintf("%s%s/updatevirtualfunctionsconfiguration", p.HostNicService.connection.URL(), p.HostNicService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.VirtualFunctionsConfiguration(p.virtualFunctionsConfiguration)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostNicService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostNicService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostNicService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostNicService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostNicService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(HostNicServiceUpdateVirtualFunctionsConfigurationResponse), nil
}

func (p *HostNicServiceUpdateVirtualFunctionsConfigurationRequest) MustSend() *HostNicServiceUpdateVirtualFunctionsConfigurationResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// The action updates virtual function configuration in case the current resource represents an SR-IOV enabled NIC.
// The input should be consisted of at least one of the following properties:
// - `allNetworksAllowed`
// - `numberOfVirtualFunctions`
// Please see the `HostNicVirtualFunctionsConfiguration` type for the meaning of the properties.
//
type HostNicServiceUpdateVirtualFunctionsConfigurationResponse struct {
}

//
// The action updates virtual function configuration in case the current resource represents an SR-IOV enabled NIC.
// The input should be consisted of at least one of the following properties:
// - `allNetworksAllowed`
// - `numberOfVirtualFunctions`
// Please see the `HostNicVirtualFunctionsConfiguration` type for the meaning of the properties.
//
func (p *HostNicService) UpdateVirtualFunctionsConfiguration() *HostNicServiceUpdateVirtualFunctionsConfigurationRequest {
	return &HostNicServiceUpdateVirtualFunctionsConfigurationRequest{HostNicService: p}
}

//
// A reference to information elements received by LLDP on the NIC.
//
func (op *HostNicService) LinkLayerDiscoveryProtocolElementsService() *LinkLayerDiscoveryProtocolService {
	return NewLinkLayerDiscoveryProtocolService(op.connection, fmt.Sprintf("%s/linklayerdiscoveryprotocolelements", op.path))
}

//
// Reference to the service that manages the network attachments assigned to this network interface.
//
func (op *HostNicService) NetworkAttachmentsService() *NetworkAttachmentsService {
	return NewNetworkAttachmentsService(op.connection, fmt.Sprintf("%s/networkattachments", op.path))
}

//
// Reference to the service that manages the network labels assigned to this network interface.
//
func (op *HostNicService) NetworkLabelsService() *NetworkLabelsService {
	return NewNetworkLabelsService(op.connection, fmt.Sprintf("%s/networklabels", op.path))
}

//
//
func (op *HostNicService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Retrieves sub-collection resource of network labels that are allowed on an the virtual functions
// in case that the current resource represents an SR-IOV physical function NIC.
//
func (op *HostNicService) VirtualFunctionAllowedLabelsService() *NetworkLabelsService {
	return NewNetworkLabelsService(op.connection, fmt.Sprintf("%s/virtualfunctionallowedlabels", op.path))
}

//
// Retrieves sub-collection resource of networks that are allowed on an the virtual functions
// in case that the current resource represents an SR-IOV physical function NIC.
//
func (op *HostNicService) VirtualFunctionAllowedNetworksService() *VirtualFunctionAllowedNetworksService {
	return NewVirtualFunctionAllowedNetworksService(op.connection, fmt.Sprintf("%s/virtualfunctionallowednetworks", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostNicService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "linklayerdiscoveryprotocolelements" {
		return op.LinkLayerDiscoveryProtocolElementsService(), nil
	}
	if strings.HasPrefix(path, "linklayerdiscoveryprotocolelements/") {
		return op.LinkLayerDiscoveryProtocolElementsService().Service(path[35:])
	}
	if path == "networkattachments" {
		return op.NetworkAttachmentsService(), nil
	}
	if strings.HasPrefix(path, "networkattachments/") {
		return op.NetworkAttachmentsService().Service(path[19:])
	}
	if path == "networklabels" {
		return op.NetworkLabelsService(), nil
	}
	if strings.HasPrefix(path, "networklabels/") {
		return op.NetworkLabelsService().Service(path[14:])
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	if path == "virtualfunctionallowedlabels" {
		return op.VirtualFunctionAllowedLabelsService(), nil
	}
	if strings.HasPrefix(path, "virtualfunctionallowedlabels/") {
		return op.VirtualFunctionAllowedLabelsService().Service(path[29:])
	}
	if path == "virtualfunctionallowednetworks" {
		return op.VirtualFunctionAllowedNetworksService(), nil
	}
	if strings.HasPrefix(path, "virtualfunctionallowednetworks/") {
		return op.VirtualFunctionAllowedNetworksService().Service(path[31:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *HostNicService) String() string {
	return fmt.Sprintf("HostNicService:%s", op.path)
}

//
//
type VmNicsService struct {
	BaseService
}

func NewVmNicsService(connection *Connection, path string) *VmNicsService {
	var result VmNicsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a NIC to the virtual machine.
// The following example adds to the virtual machine `123` a network interface named `mynic` using `virtio` and the
// NIC profile `456`.
// [source]
// ----
// POST /ovirt-engine/api/vms/123/nics
// ----
// [source,xml]
// ----
// <nic>
//   <name>mynic</name>
//   <interface>virtio</interface>
//   <vnic_profile id="456"/>
// </nic>
// ----
// The following example sends that request using `curl`:
// [source,bash]
// ----
// curl \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --user "admin@internal:mypassword" \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --data '
// <nic>
//   <name>mynic</name>
//   <interface>virtio</interface>
//   <vnic_profile id="456"/>
// </nic>
// ' \
// https://myengine.example.com/ovirt-engine/api/vms/123/nics
// ----
// [IMPORTANT]
// ====
// The hotplugging feature only supports virtual machine operating systems with hotplugging operations.
// Example operating systems include:
// - Red Hat Enterprise Linux 6
// - Red Hat Enterprise Linux 5
// - Windows Server 2008 and
// - Windows Server 2003
// ====
//
type VmNicsServiceAddRequest struct {
	VmNicsService *VmNicsService
	header        map[string]string
	query         map[string]string
	nic           *Nic
}

func (p *VmNicsServiceAddRequest) Header(key, value string) *VmNicsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmNicsServiceAddRequest) Query(key, value string) *VmNicsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmNicsServiceAddRequest) Nic(nic *Nic) *VmNicsServiceAddRequest {
	p.nic = nic
	return p
}

func (p *VmNicsServiceAddRequest) Send() (*VmNicsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmNicsService.connection.URL(), p.VmNicsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNicWriteOne(writer, p.nic, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmNicsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmNicsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmNicsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmNicsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmNicsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNicReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmNicsServiceAddResponse{nic: result}, nil
}

func (p *VmNicsServiceAddRequest) MustSend() *VmNicsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a NIC to the virtual machine.
// The following example adds to the virtual machine `123` a network interface named `mynic` using `virtio` and the
// NIC profile `456`.
// [source]
// ----
// POST /ovirt-engine/api/vms/123/nics
// ----
// [source,xml]
// ----
// <nic>
//   <name>mynic</name>
//   <interface>virtio</interface>
//   <vnic_profile id="456"/>
// </nic>
// ----
// The following example sends that request using `curl`:
// [source,bash]
// ----
// curl \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --user "admin@internal:mypassword" \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --data '
// <nic>
//   <name>mynic</name>
//   <interface>virtio</interface>
//   <vnic_profile id="456"/>
// </nic>
// ' \
// https://myengine.example.com/ovirt-engine/api/vms/123/nics
// ----
// [IMPORTANT]
// ====
// The hotplugging feature only supports virtual machine operating systems with hotplugging operations.
// Example operating systems include:
// - Red Hat Enterprise Linux 6
// - Red Hat Enterprise Linux 5
// - Windows Server 2008 and
// - Windows Server 2003
// ====
//
type VmNicsServiceAddResponse struct {
	nic *Nic
}

func (p *VmNicsServiceAddResponse) Nic() (*Nic, bool) {
	if p.nic != nil {
		return p.nic, true
	}
	return nil, false
}

func (p *VmNicsServiceAddResponse) MustNic() *Nic {
	if p.nic == nil {
		panic("nic in response does not exist")
	}
	return p.nic
}

//
// Adds a NIC to the virtual machine.
// The following example adds to the virtual machine `123` a network interface named `mynic` using `virtio` and the
// NIC profile `456`.
// [source]
// ----
// POST /ovirt-engine/api/vms/123/nics
// ----
// [source,xml]
// ----
// <nic>
//   <name>mynic</name>
//   <interface>virtio</interface>
//   <vnic_profile id="456"/>
// </nic>
// ----
// The following example sends that request using `curl`:
// [source,bash]
// ----
// curl \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --user "admin@internal:mypassword" \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --data '
// <nic>
//   <name>mynic</name>
//   <interface>virtio</interface>
//   <vnic_profile id="456"/>
// </nic>
// ' \
// https://myengine.example.com/ovirt-engine/api/vms/123/nics
// ----
// [IMPORTANT]
// ====
// The hotplugging feature only supports virtual machine operating systems with hotplugging operations.
// Example operating systems include:
// - Red Hat Enterprise Linux 6
// - Red Hat Enterprise Linux 5
// - Windows Server 2008 and
// - Windows Server 2003
// ====
//
func (p *VmNicsService) Add() *VmNicsServiceAddRequest {
	return &VmNicsServiceAddRequest{VmNicsService: p}
}

//
// Returns the list of NICs of the virtual machine.
// The order of the returned list of NICs isn't guaranteed.
//
type VmNicsServiceListRequest struct {
	VmNicsService *VmNicsService
	header        map[string]string
	query         map[string]string
	follow        *string
	max           *int64
}

func (p *VmNicsServiceListRequest) Header(key, value string) *VmNicsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmNicsServiceListRequest) Query(key, value string) *VmNicsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmNicsServiceListRequest) Follow(follow string) *VmNicsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *VmNicsServiceListRequest) Max(max int64) *VmNicsServiceListRequest {
	p.max = &max
	return p
}

func (p *VmNicsServiceListRequest) Send() (*VmNicsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmNicsService.connection.URL(), p.VmNicsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmNicsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmNicsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmNicsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmNicsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmNicsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNicReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &VmNicsServiceListResponse{nics: result}, nil
}

func (p *VmNicsServiceListRequest) MustSend() *VmNicsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of NICs of the virtual machine.
// The order of the returned list of NICs isn't guaranteed.
//
type VmNicsServiceListResponse struct {
	nics *NicSlice
}

func (p *VmNicsServiceListResponse) Nics() (*NicSlice, bool) {
	if p.nics != nil {
		return p.nics, true
	}
	return nil, false
}

func (p *VmNicsServiceListResponse) MustNics() *NicSlice {
	if p.nics == nil {
		panic("nics in response does not exist")
	}
	return p.nics
}

//
// Returns the list of NICs of the virtual machine.
// The order of the returned list of NICs isn't guaranteed.
//
func (p *VmNicsService) List() *VmNicsServiceListRequest {
	return &VmNicsServiceListRequest{VmNicsService: p}
}

//
//
func (op *VmNicsService) NicService(id string) *VmNicService {
	return NewVmNicService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmNicsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NicService(path), nil
	}
	return op.NicService(path[:index]).Service(path[index+1:])
}

func (op *VmNicsService) String() string {
	return fmt.Sprintf("VmNicsService:%s", op.path)
}

//
// This service manages a vNIC profile.
//
type VnicProfileService struct {
	BaseService
}

func NewVnicProfileService(connection *Connection, path string) *VnicProfileService {
	var result VnicProfileService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves details about a vNIC profile.
//
type VnicProfileServiceGetRequest struct {
	VnicProfileService *VnicProfileService
	header             map[string]string
	query              map[string]string
	follow             *string
}

func (p *VnicProfileServiceGetRequest) Header(key, value string) *VnicProfileServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VnicProfileServiceGetRequest) Query(key, value string) *VnicProfileServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VnicProfileServiceGetRequest) Follow(follow string) *VnicProfileServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *VnicProfileServiceGetRequest) Send() (*VnicProfileServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VnicProfileService.connection.URL(), p.VnicProfileService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VnicProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VnicProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VnicProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VnicProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VnicProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVnicProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VnicProfileServiceGetResponse{profile: result}, nil
}

func (p *VnicProfileServiceGetRequest) MustSend() *VnicProfileServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves details about a vNIC profile.
//
type VnicProfileServiceGetResponse struct {
	profile *VnicProfile
}

func (p *VnicProfileServiceGetResponse) Profile() (*VnicProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *VnicProfileServiceGetResponse) MustProfile() *VnicProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Retrieves details about a vNIC profile.
//
func (p *VnicProfileService) Get() *VnicProfileServiceGetRequest {
	return &VnicProfileServiceGetRequest{VnicProfileService: p}
}

//
// Removes the vNIC profile.
//
type VnicProfileServiceRemoveRequest struct {
	VnicProfileService *VnicProfileService
	header             map[string]string
	query              map[string]string
	async              *bool
}

func (p *VnicProfileServiceRemoveRequest) Header(key, value string) *VnicProfileServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VnicProfileServiceRemoveRequest) Query(key, value string) *VnicProfileServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VnicProfileServiceRemoveRequest) Async(async bool) *VnicProfileServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *VnicProfileServiceRemoveRequest) Send() (*VnicProfileServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VnicProfileService.connection.URL(), p.VnicProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VnicProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VnicProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VnicProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VnicProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VnicProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(VnicProfileServiceRemoveResponse), nil
}

func (p *VnicProfileServiceRemoveRequest) MustSend() *VnicProfileServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the vNIC profile.
//
type VnicProfileServiceRemoveResponse struct {
}

//
// Removes the vNIC profile.
//
func (p *VnicProfileService) Remove() *VnicProfileServiceRemoveRequest {
	return &VnicProfileServiceRemoveRequest{VnicProfileService: p}
}

//
// Updates details of a vNIC profile.
//
type VnicProfileServiceUpdateRequest struct {
	VnicProfileService *VnicProfileService
	header             map[string]string
	query              map[string]string
	async              *bool
	profile            *VnicProfile
}

func (p *VnicProfileServiceUpdateRequest) Header(key, value string) *VnicProfileServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VnicProfileServiceUpdateRequest) Query(key, value string) *VnicProfileServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VnicProfileServiceUpdateRequest) Async(async bool) *VnicProfileServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *VnicProfileServiceUpdateRequest) Profile(profile *VnicProfile) *VnicProfileServiceUpdateRequest {
	p.profile = profile
	return p
}

func (p *VnicProfileServiceUpdateRequest) Send() (*VnicProfileServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VnicProfileService.connection.URL(), p.VnicProfileService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLVnicProfileWriteOne(writer, p.profile, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VnicProfileService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VnicProfileService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VnicProfileService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VnicProfileService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VnicProfileService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVnicProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VnicProfileServiceUpdateResponse{profile: result}, nil
}

func (p *VnicProfileServiceUpdateRequest) MustSend() *VnicProfileServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates details of a vNIC profile.
//
type VnicProfileServiceUpdateResponse struct {
	profile *VnicProfile
}

func (p *VnicProfileServiceUpdateResponse) Profile() (*VnicProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *VnicProfileServiceUpdateResponse) MustProfile() *VnicProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Updates details of a vNIC profile.
//
func (p *VnicProfileService) Update() *VnicProfileServiceUpdateRequest {
	return &VnicProfileServiceUpdateRequest{VnicProfileService: p}
}

//
//
func (op *VnicProfileService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VnicProfileService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *VnicProfileService) String() string {
	return fmt.Sprintf("VnicProfileService:%s", op.path)
}

//
// Represents a readonly network filters sub-collection.
// The network filter enables to filter packets send to/from the VM's nic according to defined rules.
// For more information please refer to <<services/network_filter,NetworkFilter>> service documentation
// Network filters are supported in different versions, starting from version 3.0.
// A network filter is defined for each vnic profile.
// A vnic profile is defined for a specific network.
// A network can be assigned to several different clusters. In the future, each network will be defined in
// cluster level.
// Currently, each network is being defined at data center level. Potential network filters for each network
// are determined by the network's data center compatibility version V.
// V must be >= the network filter version in order to configure this network filter for a specific network.
// Please note, that if a network is assigned to cluster with a version supporting a network filter, the filter
// may not be available due to the data center version being smaller then the network filter's version.
// Example of listing all of the supported network filters for a specific cluster:
// [source]
// ----
// GET http://localhost:8080/ovirt-engine/api/clusters/{cluster:id}/networkfilters
// ----
// Output:
// [source,xml]
// ----
// <network_filters>
//   <network_filter id="00000019-0019-0019-0019-00000000026c">
//     <name>example-network-filter-a</name>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//       <build>-1</build>
//       <revision>-1</revision>
//     </version>
//   </network_filter>
//   <network_filter id="00000019-0019-0019-0019-00000000026b">
//     <name>example-network-filter-b</name>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//       <build>-1</build>
//       <revision>-1</revision>
//     </version>
//   </network_filter>
//   <network_filter id="00000019-0019-0019-0019-00000000026a">
//     <name>example-network-filter-a</name>
//     <version>
//       <major>3</major>
//       <minor>0</minor>
//       <build>-1</build>
//       <revision>-1</revision>
//     </version>
//   </network_filter>
// </network_filters>
// ----
//
type NetworkFiltersService struct {
	BaseService
}

func NewNetworkFiltersService(connection *Connection, path string) *NetworkFiltersService {
	var result NetworkFiltersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the representations of the network filters.
// The order of the returned list of network filters isn't guaranteed.
//
type NetworkFiltersServiceListRequest struct {
	NetworkFiltersService *NetworkFiltersService
	header                map[string]string
	query                 map[string]string
	follow                *string
}

func (p *NetworkFiltersServiceListRequest) Header(key, value string) *NetworkFiltersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkFiltersServiceListRequest) Query(key, value string) *NetworkFiltersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkFiltersServiceListRequest) Follow(follow string) *NetworkFiltersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *NetworkFiltersServiceListRequest) Send() (*NetworkFiltersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkFiltersService.connection.URL(), p.NetworkFiltersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkFiltersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkFiltersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkFiltersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkFiltersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkFiltersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkFilterReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &NetworkFiltersServiceListResponse{filters: result}, nil
}

func (p *NetworkFiltersServiceListRequest) MustSend() *NetworkFiltersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the representations of the network filters.
// The order of the returned list of network filters isn't guaranteed.
//
type NetworkFiltersServiceListResponse struct {
	filters *NetworkFilterSlice
}

func (p *NetworkFiltersServiceListResponse) Filters() (*NetworkFilterSlice, bool) {
	if p.filters != nil {
		return p.filters, true
	}
	return nil, false
}

func (p *NetworkFiltersServiceListResponse) MustFilters() *NetworkFilterSlice {
	if p.filters == nil {
		panic("filters in response does not exist")
	}
	return p.filters
}

//
// Retrieves the representations of the network filters.
// The order of the returned list of network filters isn't guaranteed.
//
func (p *NetworkFiltersService) List() *NetworkFiltersServiceListRequest {
	return &NetworkFiltersServiceListRequest{NetworkFiltersService: p}
}

//
//
func (op *NetworkFiltersService) NetworkFilterService(id string) *NetworkFilterService {
	return NewNetworkFilterService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *NetworkFiltersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NetworkFilterService(path), nil
	}
	return op.NetworkFilterService(path[:index]).Service(path[index+1:])
}

func (op *NetworkFiltersService) String() string {
	return fmt.Sprintf("NetworkFiltersService:%s", op.path)
}

//
// A service that provides information about an application installed in a virtual machine.
//
type VmApplicationService struct {
	BaseService
}

func NewVmApplicationService(connection *Connection, path string) *VmApplicationService {
	var result VmApplicationService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the information about the application.
//
type VmApplicationServiceGetRequest struct {
	VmApplicationService *VmApplicationService
	header               map[string]string
	query                map[string]string
	filter               *bool
	follow               *string
}

func (p *VmApplicationServiceGetRequest) Header(key, value string) *VmApplicationServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmApplicationServiceGetRequest) Query(key, value string) *VmApplicationServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmApplicationServiceGetRequest) Filter(filter bool) *VmApplicationServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *VmApplicationServiceGetRequest) Follow(follow string) *VmApplicationServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *VmApplicationServiceGetRequest) Send() (*VmApplicationServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmApplicationService.connection.URL(), p.VmApplicationService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmApplicationService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmApplicationService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmApplicationService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmApplicationService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmApplicationService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLApplicationReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmApplicationServiceGetResponse{application: result}, nil
}

func (p *VmApplicationServiceGetRequest) MustSend() *VmApplicationServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the information about the application.
//
type VmApplicationServiceGetResponse struct {
	application *Application
}

func (p *VmApplicationServiceGetResponse) Application() (*Application, bool) {
	if p.application != nil {
		return p.application, true
	}
	return nil, false
}

func (p *VmApplicationServiceGetResponse) MustApplication() *Application {
	if p.application == nil {
		panic("application in response does not exist")
	}
	return p.application
}

//
// Returns the information about the application.
//
func (p *VmApplicationService) Get() *VmApplicationServiceGetRequest {
	return &VmApplicationServiceGetRequest{VmApplicationService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmApplicationService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *VmApplicationService) String() string {
	return fmt.Sprintf("VmApplicationService:%s", op.path)
}

//
//
type VmNumaNodesService struct {
	BaseService
}

func NewVmNumaNodesService(connection *Connection, path string) *VmNumaNodesService {
	var result VmNumaNodesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new virtual NUMA node for the virtual machine.
// An example of creating a NUMA node:
// [source]
// ----
// POST /ovirt-engine/api/vms/c7ecd2dc/numanodes
// Accept: application/xml
// Content-type: application/xml
// ----
// The request body can contain the following:
// [source,xml]
// ----
// <vm_numa_node>
//   <cpu>
//     <cores>
//       <core>
//         <index>0</index>
//       </core>
//     </cores>
//   </cpu>
//   <index>0</index>
//   <memory>1024</memory>
// </vm_numa_node>
// ----
//
type VmNumaNodesServiceAddRequest struct {
	VmNumaNodesService *VmNumaNodesService
	header             map[string]string
	query              map[string]string
	node               *VirtualNumaNode
}

func (p *VmNumaNodesServiceAddRequest) Header(key, value string) *VmNumaNodesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmNumaNodesServiceAddRequest) Query(key, value string) *VmNumaNodesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmNumaNodesServiceAddRequest) Node(node *VirtualNumaNode) *VmNumaNodesServiceAddRequest {
	p.node = node
	return p
}

func (p *VmNumaNodesServiceAddRequest) Send() (*VmNumaNodesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmNumaNodesService.connection.URL(), p.VmNumaNodesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLVirtualNumaNodeWriteOne(writer, p.node, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmNumaNodesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmNumaNodesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmNumaNodesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmNumaNodesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmNumaNodesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVirtualNumaNodeReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmNumaNodesServiceAddResponse{node: result}, nil
}

func (p *VmNumaNodesServiceAddRequest) MustSend() *VmNumaNodesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new virtual NUMA node for the virtual machine.
// An example of creating a NUMA node:
// [source]
// ----
// POST /ovirt-engine/api/vms/c7ecd2dc/numanodes
// Accept: application/xml
// Content-type: application/xml
// ----
// The request body can contain the following:
// [source,xml]
// ----
// <vm_numa_node>
//   <cpu>
//     <cores>
//       <core>
//         <index>0</index>
//       </core>
//     </cores>
//   </cpu>
//   <index>0</index>
//   <memory>1024</memory>
// </vm_numa_node>
// ----
//
type VmNumaNodesServiceAddResponse struct {
	node *VirtualNumaNode
}

func (p *VmNumaNodesServiceAddResponse) Node() (*VirtualNumaNode, bool) {
	if p.node != nil {
		return p.node, true
	}
	return nil, false
}

func (p *VmNumaNodesServiceAddResponse) MustNode() *VirtualNumaNode {
	if p.node == nil {
		panic("node in response does not exist")
	}
	return p.node
}

//
// Creates a new virtual NUMA node for the virtual machine.
// An example of creating a NUMA node:
// [source]
// ----
// POST /ovirt-engine/api/vms/c7ecd2dc/numanodes
// Accept: application/xml
// Content-type: application/xml
// ----
// The request body can contain the following:
// [source,xml]
// ----
// <vm_numa_node>
//   <cpu>
//     <cores>
//       <core>
//         <index>0</index>
//       </core>
//     </cores>
//   </cpu>
//   <index>0</index>
//   <memory>1024</memory>
// </vm_numa_node>
// ----
//
func (p *VmNumaNodesService) Add() *VmNumaNodesServiceAddRequest {
	return &VmNumaNodesServiceAddRequest{VmNumaNodesService: p}
}

//
// Lists virtual NUMA nodes of a virtual machine.
// The order of the returned list of NUMA nodes isn't guaranteed.
//
type VmNumaNodesServiceListRequest struct {
	VmNumaNodesService *VmNumaNodesService
	header             map[string]string
	query              map[string]string
	follow             *string
	max                *int64
}

func (p *VmNumaNodesServiceListRequest) Header(key, value string) *VmNumaNodesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmNumaNodesServiceListRequest) Query(key, value string) *VmNumaNodesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmNumaNodesServiceListRequest) Follow(follow string) *VmNumaNodesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *VmNumaNodesServiceListRequest) Max(max int64) *VmNumaNodesServiceListRequest {
	p.max = &max
	return p
}

func (p *VmNumaNodesServiceListRequest) Send() (*VmNumaNodesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmNumaNodesService.connection.URL(), p.VmNumaNodesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmNumaNodesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmNumaNodesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmNumaNodesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmNumaNodesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmNumaNodesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVirtualNumaNodeReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &VmNumaNodesServiceListResponse{nodes: result}, nil
}

func (p *VmNumaNodesServiceListRequest) MustSend() *VmNumaNodesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists virtual NUMA nodes of a virtual machine.
// The order of the returned list of NUMA nodes isn't guaranteed.
//
type VmNumaNodesServiceListResponse struct {
	nodes *VirtualNumaNodeSlice
}

func (p *VmNumaNodesServiceListResponse) Nodes() (*VirtualNumaNodeSlice, bool) {
	if p.nodes != nil {
		return p.nodes, true
	}
	return nil, false
}

func (p *VmNumaNodesServiceListResponse) MustNodes() *VirtualNumaNodeSlice {
	if p.nodes == nil {
		panic("nodes in response does not exist")
	}
	return p.nodes
}

//
// Lists virtual NUMA nodes of a virtual machine.
// The order of the returned list of NUMA nodes isn't guaranteed.
//
func (p *VmNumaNodesService) List() *VmNumaNodesServiceListRequest {
	return &VmNumaNodesServiceListRequest{VmNumaNodesService: p}
}

//
//
func (op *VmNumaNodesService) NodeService(id string) *VmNumaNodeService {
	return NewVmNumaNodeService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmNumaNodesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NodeService(path), nil
	}
	return op.NodeService(path[:index]).Service(path[index+1:])
}

func (op *VmNumaNodesService) String() string {
	return fmt.Sprintf("VmNumaNodesService:%s", op.path)
}

//
//
type StorageService struct {
	BaseService
}

func NewStorageService(connection *Connection, path string) *StorageService {
	var result StorageService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type StorageServiceGetRequest struct {
	StorageService *StorageService
	header         map[string]string
	query          map[string]string
	follow         *string
	reportStatus   *bool
}

func (p *StorageServiceGetRequest) Header(key, value string) *StorageServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageServiceGetRequest) Query(key, value string) *StorageServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageServiceGetRequest) Follow(follow string) *StorageServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *StorageServiceGetRequest) ReportStatus(reportStatus bool) *StorageServiceGetRequest {
	p.reportStatus = &reportStatus
	return p
}

func (p *StorageServiceGetRequest) Send() (*StorageServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageService.connection.URL(), p.StorageService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.reportStatus != nil {
		values["report_status"] = []string{fmt.Sprintf("%v", *p.reportStatus)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostStorageReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageServiceGetResponse{storage: result}, nil
}

func (p *StorageServiceGetRequest) MustSend() *StorageServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type StorageServiceGetResponse struct {
	storage *HostStorage
}

func (p *StorageServiceGetResponse) Storage() (*HostStorage, bool) {
	if p.storage != nil {
		return p.storage, true
	}
	return nil, false
}

func (p *StorageServiceGetResponse) MustStorage() *HostStorage {
	if p.storage == nil {
		panic("storage in response does not exist")
	}
	return p.storage
}

//
//
func (p *StorageService) Get() *StorageServiceGetRequest {
	return &StorageServiceGetRequest{StorageService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *StorageService) String() string {
	return fmt.Sprintf("StorageService:%s", op.path)
}

//
//
type UnmanagedNetworkService struct {
	BaseService
}

func NewUnmanagedNetworkService(connection *Connection, path string) *UnmanagedNetworkService {
	var result UnmanagedNetworkService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type UnmanagedNetworkServiceGetRequest struct {
	UnmanagedNetworkService *UnmanagedNetworkService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
}

func (p *UnmanagedNetworkServiceGetRequest) Header(key, value string) *UnmanagedNetworkServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *UnmanagedNetworkServiceGetRequest) Query(key, value string) *UnmanagedNetworkServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *UnmanagedNetworkServiceGetRequest) Follow(follow string) *UnmanagedNetworkServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *UnmanagedNetworkServiceGetRequest) Send() (*UnmanagedNetworkServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.UnmanagedNetworkService.connection.URL(), p.UnmanagedNetworkService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.UnmanagedNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.UnmanagedNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.UnmanagedNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.UnmanagedNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.UnmanagedNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLUnmanagedNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &UnmanagedNetworkServiceGetResponse{network: result}, nil
}

func (p *UnmanagedNetworkServiceGetRequest) MustSend() *UnmanagedNetworkServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type UnmanagedNetworkServiceGetResponse struct {
	network *UnmanagedNetwork
}

func (p *UnmanagedNetworkServiceGetResponse) Network() (*UnmanagedNetwork, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *UnmanagedNetworkServiceGetResponse) MustNetwork() *UnmanagedNetwork {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
//
func (p *UnmanagedNetworkService) Get() *UnmanagedNetworkServiceGetRequest {
	return &UnmanagedNetworkServiceGetRequest{UnmanagedNetworkService: p}
}

//
//
type UnmanagedNetworkServiceRemoveRequest struct {
	UnmanagedNetworkService *UnmanagedNetworkService
	header                  map[string]string
	query                   map[string]string
	async                   *bool
}

func (p *UnmanagedNetworkServiceRemoveRequest) Header(key, value string) *UnmanagedNetworkServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *UnmanagedNetworkServiceRemoveRequest) Query(key, value string) *UnmanagedNetworkServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *UnmanagedNetworkServiceRemoveRequest) Async(async bool) *UnmanagedNetworkServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *UnmanagedNetworkServiceRemoveRequest) Send() (*UnmanagedNetworkServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.UnmanagedNetworkService.connection.URL(), p.UnmanagedNetworkService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.UnmanagedNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.UnmanagedNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.UnmanagedNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.UnmanagedNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.UnmanagedNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(UnmanagedNetworkServiceRemoveResponse), nil
}

func (p *UnmanagedNetworkServiceRemoveRequest) MustSend() *UnmanagedNetworkServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type UnmanagedNetworkServiceRemoveResponse struct {
}

//
//
func (p *UnmanagedNetworkService) Remove() *UnmanagedNetworkServiceRemoveRequest {
	return &UnmanagedNetworkServiceRemoveRequest{UnmanagedNetworkService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *UnmanagedNetworkService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *UnmanagedNetworkService) String() string {
	return fmt.Sprintf("UnmanagedNetworkService:%s", op.path)
}

//
//
type AttachedStorageDomainService struct {
	BaseService
}

func NewAttachedStorageDomainService(connection *Connection, path string) *AttachedStorageDomainService {
	var result AttachedStorageDomainService
	result.connection = connection
	result.path = path
	return &result
}

//
// This operation activates an attached storage domain.
// Once the storage domain is activated it is ready for use with the data center.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/activate
// ----
// The activate action does not take any action specific parameters,
// so the request body should contain an empty `action`:
// [source,xml]
// ----
// <action/>
// ----
//
type AttachedStorageDomainServiceActivateRequest struct {
	AttachedStorageDomainService *AttachedStorageDomainService
	header                       map[string]string
	query                        map[string]string
	async                        *bool
}

func (p *AttachedStorageDomainServiceActivateRequest) Header(key, value string) *AttachedStorageDomainServiceActivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainServiceActivateRequest) Query(key, value string) *AttachedStorageDomainServiceActivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainServiceActivateRequest) Async(async bool) *AttachedStorageDomainServiceActivateRequest {
	p.async = &async
	return p
}

func (p *AttachedStorageDomainServiceActivateRequest) Send() (*AttachedStorageDomainServiceActivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/activate", p.AttachedStorageDomainService.connection.URL(), p.AttachedStorageDomainService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainServiceActivateResponse), nil
}

func (p *AttachedStorageDomainServiceActivateRequest) MustSend() *AttachedStorageDomainServiceActivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation activates an attached storage domain.
// Once the storage domain is activated it is ready for use with the data center.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/activate
// ----
// The activate action does not take any action specific parameters,
// so the request body should contain an empty `action`:
// [source,xml]
// ----
// <action/>
// ----
//
type AttachedStorageDomainServiceActivateResponse struct {
}

//
// This operation activates an attached storage domain.
// Once the storage domain is activated it is ready for use with the data center.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/activate
// ----
// The activate action does not take any action specific parameters,
// so the request body should contain an empty `action`:
// [source,xml]
// ----
// <action/>
// ----
//
func (p *AttachedStorageDomainService) Activate() *AttachedStorageDomainServiceActivateRequest {
	return &AttachedStorageDomainServiceActivateRequest{AttachedStorageDomainService: p}
}

//
// This operation deactivates an attached storage domain.
// Once the storage domain is deactivated it will not be used with the data center.
// For example, to deactivate storage domain `456`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/deactivate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// If the `force` parameter is `true` then the operation will succeed, even if the OVF update which takes place
// before the deactivation of the storage domain failed. If the `force` parameter is `false` and the OVF update failed,
// the deactivation of the storage domain will also fail.
//
type AttachedStorageDomainServiceDeactivateRequest struct {
	AttachedStorageDomainService *AttachedStorageDomainService
	header                       map[string]string
	query                        map[string]string
	async                        *bool
	force                        *bool
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Header(key, value string) *AttachedStorageDomainServiceDeactivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Query(key, value string) *AttachedStorageDomainServiceDeactivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Async(async bool) *AttachedStorageDomainServiceDeactivateRequest {
	p.async = &async
	return p
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Force(force bool) *AttachedStorageDomainServiceDeactivateRequest {
	p.force = &force
	return p
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Send() (*AttachedStorageDomainServiceDeactivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/deactivate", p.AttachedStorageDomainService.connection.URL(), p.AttachedStorageDomainService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainServiceDeactivateResponse), nil
}

func (p *AttachedStorageDomainServiceDeactivateRequest) MustSend() *AttachedStorageDomainServiceDeactivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation deactivates an attached storage domain.
// Once the storage domain is deactivated it will not be used with the data center.
// For example, to deactivate storage domain `456`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/deactivate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// If the `force` parameter is `true` then the operation will succeed, even if the OVF update which takes place
// before the deactivation of the storage domain failed. If the `force` parameter is `false` and the OVF update failed,
// the deactivation of the storage domain will also fail.
//
type AttachedStorageDomainServiceDeactivateResponse struct {
}

//
// This operation deactivates an attached storage domain.
// Once the storage domain is deactivated it will not be used with the data center.
// For example, to deactivate storage domain `456`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/deactivate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// If the `force` parameter is `true` then the operation will succeed, even if the OVF update which takes place
// before the deactivation of the storage domain failed. If the `force` parameter is `false` and the OVF update failed,
// the deactivation of the storage domain will also fail.
//
func (p *AttachedStorageDomainService) Deactivate() *AttachedStorageDomainServiceDeactivateRequest {
	return &AttachedStorageDomainServiceDeactivateRequest{AttachedStorageDomainService: p}
}

//
//
type AttachedStorageDomainServiceGetRequest struct {
	AttachedStorageDomainService *AttachedStorageDomainService
	header                       map[string]string
	query                        map[string]string
	follow                       *string
}

func (p *AttachedStorageDomainServiceGetRequest) Header(key, value string) *AttachedStorageDomainServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainServiceGetRequest) Query(key, value string) *AttachedStorageDomainServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainServiceGetRequest) Follow(follow string) *AttachedStorageDomainServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AttachedStorageDomainServiceGetRequest) Send() (*AttachedStorageDomainServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainService.connection.URL(), p.AttachedStorageDomainService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageDomainReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainServiceGetResponse{storageDomain: result}, nil
}

func (p *AttachedStorageDomainServiceGetRequest) MustSend() *AttachedStorageDomainServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AttachedStorageDomainServiceGetResponse struct {
	storageDomain *StorageDomain
}

func (p *AttachedStorageDomainServiceGetResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *AttachedStorageDomainServiceGetResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
//
func (p *AttachedStorageDomainService) Get() *AttachedStorageDomainServiceGetRequest {
	return &AttachedStorageDomainServiceGetRequest{AttachedStorageDomainService: p}
}

//
//
type AttachedStorageDomainServiceRemoveRequest struct {
	AttachedStorageDomainService *AttachedStorageDomainService
	header                       map[string]string
	query                        map[string]string
	async                        *bool
}

func (p *AttachedStorageDomainServiceRemoveRequest) Header(key, value string) *AttachedStorageDomainServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainServiceRemoveRequest) Query(key, value string) *AttachedStorageDomainServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainServiceRemoveRequest) Async(async bool) *AttachedStorageDomainServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AttachedStorageDomainServiceRemoveRequest) Send() (*AttachedStorageDomainServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainService.connection.URL(), p.AttachedStorageDomainService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AttachedStorageDomainServiceRemoveResponse), nil
}

func (p *AttachedStorageDomainServiceRemoveRequest) MustSend() *AttachedStorageDomainServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AttachedStorageDomainServiceRemoveResponse struct {
}

//
//
func (p *AttachedStorageDomainService) Remove() *AttachedStorageDomainServiceRemoveRequest {
	return &AttachedStorageDomainServiceRemoveRequest{AttachedStorageDomainService: p}
}

//
//
func (op *AttachedStorageDomainService) DisksService() *AttachedStorageDomainDisksService {
	return NewAttachedStorageDomainDisksService(op.connection, fmt.Sprintf("%s/disks", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AttachedStorageDomainService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "disks" {
		return op.DisksService(), nil
	}
	if strings.HasPrefix(path, "disks/") {
		return op.DisksService().Service(path[6:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AttachedStorageDomainService) String() string {
	return fmt.Sprintf("AttachedStorageDomainService:%s", op.path)
}

//
// Represents a permits sub-collection of the specific role.
//
type PermitsService struct {
	BaseService
}

func NewPermitsService(connection *Connection, path string) *PermitsService {
	var result PermitsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a permit to the role. The permit name can be retrieved from the <<services/cluster_levels>> service.
// For example to assign a permit `create_vm` to the role with id `123` send a request like this:
// ....
// POST /ovirt-engine/api/roles/123/permits
// ....
// With a request body like this:
// [source,xml]
// ----
// <permit>
//   <name>create_vm</name>
// </permit>
// ----
//
type PermitsServiceAddRequest struct {
	PermitsService *PermitsService
	header         map[string]string
	query          map[string]string
	permit         *Permit
}

func (p *PermitsServiceAddRequest) Header(key, value string) *PermitsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *PermitsServiceAddRequest) Query(key, value string) *PermitsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *PermitsServiceAddRequest) Permit(permit *Permit) *PermitsServiceAddRequest {
	p.permit = permit
	return p
}

func (p *PermitsServiceAddRequest) Send() (*PermitsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.PermitsService.connection.URL(), p.PermitsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLPermitWriteOne(writer, p.permit, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.PermitsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.PermitsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.PermitsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.PermitsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.PermitsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLPermitReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &PermitsServiceAddResponse{permit: result}, nil
}

func (p *PermitsServiceAddRequest) MustSend() *PermitsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a permit to the role. The permit name can be retrieved from the <<services/cluster_levels>> service.
// For example to assign a permit `create_vm` to the role with id `123` send a request like this:
// ....
// POST /ovirt-engine/api/roles/123/permits
// ....
// With a request body like this:
// [source,xml]
// ----
// <permit>
//   <name>create_vm</name>
// </permit>
// ----
//
type PermitsServiceAddResponse struct {
	permit *Permit
}

func (p *PermitsServiceAddResponse) Permit() (*Permit, bool) {
	if p.permit != nil {
		return p.permit, true
	}
	return nil, false
}

func (p *PermitsServiceAddResponse) MustPermit() *Permit {
	if p.permit == nil {
		panic("permit in response does not exist")
	}
	return p.permit
}

//
// Adds a permit to the role. The permit name can be retrieved from the <<services/cluster_levels>> service.
// For example to assign a permit `create_vm` to the role with id `123` send a request like this:
// ....
// POST /ovirt-engine/api/roles/123/permits
// ....
// With a request body like this:
// [source,xml]
// ----
// <permit>
//   <name>create_vm</name>
// </permit>
// ----
//
func (p *PermitsService) Add() *PermitsServiceAddRequest {
	return &PermitsServiceAddRequest{PermitsService: p}
}

//
// List the permits of the role.
// For example to list the permits of the role with the id `123` send a request like this:
// ....
// GET /ovirt-engine/api/roles/123/permits
// ....
// [source,xml]
// ----
// <permits>
//   <permit href="/ovirt-engine/api/roles/123/permits/5" id="5">
//     <name>change_vm_cd</name>
//     <administrative>false</administrative>
//     <role href="/ovirt-engine/api/roles/123" id="123"/>
//   </permit>
//   <permit href="/ovirt-engine/api/roles/123/permits/7" id="7">
//     <name>connect_to_vm</name>
//     <administrative>false</administrative>
//     <role href="/ovirt-engine/api/roles/123" id="123"/>
//   </permit>
// </permits>
// ----
// The order of the returned list of permits isn't guaranteed.
//
type PermitsServiceListRequest struct {
	PermitsService *PermitsService
	header         map[string]string
	query          map[string]string
	follow         *string
	max            *int64
}

func (p *PermitsServiceListRequest) Header(key, value string) *PermitsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *PermitsServiceListRequest) Query(key, value string) *PermitsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *PermitsServiceListRequest) Follow(follow string) *PermitsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *PermitsServiceListRequest) Max(max int64) *PermitsServiceListRequest {
	p.max = &max
	return p
}

func (p *PermitsServiceListRequest) Send() (*PermitsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.PermitsService.connection.URL(), p.PermitsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.PermitsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.PermitsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.PermitsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.PermitsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.PermitsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLPermitReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &PermitsServiceListResponse{permits: result}, nil
}

func (p *PermitsServiceListRequest) MustSend() *PermitsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List the permits of the role.
// For example to list the permits of the role with the id `123` send a request like this:
// ....
// GET /ovirt-engine/api/roles/123/permits
// ....
// [source,xml]
// ----
// <permits>
//   <permit href="/ovirt-engine/api/roles/123/permits/5" id="5">
//     <name>change_vm_cd</name>
//     <administrative>false</administrative>
//     <role href="/ovirt-engine/api/roles/123" id="123"/>
//   </permit>
//   <permit href="/ovirt-engine/api/roles/123/permits/7" id="7">
//     <name>connect_to_vm</name>
//     <administrative>false</administrative>
//     <role href="/ovirt-engine/api/roles/123" id="123"/>
//   </permit>
// </permits>
// ----
// The order of the returned list of permits isn't guaranteed.
//
type PermitsServiceListResponse struct {
	permits *PermitSlice
}

func (p *PermitsServiceListResponse) Permits() (*PermitSlice, bool) {
	if p.permits != nil {
		return p.permits, true
	}
	return nil, false
}

func (p *PermitsServiceListResponse) MustPermits() *PermitSlice {
	if p.permits == nil {
		panic("permits in response does not exist")
	}
	return p.permits
}

//
// List the permits of the role.
// For example to list the permits of the role with the id `123` send a request like this:
// ....
// GET /ovirt-engine/api/roles/123/permits
// ....
// [source,xml]
// ----
// <permits>
//   <permit href="/ovirt-engine/api/roles/123/permits/5" id="5">
//     <name>change_vm_cd</name>
//     <administrative>false</administrative>
//     <role href="/ovirt-engine/api/roles/123" id="123"/>
//   </permit>
//   <permit href="/ovirt-engine/api/roles/123/permits/7" id="7">
//     <name>connect_to_vm</name>
//     <administrative>false</administrative>
//     <role href="/ovirt-engine/api/roles/123" id="123"/>
//   </permit>
// </permits>
// ----
// The order of the returned list of permits isn't guaranteed.
//
func (p *PermitsService) List() *PermitsServiceListRequest {
	return &PermitsServiceListRequest{PermitsService: p}
}

//
// Sub-resource locator method, returns individual permit resource on which the remainder of the URI is dispatched.
//
func (op *PermitsService) PermitService(id string) *PermitService {
	return NewPermitService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *PermitsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.PermitService(path), nil
	}
	return op.PermitService(path[:index]).Service(path[index+1:])
}

func (op *PermitsService) String() string {
	return fmt.Sprintf("PermitsService:%s", op.path)
}

//
// Manages the set of connections to storage servers that exist in a storage domain.
//
type StorageDomainServerConnectionsService struct {
	BaseService
}

func NewStorageDomainServerConnectionsService(connection *Connection, path string) *StorageDomainServerConnectionsService {
	var result StorageDomainServerConnectionsService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type StorageDomainServerConnectionsServiceAddRequest struct {
	StorageDomainServerConnectionsService *StorageDomainServerConnectionsService
	header                                map[string]string
	query                                 map[string]string
	connection                            *StorageConnection
}

func (p *StorageDomainServerConnectionsServiceAddRequest) Header(key, value string) *StorageDomainServerConnectionsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServerConnectionsServiceAddRequest) Query(key, value string) *StorageDomainServerConnectionsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServerConnectionsServiceAddRequest) Connection(connection *StorageConnection) *StorageDomainServerConnectionsServiceAddRequest {
	p.connection = connection
	return p
}

func (p *StorageDomainServerConnectionsServiceAddRequest) Send() (*StorageDomainServerConnectionsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainServerConnectionsService.connection.URL(), p.StorageDomainServerConnectionsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLStorageConnectionWriteOne(writer, p.connection, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageConnectionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageDomainServerConnectionsServiceAddResponse{connection: result}, nil
}

func (p *StorageDomainServerConnectionsServiceAddRequest) MustSend() *StorageDomainServerConnectionsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type StorageDomainServerConnectionsServiceAddResponse struct {
	connection *StorageConnection
}

func (p *StorageDomainServerConnectionsServiceAddResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageDomainServerConnectionsServiceAddResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
//
func (p *StorageDomainServerConnectionsService) Add() *StorageDomainServerConnectionsServiceAddRequest {
	return &StorageDomainServerConnectionsServiceAddRequest{StorageDomainServerConnectionsService: p}
}

//
// Returns the list of connections to storage servers that existin the storage domain.
// The order of the returned list of connections isn't guaranteed.
//
type StorageDomainServerConnectionsServiceListRequest struct {
	StorageDomainServerConnectionsService *StorageDomainServerConnectionsService
	header                                map[string]string
	query                                 map[string]string
	follow                                *string
	max                                   *int64
}

func (p *StorageDomainServerConnectionsServiceListRequest) Header(key, value string) *StorageDomainServerConnectionsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServerConnectionsServiceListRequest) Query(key, value string) *StorageDomainServerConnectionsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServerConnectionsServiceListRequest) Follow(follow string) *StorageDomainServerConnectionsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *StorageDomainServerConnectionsServiceListRequest) Max(max int64) *StorageDomainServerConnectionsServiceListRequest {
	p.max = &max
	return p
}

func (p *StorageDomainServerConnectionsServiceListRequest) Send() (*StorageDomainServerConnectionsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainServerConnectionsService.connection.URL(), p.StorageDomainServerConnectionsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainServerConnectionsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainServerConnectionsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainServerConnectionsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainServerConnectionsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainServerConnectionsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageConnectionReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &StorageDomainServerConnectionsServiceListResponse{connections: result}, nil
}

func (p *StorageDomainServerConnectionsServiceListRequest) MustSend() *StorageDomainServerConnectionsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of connections to storage servers that existin the storage domain.
// The order of the returned list of connections isn't guaranteed.
//
type StorageDomainServerConnectionsServiceListResponse struct {
	connections *StorageConnectionSlice
}

func (p *StorageDomainServerConnectionsServiceListResponse) Connections() (*StorageConnectionSlice, bool) {
	if p.connections != nil {
		return p.connections, true
	}
	return nil, false
}

func (p *StorageDomainServerConnectionsServiceListResponse) MustConnections() *StorageConnectionSlice {
	if p.connections == nil {
		panic("connections in response does not exist")
	}
	return p.connections
}

//
// Returns the list of connections to storage servers that existin the storage domain.
// The order of the returned list of connections isn't guaranteed.
//
func (p *StorageDomainServerConnectionsService) List() *StorageDomainServerConnectionsServiceListRequest {
	return &StorageDomainServerConnectionsServiceListRequest{StorageDomainServerConnectionsService: p}
}

//
//
func (op *StorageDomainServerConnectionsService) ConnectionService(id string) *StorageDomainServerConnectionService {
	return NewStorageDomainServerConnectionService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageDomainServerConnectionsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ConnectionService(path), nil
	}
	return op.ConnectionService(path[:index]).Service(path[index+1:])
}

func (op *StorageDomainServerConnectionsService) String() string {
	return fmt.Sprintf("StorageDomainServerConnectionsService:%s", op.path)
}

//
// Manages the set of storage domains in the system.
//
type StorageDomainsService struct {
	BaseService
}

func NewStorageDomainsService(connection *Connection, path string) *StorageDomainsService {
	var result StorageDomainsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a new storage domain.
// Creation of a new <<types/storage_domain,StorageDomain>> requires the `name`, `type`, `host`, and `storage`
// attributes. Identify the `host` attribute with the `id` or `name` attributes. In {product-name} 3.6 and
// later you can enable the wipe after delete option by default on the storage domain. To configure this, specify
// `wipe_after_delete` in the POST request. This option can be edited after the domain is created, but doing so will
// not change the wipe after delete property of disks that already exist.
// To add a new storage domain with specified `name`, `type`, `storage.type`, `storage.address`, and `storage.path`,
// and using a host with an id `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageDomains
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>mydata</name>
//   <type>data</type>
//   <storage>
//     <type>nfs</type>
//     <address>mynfs.example.com</address>
//     <path>/exports/mydata</path>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
// To create a new NFS ISO storage domain send a request like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>myisos</name>
//   <type>iso</type>
//   <storage>
//     <type>nfs</type>
//     <address>mynfs.example.com</address>
//     <path>/export/myisos</path>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
// To create a new iSCSI storage domain send a request like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>myiscsi</name>
//   <type>data</type>
//   <storage>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="3600144f09dbd050000004eedbd340001"/>
//       <logical_unit id="3600144f09dbd050000004eedbd340002"/>
//     </logical_units>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
//
type StorageDomainsServiceAddRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	storageDomain         *StorageDomain
}

func (p *StorageDomainsServiceAddRequest) Header(key, value string) *StorageDomainsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceAddRequest) Query(key, value string) *StorageDomainsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceAddRequest) StorageDomain(storageDomain *StorageDomain) *StorageDomainsServiceAddRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *StorageDomainsServiceAddRequest) Send() (*StorageDomainsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLStorageDomainWriteOne(writer, p.storageDomain, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageDomainReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageDomainsServiceAddResponse{storageDomain: result}, nil
}

func (p *StorageDomainsServiceAddRequest) MustSend() *StorageDomainsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a new storage domain.
// Creation of a new <<types/storage_domain,StorageDomain>> requires the `name`, `type`, `host`, and `storage`
// attributes. Identify the `host` attribute with the `id` or `name` attributes. In {product-name} 3.6 and
// later you can enable the wipe after delete option by default on the storage domain. To configure this, specify
// `wipe_after_delete` in the POST request. This option can be edited after the domain is created, but doing so will
// not change the wipe after delete property of disks that already exist.
// To add a new storage domain with specified `name`, `type`, `storage.type`, `storage.address`, and `storage.path`,
// and using a host with an id `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageDomains
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>mydata</name>
//   <type>data</type>
//   <storage>
//     <type>nfs</type>
//     <address>mynfs.example.com</address>
//     <path>/exports/mydata</path>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
// To create a new NFS ISO storage domain send a request like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>myisos</name>
//   <type>iso</type>
//   <storage>
//     <type>nfs</type>
//     <address>mynfs.example.com</address>
//     <path>/export/myisos</path>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
// To create a new iSCSI storage domain send a request like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>myiscsi</name>
//   <type>data</type>
//   <storage>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="3600144f09dbd050000004eedbd340001"/>
//       <logical_unit id="3600144f09dbd050000004eedbd340002"/>
//     </logical_units>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
//
type StorageDomainsServiceAddResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainsServiceAddResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainsServiceAddResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Adds a new storage domain.
// Creation of a new <<types/storage_domain,StorageDomain>> requires the `name`, `type`, `host`, and `storage`
// attributes. Identify the `host` attribute with the `id` or `name` attributes. In {product-name} 3.6 and
// later you can enable the wipe after delete option by default on the storage domain. To configure this, specify
// `wipe_after_delete` in the POST request. This option can be edited after the domain is created, but doing so will
// not change the wipe after delete property of disks that already exist.
// To add a new storage domain with specified `name`, `type`, `storage.type`, `storage.address`, and `storage.path`,
// and using a host with an id `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageDomains
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>mydata</name>
//   <type>data</type>
//   <storage>
//     <type>nfs</type>
//     <address>mynfs.example.com</address>
//     <path>/exports/mydata</path>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
// To create a new NFS ISO storage domain send a request like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>myisos</name>
//   <type>iso</type>
//   <storage>
//     <type>nfs</type>
//     <address>mynfs.example.com</address>
//     <path>/export/myisos</path>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
// To create a new iSCSI storage domain send a request like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>myiscsi</name>
//   <type>data</type>
//   <storage>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="3600144f09dbd050000004eedbd340001"/>
//       <logical_unit id="3600144f09dbd050000004eedbd340002"/>
//     </logical_units>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
//
func (p *StorageDomainsService) Add() *StorageDomainsServiceAddRequest {
	return &StorageDomainsServiceAddRequest{StorageDomainsService: p}
}

//
// Import an existing block storage domain to the system using the targets already connected to the host.
//
type StorageDomainsServiceAddBlockDomainRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	storageDomain         *StorageDomain
}

func (p *StorageDomainsServiceAddBlockDomainRequest) Header(key, value string) *StorageDomainsServiceAddBlockDomainRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceAddBlockDomainRequest) Query(key, value string) *StorageDomainsServiceAddBlockDomainRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceAddBlockDomainRequest) StorageDomain(storageDomain *StorageDomain) *StorageDomainsServiceAddBlockDomainRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *StorageDomainsServiceAddBlockDomainRequest) Send() (*StorageDomainsServiceAddBlockDomainResponse, error) {
	rawURL := fmt.Sprintf("%s%s/blockdomain", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustStorageDomain()
	return &StorageDomainsServiceAddBlockDomainResponse{storageDomain: result}, nil
}

func (p *StorageDomainsServiceAddBlockDomainRequest) MustSend() *StorageDomainsServiceAddBlockDomainResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Import an existing block storage domain to the system using the targets already connected to the host.
//
type StorageDomainsServiceAddBlockDomainResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainsServiceAddBlockDomainResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainsServiceAddBlockDomainResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Import an existing block storage domain to the system using the targets already connected to the host.
//
func (p *StorageDomainsService) AddBlockDomain() *StorageDomainsServiceAddBlockDomainRequest {
	return &StorageDomainsServiceAddBlockDomainRequest{StorageDomainsService: p}
}

//
// Add a new storage domain to the system using the storage on the given host and path.
//
type StorageDomainsServiceAddByPathRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	storageDomain         *StorageDomain
}

func (p *StorageDomainsServiceAddByPathRequest) Header(key, value string) *StorageDomainsServiceAddByPathRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceAddByPathRequest) Query(key, value string) *StorageDomainsServiceAddByPathRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceAddByPathRequest) StorageDomain(storageDomain *StorageDomain) *StorageDomainsServiceAddByPathRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *StorageDomainsServiceAddByPathRequest) Send() (*StorageDomainsServiceAddByPathResponse, error) {
	rawURL := fmt.Sprintf("%s%s/bypath", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustStorageDomain()
	return &StorageDomainsServiceAddByPathResponse{storageDomain: result}, nil
}

func (p *StorageDomainsServiceAddByPathRequest) MustSend() *StorageDomainsServiceAddByPathResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new storage domain to the system using the storage on the given host and path.
//
type StorageDomainsServiceAddByPathResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainsServiceAddByPathResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainsServiceAddByPathResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Add a new storage domain to the system using the storage on the given host and path.
//
func (p *StorageDomainsService) AddByPath() *StorageDomainsServiceAddByPathRequest {
	return &StorageDomainsServiceAddByPathRequest{StorageDomainsService: p}
}

//
// Add a new storage domain to the system using a direct LUN.
//
type StorageDomainsServiceAddDirectLunRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	storageDomain         *StorageDomain
}

func (p *StorageDomainsServiceAddDirectLunRequest) Header(key, value string) *StorageDomainsServiceAddDirectLunRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceAddDirectLunRequest) Query(key, value string) *StorageDomainsServiceAddDirectLunRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceAddDirectLunRequest) StorageDomain(storageDomain *StorageDomain) *StorageDomainsServiceAddDirectLunRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *StorageDomainsServiceAddDirectLunRequest) Send() (*StorageDomainsServiceAddDirectLunResponse, error) {
	rawURL := fmt.Sprintf("%s%s/directlun", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustStorageDomain()
	return &StorageDomainsServiceAddDirectLunResponse{storageDomain: result}, nil
}

func (p *StorageDomainsServiceAddDirectLunRequest) MustSend() *StorageDomainsServiceAddDirectLunResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new storage domain to the system using a direct LUN.
//
type StorageDomainsServiceAddDirectLunResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainsServiceAddDirectLunResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainsServiceAddDirectLunResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Add a new storage domain to the system using a direct LUN.
//
func (p *StorageDomainsService) AddDirectLun() *StorageDomainsServiceAddDirectLunRequest {
	return &StorageDomainsServiceAddDirectLunRequest{StorageDomainsService: p}
}

//
// Add a new storage domain to the system using Gluster or POSIX FS storage.
//
type StorageDomainsServiceAddGlusterOrPostfsRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	storageDomain         *StorageDomain
}

func (p *StorageDomainsServiceAddGlusterOrPostfsRequest) Header(key, value string) *StorageDomainsServiceAddGlusterOrPostfsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceAddGlusterOrPostfsRequest) Query(key, value string) *StorageDomainsServiceAddGlusterOrPostfsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceAddGlusterOrPostfsRequest) StorageDomain(storageDomain *StorageDomain) *StorageDomainsServiceAddGlusterOrPostfsRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *StorageDomainsServiceAddGlusterOrPostfsRequest) Send() (*StorageDomainsServiceAddGlusterOrPostfsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/glusterorpostfs", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustStorageDomain()
	return &StorageDomainsServiceAddGlusterOrPostfsResponse{storageDomain: result}, nil
}

func (p *StorageDomainsServiceAddGlusterOrPostfsRequest) MustSend() *StorageDomainsServiceAddGlusterOrPostfsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new storage domain to the system using Gluster or POSIX FS storage.
//
type StorageDomainsServiceAddGlusterOrPostfsResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainsServiceAddGlusterOrPostfsResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainsServiceAddGlusterOrPostfsResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Add a new storage domain to the system using Gluster or POSIX FS storage.
//
func (p *StorageDomainsService) AddGlusterOrPostfs() *StorageDomainsServiceAddGlusterOrPostfsRequest {
	return &StorageDomainsServiceAddGlusterOrPostfsRequest{StorageDomainsService: p}
}

//
// Returns the list of storage domains in the system.
// The order of the returned list of storage domains is guaranteed only if the `sortby` clause is included
// in the `search` parameter.
//
type StorageDomainsServiceListRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	caseSensitive         *bool
	filter                *bool
	follow                *string
	max                   *int64
	search                *string
}

func (p *StorageDomainsServiceListRequest) Header(key, value string) *StorageDomainsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceListRequest) Query(key, value string) *StorageDomainsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceListRequest) CaseSensitive(caseSensitive bool) *StorageDomainsServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *StorageDomainsServiceListRequest) Filter(filter bool) *StorageDomainsServiceListRequest {
	p.filter = &filter
	return p
}

func (p *StorageDomainsServiceListRequest) Follow(follow string) *StorageDomainsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *StorageDomainsServiceListRequest) Max(max int64) *StorageDomainsServiceListRequest {
	p.max = &max
	return p
}

func (p *StorageDomainsServiceListRequest) Search(search string) *StorageDomainsServiceListRequest {
	p.search = &search
	return p
}

func (p *StorageDomainsServiceListRequest) Send() (*StorageDomainsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageDomainReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &StorageDomainsServiceListResponse{storageDomains: result}, nil
}

func (p *StorageDomainsServiceListRequest) MustSend() *StorageDomainsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of storage domains in the system.
// The order of the returned list of storage domains is guaranteed only if the `sortby` clause is included
// in the `search` parameter.
//
type StorageDomainsServiceListResponse struct {
	storageDomains *StorageDomainSlice
}

func (p *StorageDomainsServiceListResponse) StorageDomains() (*StorageDomainSlice, bool) {
	if p.storageDomains != nil {
		return p.storageDomains, true
	}
	return nil, false
}

func (p *StorageDomainsServiceListResponse) MustStorageDomains() *StorageDomainSlice {
	if p.storageDomains == nil {
		panic("storageDomains in response does not exist")
	}
	return p.storageDomains
}

//
// Returns the list of storage domains in the system.
// The order of the returned list of storage domains is guaranteed only if the `sortby` clause is included
// in the `search` parameter.
//
func (p *StorageDomainsService) List() *StorageDomainsServiceListRequest {
	return &StorageDomainsServiceListRequest{StorageDomainsService: p}
}

//
// Add a new storage domain to the system using the storage on the local host at the given path.
//
type StorageDomainsServiceAddLocalRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	storageDomain         *StorageDomain
}

func (p *StorageDomainsServiceAddLocalRequest) Header(key, value string) *StorageDomainsServiceAddLocalRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceAddLocalRequest) Query(key, value string) *StorageDomainsServiceAddLocalRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceAddLocalRequest) StorageDomain(storageDomain *StorageDomain) *StorageDomainsServiceAddLocalRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *StorageDomainsServiceAddLocalRequest) Send() (*StorageDomainsServiceAddLocalResponse, error) {
	rawURL := fmt.Sprintf("%s%s/local", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustStorageDomain()
	return &StorageDomainsServiceAddLocalResponse{storageDomain: result}, nil
}

func (p *StorageDomainsServiceAddLocalRequest) MustSend() *StorageDomainsServiceAddLocalResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new storage domain to the system using the storage on the local host at the given path.
//
type StorageDomainsServiceAddLocalResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainsServiceAddLocalResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainsServiceAddLocalResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Add a new storage domain to the system using the storage on the local host at the given path.
//
func (p *StorageDomainsService) AddLocal() *StorageDomainsServiceAddLocalRequest {
	return &StorageDomainsServiceAddLocalRequest{StorageDomainsService: p}
}

//
//
func (op *StorageDomainsService) StorageDomainService(id string) *StorageDomainService {
	return NewStorageDomainService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageDomainsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.StorageDomainService(path), nil
	}
	return op.StorageDomainService(path[:index]).Service(path[index+1:])
}

func (op *StorageDomainsService) String() string {
	return fmt.Sprintf("StorageDomainsService:%s", op.path)
}

//
// A service to manage data center networks.
//
type DataCenterNetworksService struct {
	BaseService
}

func NewDataCenterNetworksService(connection *Connection, path string) *DataCenterNetworksService {
	var result DataCenterNetworksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Create a new network in a data center.
// Post a request like in the example below to create a new network in a data center with an ID of `123`.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/networks
// ----
// Use the following example in its body:
// [source,xml]
// ----
// <network>
//   <name>mynetwork</name>
// </network>
// ----
//
type DataCenterNetworksServiceAddRequest struct {
	DataCenterNetworksService *DataCenterNetworksService
	header                    map[string]string
	query                     map[string]string
	network                   *Network
}

func (p *DataCenterNetworksServiceAddRequest) Header(key, value string) *DataCenterNetworksServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterNetworksServiceAddRequest) Query(key, value string) *DataCenterNetworksServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterNetworksServiceAddRequest) Network(network *Network) *DataCenterNetworksServiceAddRequest {
	p.network = network
	return p
}

func (p *DataCenterNetworksServiceAddRequest) Send() (*DataCenterNetworksServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCenterNetworksService.connection.URL(), p.DataCenterNetworksService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkWriteOne(writer, p.network, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterNetworksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterNetworksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterNetworksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterNetworksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterNetworksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DataCenterNetworksServiceAddResponse{network: result}, nil
}

func (p *DataCenterNetworksServiceAddRequest) MustSend() *DataCenterNetworksServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Create a new network in a data center.
// Post a request like in the example below to create a new network in a data center with an ID of `123`.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/networks
// ----
// Use the following example in its body:
// [source,xml]
// ----
// <network>
//   <name>mynetwork</name>
// </network>
// ----
//
type DataCenterNetworksServiceAddResponse struct {
	network *Network
}

func (p *DataCenterNetworksServiceAddResponse) Network() (*Network, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *DataCenterNetworksServiceAddResponse) MustNetwork() *Network {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
// Create a new network in a data center.
// Post a request like in the example below to create a new network in a data center with an ID of `123`.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/networks
// ----
// Use the following example in its body:
// [source,xml]
// ----
// <network>
//   <name>mynetwork</name>
// </network>
// ----
//
func (p *DataCenterNetworksService) Add() *DataCenterNetworksServiceAddRequest {
	return &DataCenterNetworksServiceAddRequest{DataCenterNetworksService: p}
}

//
// Lists networks in the data center.
// The order of the returned list of networks isn't guaranteed.
//
type DataCenterNetworksServiceListRequest struct {
	DataCenterNetworksService *DataCenterNetworksService
	header                    map[string]string
	query                     map[string]string
	follow                    *string
	max                       *int64
}

func (p *DataCenterNetworksServiceListRequest) Header(key, value string) *DataCenterNetworksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DataCenterNetworksServiceListRequest) Query(key, value string) *DataCenterNetworksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DataCenterNetworksServiceListRequest) Follow(follow string) *DataCenterNetworksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DataCenterNetworksServiceListRequest) Max(max int64) *DataCenterNetworksServiceListRequest {
	p.max = &max
	return p
}

func (p *DataCenterNetworksServiceListRequest) Send() (*DataCenterNetworksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DataCenterNetworksService.connection.URL(), p.DataCenterNetworksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DataCenterNetworksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DataCenterNetworksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DataCenterNetworksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DataCenterNetworksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DataCenterNetworksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DataCenterNetworksServiceListResponse{networks: result}, nil
}

func (p *DataCenterNetworksServiceListRequest) MustSend() *DataCenterNetworksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists networks in the data center.
// The order of the returned list of networks isn't guaranteed.
//
type DataCenterNetworksServiceListResponse struct {
	networks *NetworkSlice
}

func (p *DataCenterNetworksServiceListResponse) Networks() (*NetworkSlice, bool) {
	if p.networks != nil {
		return p.networks, true
	}
	return nil, false
}

func (p *DataCenterNetworksServiceListResponse) MustNetworks() *NetworkSlice {
	if p.networks == nil {
		panic("networks in response does not exist")
	}
	return p.networks
}

//
// Lists networks in the data center.
// The order of the returned list of networks isn't guaranteed.
//
func (p *DataCenterNetworksService) List() *DataCenterNetworksServiceListRequest {
	return &DataCenterNetworksServiceListRequest{DataCenterNetworksService: p}
}

//
// Access the data center network service that manages the data center network specified by an ID.
//
func (op *DataCenterNetworksService) NetworkService(id string) *DataCenterNetworkService {
	return NewDataCenterNetworkService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DataCenterNetworksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NetworkService(path), nil
	}
	return op.NetworkService(path[:index]).Service(path[index+1:])
}

func (op *DataCenterNetworksService) String() string {
	return fmt.Sprintf("DataCenterNetworksService:%s", op.path)
}

//
//
type QuotaStorageLimitService struct {
	BaseService
}

func NewQuotaStorageLimitService(connection *Connection, path string) *QuotaStorageLimitService {
	var result QuotaStorageLimitService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type QuotaStorageLimitServiceGetRequest struct {
	QuotaStorageLimitService *QuotaStorageLimitService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
}

func (p *QuotaStorageLimitServiceGetRequest) Header(key, value string) *QuotaStorageLimitServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *QuotaStorageLimitServiceGetRequest) Query(key, value string) *QuotaStorageLimitServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *QuotaStorageLimitServiceGetRequest) Follow(follow string) *QuotaStorageLimitServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *QuotaStorageLimitServiceGetRequest) Send() (*QuotaStorageLimitServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.QuotaStorageLimitService.connection.URL(), p.QuotaStorageLimitService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.QuotaStorageLimitService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.QuotaStorageLimitService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.QuotaStorageLimitService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.QuotaStorageLimitService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.QuotaStorageLimitService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLQuotaStorageLimitReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &QuotaStorageLimitServiceGetResponse{limit: result}, nil
}

func (p *QuotaStorageLimitServiceGetRequest) MustSend() *QuotaStorageLimitServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type QuotaStorageLimitServiceGetResponse struct {
	limit *QuotaStorageLimit
}

func (p *QuotaStorageLimitServiceGetResponse) Limit() (*QuotaStorageLimit, bool) {
	if p.limit != nil {
		return p.limit, true
	}
	return nil, false
}

func (p *QuotaStorageLimitServiceGetResponse) MustLimit() *QuotaStorageLimit {
	if p.limit == nil {
		panic("limit in response does not exist")
	}
	return p.limit
}

//
//
func (p *QuotaStorageLimitService) Get() *QuotaStorageLimitServiceGetRequest {
	return &QuotaStorageLimitServiceGetRequest{QuotaStorageLimitService: p}
}

//
//
type QuotaStorageLimitServiceRemoveRequest struct {
	QuotaStorageLimitService *QuotaStorageLimitService
	header                   map[string]string
	query                    map[string]string
	async                    *bool
}

func (p *QuotaStorageLimitServiceRemoveRequest) Header(key, value string) *QuotaStorageLimitServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *QuotaStorageLimitServiceRemoveRequest) Query(key, value string) *QuotaStorageLimitServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *QuotaStorageLimitServiceRemoveRequest) Async(async bool) *QuotaStorageLimitServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *QuotaStorageLimitServiceRemoveRequest) Send() (*QuotaStorageLimitServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.QuotaStorageLimitService.connection.URL(), p.QuotaStorageLimitService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.QuotaStorageLimitService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.QuotaStorageLimitService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.QuotaStorageLimitService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.QuotaStorageLimitService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.QuotaStorageLimitService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(QuotaStorageLimitServiceRemoveResponse), nil
}

func (p *QuotaStorageLimitServiceRemoveRequest) MustSend() *QuotaStorageLimitServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type QuotaStorageLimitServiceRemoveResponse struct {
}

//
//
func (p *QuotaStorageLimitService) Remove() *QuotaStorageLimitServiceRemoveRequest {
	return &QuotaStorageLimitServiceRemoveRequest{QuotaStorageLimitService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *QuotaStorageLimitService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *QuotaStorageLimitService) String() string {
	return fmt.Sprintf("QuotaStorageLimitService:%s", op.path)
}

//
//
type StorageDomainServerConnectionService struct {
	BaseService
}

func NewStorageDomainServerConnectionService(connection *Connection, path string) *StorageDomainServerConnectionService {
	var result StorageDomainServerConnectionService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type StorageDomainServerConnectionServiceGetRequest struct {
	StorageDomainServerConnectionService *StorageDomainServerConnectionService
	header                               map[string]string
	query                                map[string]string
	follow                               *string
}

func (p *StorageDomainServerConnectionServiceGetRequest) Header(key, value string) *StorageDomainServerConnectionServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServerConnectionServiceGetRequest) Query(key, value string) *StorageDomainServerConnectionServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServerConnectionServiceGetRequest) Follow(follow string) *StorageDomainServerConnectionServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *StorageDomainServerConnectionServiceGetRequest) Send() (*StorageDomainServerConnectionServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainServerConnectionService.connection.URL(), p.StorageDomainServerConnectionService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainServerConnectionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainServerConnectionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainServerConnectionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainServerConn