/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.xml;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import org.ovirt.api.metamodel.runtime.util.ArrayListWithHref;
import org.ovirt.api.metamodel.runtime.util.ListWithHref;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.engine.sdk4.internal.containers.StepContainer;
import org.ovirt.engine.sdk4.types.Step;

public class XmlStepReader {
    
    public static Step readOne(XmlReader reader) {
        // Do nothing if there aren't more tags:
        if (!reader.forward()) {
            return null;
        }
        
        // Create the object:
        StepContainer object = new StepContainer();
        
        // Process the attributes:
        for (int i = 0; i < reader.getAttributeCount(); i++) {
            String name = reader.getAttributeLocalName(i);
            String image = reader.getAttributeValue(i);
            switch (name) {
                case "href":
                object.href(image);
                break;
                case "id":
                object.id(image);
                break;
                default:
                break;
            }
        }
        
        // Process the inner elements:
        reader.next();
        while (reader.forward()) {
            String name = reader.getLocalName();
            switch (name) {
                case "comment":
                object.comment(reader.readString());
                break;
                case "description":
                object.description(reader.readString());
                break;
                case "end_time":
                object.endTime(reader.readDate());
                break;
                case "execution_host":
                object.executionHost(XmlHostReader.readOne(reader));
                break;
                case "external":
                object.external(reader.readBoolean());
                break;
                case "external_type":
                object.externalType(XmlExternalSystemTypeReader.readOne(reader));
                break;
                case "job":
                object.job(XmlJobReader.readOne(reader));
                break;
                case "name":
                object.name(reader.readString());
                break;
                case "number":
                object.number(reader.readInteger());
                break;
                case "parent_step":
                object.parentStep(XmlStepReader.readOne(reader));
                break;
                case "progress":
                object.progress(reader.readInteger());
                break;
                case "start_time":
                object.startTime(reader.readDate());
                break;
                case "statistics":
                object.statistics(XmlStatisticReader.readMany(reader));
                break;
                case "status":
                object.status(XmlStepStatusReader.readOne(reader));
                break;
                case "type":
                object.type(XmlStepEnumReader.readOne(reader));
                break;
                case "link":
                readLink(reader, object);
                break;
                default:
                reader.skip();
                break;
            }
        }
        
        // Discard the end tag:
        reader.next();
        
        return object;
    }
    
    public static Iterator<Step> iterateMany(final XmlReader reader) {
        return new Iterator<Step>() {
            private boolean first = true;
            
            @Override
            public void remove() {
                // Empty on purpose
            }
            
            @Override
            public boolean hasNext() {
                if (first) {
                    if (!reader.forward()) {
                        return false;
                    }
                    reader.next();
                    first = false;
                }
                if (!reader.forward()) {
                    reader.next();
                    return false;
                }
                return true;
            }
            
            @Override
            public Step next() {
                Step next = readOne(reader);
                if (next == null) {
                    throw new NoSuchElementException();
                }
                return next;
            }
        };
    }
    
    public static List<Step> readMany(XmlReader reader) {
        List<Step> list = new ArrayList<>();
        Iterator<Step> iterator = iterateMany(reader);
        while (iterator.hasNext()) {
            list.add(iterator.next());
        }
        return list;
    }
    
    private static void readLink(XmlReader reader, StepContainer object) {
        // Process the attributes:
        String rel = null;
        String href = null;
        for (int i = 0; i < reader.getAttributeCount(); i++) {
            String attrName = reader.getAttributeLocalName(i);
            String attrVal = reader.getAttributeValue(i);
            switch (attrName) {
                case "href":
                href = attrVal;
                break;
                case "rel":
                rel = attrVal;
                break;
                default:
                reader.skip();
                break;
            }
        }
        if (href != null) {
            ListWithHref list = new ArrayListWithHref();
            list.href(href);
            switch (rel) {
                case "statistics":
                object.statistics(list);
                break;
                default:
                break;
            }
        }
        reader.skip();
    }
}

