/*
 * Copyright 2009-2019 Ping Identity Corporation
 * All Rights Reserved.
 */
/*
 * Copyright (C) 2009-2019 Ping Identity Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License (GPLv2 only)
 * or the terms of the GNU Lesser General Public License (LGPLv2.1 only)
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses>.
 */
package com.unboundid.ldap.sdk.unboundidds.controls;



import org.testng.annotations.Test;

import com.unboundid.asn1.ASN1OctetString;
import com.unboundid.ldap.sdk.Control;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.LDAPResult;
import com.unboundid.ldap.sdk.LDAPSDKTestCase;
import com.unboundid.ldap.sdk.ResultCode;



/**
 * This class provides a set of test cases for the unsolicited cancel response
 * control.
 */
public class UnsolicitedCancelResponseControlTestCase
       extends LDAPSDKTestCase
{
  /**
   * Tests the first constructor, which does not take any arguments.
   *
   * @throws  Exception  If an unexpected problem occurs.
   */
  @Test()
  public void testConstructor1()
         throws Exception
  {
    UnsolicitedCancelResponseControl c = new UnsolicitedCancelResponseControl();
    c = c.decodeControl(c.getOID(), c.isCritical(), c.getValue());

    assertNotNull(c);

    assertNotNull(c.getOID());
    assertEquals(c.getOID(), "1.3.6.1.4.1.30221.2.5.7");

    assertFalse(c.isCritical());

    assertNull(c.getValue());

    assertNotNull(c.getControlName());

    assertNotNull(c.toString());
  }



  /**
   * Tests the {@code decodeControl} method for a control that includes a value.
   *
   * @throws  Exception  If an unexpected problem occurs.
   */
  @Test(expectedExceptions = { LDAPException.class })
  public void testDecodeHasValue()
         throws Exception
  {
    new UnsolicitedCancelResponseControl().decodeControl(
         "1.3.6.1.4.1.30221.2.5.7", false, new ASN1OctetString());
  }



  /**
   * Tests the {@code get} method with a result that does not contain an
   * unsolicited cancel response control.
   *
   * @throws  Exception  If an unexpected problem occurs.
   */
  @Test()
  public void testGetMissing()
         throws Exception
  {
    final Control[] controls = new Control[0];

    final LDAPResult r = new LDAPResult(1, ResultCode.SUCCESS);

    final UnsolicitedCancelResponseControl c =
         UnsolicitedCancelResponseControl.get(r);
    assertNull(c);
  }



  /**
   * Tests the {@code get} method with a result that contains a response control
   * that is already of the appropriate type.
   *
   * @throws  Exception  If an unexpected problem occurs.
   */
  @Test()
  public void testGetValidCorrectType()
         throws Exception
  {
    final Control[] controls =
    {
      new UnsolicitedCancelResponseControl()
    };

    final LDAPResult r = new LDAPResult(1, ResultCode.CANCELED, null, null,
         null, controls);

    final UnsolicitedCancelResponseControl c =
         UnsolicitedCancelResponseControl.get(r);
    assertNotNull(c);
  }



  /**
   * Tests the {@code get} method with a result that contains a response control
   * that is a generic control that can be parsed as an unsolicited cancel
   * response control.
   *
   * @throws  Exception  If an unexpected problem occurs.
   */
  @Test()
  public void testGetValidGenericType()
         throws Exception
  {
    final Control tmp = new UnsolicitedCancelResponseControl();

    final Control[] controls =
    {
      new Control(tmp.getOID(), tmp.isCritical(), tmp.getValue())
    };

    final LDAPResult r = new LDAPResult(1, ResultCode.CANCELED, null, null,
         null, controls);

    final UnsolicitedCancelResponseControl c =
         UnsolicitedCancelResponseControl.get(r);
    assertNotNull(c);
  }



  /**
   * Tests the {@code get} method with a result that contains a response control
   * that is a generic control that cannot be parsed as an unsolicited cancel
   * response control.
   *
   * @throws  Exception  If an unexpected problem occurs.
   */
  @Test(expectedExceptions = { LDAPException.class })
  public void testGetInvalidGenericType()
         throws Exception
  {
    final Control[] controls =
    {
      new Control(
           UnsolicitedCancelResponseControl.UNSOLICITED_CANCEL_RESPONSE_OID,
           false, new ASN1OctetString("foo"))
    };

    final LDAPResult r = new LDAPResult(1, ResultCode.CANCELED, null, null,
         null, controls);

    UnsolicitedCancelResponseControl.get(r);
  }
}
