package org.ovirt.engine.ui.uicommonweb.models.storage;

import org.ovirt.engine.core.common.businessentities.StorageDomainType;
import org.ovirt.engine.core.common.businessentities.StorageType;
import org.ovirt.engine.core.common.businessentities.storage_pool;
import org.ovirt.engine.core.compat.Event;
import org.ovirt.engine.core.compat.EventArgs;
import org.ovirt.engine.core.compat.EventDefinition;
import org.ovirt.engine.core.compat.Guid;
import org.ovirt.engine.core.compat.IEventListener;
import org.ovirt.engine.core.compat.Version;
import org.ovirt.engine.ui.uicommonweb.UICommand;
import org.ovirt.engine.ui.uicommonweb.models.EntityModel;
import org.ovirt.engine.ui.uicommonweb.models.ListModel;
import org.ovirt.engine.ui.uicommonweb.models.Model;
import org.ovirt.engine.ui.uicommonweb.validation.IValidation;
import org.ovirt.engine.ui.uicommonweb.validation.IntegerValidation;
import org.ovirt.engine.ui.uicommonweb.validation.LinuxMountPointValidation;
import org.ovirt.engine.ui.uicommonweb.validation.NonUtfValidation;
import org.ovirt.engine.ui.uicommonweb.validation.NotEmptyValidation;
import org.ovirt.engine.ui.uicompat.Constants;
import org.ovirt.engine.ui.uicompat.ConstantsManager;

import java.util.ArrayList;
import java.util.List;

@SuppressWarnings("unused")
public class NfsStorageModel extends Model implements IStorageModel {

    public static EventDefinition PathChangedEventDefinition;
    private Event pathChangedEvent;

    public Event getPathChangedEvent() {
        return pathChangedEvent;
    }

    private void setPathChangedEvent(Event value) {
        pathChangedEvent = value;
    }

    private UICommand updateCommand;

    @Override
    public UICommand getUpdateCommand() {
        return updateCommand;
    }

    private void setUpdateCommand(UICommand value) {
        updateCommand = value;
    }

    private StorageModel container;

    @Override
    public StorageModel getContainer() {
        return container;
    }

    @Override
    public void setContainer(StorageModel value) {
        if (container != value) {
            container = value;
            containerChanged();
        }
    }

    private StorageDomainType role = StorageDomainType.values()[0];

    @Override
    public StorageDomainType getRole() {
        return role;
    }

    @Override
    public void setRole(StorageDomainType value) {
        role = value;
    }

    private EntityModel path;

    public EntityModel getPath() {
        return path;
    }

    private void setPath(EntityModel value) {
        path = value;
    }

    private ListModel version;

    public ListModel getVersion() {
        return version;
    }

    private void setVersion(ListModel value) {
        version = value;
    }

    private EntityModel retransmissions;

    public EntityModel getRetransmissions() {
        return retransmissions;
    }

    private void setRetransmissions(EntityModel value) {
        retransmissions = value;
    }

    private EntityModel timeout;

    public EntityModel getTimeout() {
        return timeout;
    }

    private void setTimeout(EntityModel value) {
        timeout = value;
    }


    static {

        PathChangedEventDefinition = new EventDefinition("PathChanged", NfsStorageModel.class); //$NON-NLS-1$
    }

    public NfsStorageModel() {

        setPathChangedEvent(new Event(PathChangedEventDefinition));

        setUpdateCommand(new UICommand("Update", this)); //$NON-NLS-1$

        setPath(new EntityModel());
        getPath().getEntityChangedEvent().addListener(this);

        Constants constants = ConstantsManager.getInstance().getConstants();

        // Initialize version list.
        setVersion(new ListModel());

        List<EntityModel> versionItems = new ArrayList<EntityModel>();
        versionItems.add(new EntityModel(constants.nfsVersionAutoNegotiate(), null));
        versionItems.add(new EntityModel(constants.nfsVersion3(), (short) 3));
        versionItems.add(new EntityModel(constants.nfsVersion4(), (short) 4));
        getVersion().setItems(versionItems);

        setRetransmissions(new EntityModel());
        setTimeout(new EntityModel());
    }

    @Override
    public void eventRaised(Event ev, Object sender, EventArgs args) {
        super.eventRaised(ev, sender, args);
        if (ev.equals(EntityModel.EntityChangedEventDefinition) && sender == getPath()) {
            // Notify about path change.
            getPathChangedEvent().raise(this, EventArgs.Empty);
        }
    }

    @Override
    public boolean Validate() {

        getPath().ValidateEntity(new IValidation[] {
            new NotEmptyValidation(),
            new LinuxMountPointValidation(),
            new NonUtfValidation()
        });

        getRetransmissions().ValidateEntity(new IValidation[] {
            new IntegerValidation(0, Integer.MAX_VALUE)
        });

        getTimeout().ValidateEntity(new IValidation[] {
            new IntegerValidation(1, Integer.MAX_VALUE)
        });

        return getPath().getIsValid()
            && getRetransmissions().getIsValid()
            && getTimeout().getIsValid();
    }

    @Override
    public StorageType getType() {
        return StorageType.NFS;
    }

    private void containerChanged() {
        // Subscribe to the data center change.
        if (getContainer() == null) {
            return;
        }

        ListModel dataCenter = getContainer().getDataCenter();
        dataCenter.getSelectedItemChangedEvent().addListener(new IEventListener() {
            @Override
            public void eventRaised(Event ev, Object sender, EventArgs args) {

                containerDataCenterChanged();
            }
        });

        // Call handler if there some data center is already selected.
        if (dataCenter.getSelectedItem() != null) {
            containerDataCenterChanged();
        }
    }

    private void containerDataCenterChanged() {

        // Show advanced NFS options for <=3.1
        storage_pool dataCenter = (storage_pool) getContainer().getDataCenter().getSelectedItem();
        Version ver31 = new Version(3, 1);

        boolean available = dataCenter != null && (dataCenter.getcompatibility_version().compareTo(ver31) >= 0 || dataCenter.getId().equals(Guid.Empty));

        getVersion().setIsAvailable(available);
        getRetransmissions().setIsAvailable(available);
        getTimeout().setIsAvailable(available);
    }
}
