# Copyright 2013 Red Hat, Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
#
# Refer to the README and COPYING files for full details of the license
#

import os
import sys
import grp
import argparse
import pwd

from .. import utils
from . import service, expose
from ..constants import P_VDSM_EXEC, QEMU_PROCESS_GROUP, VDSM_GROUP

# Declare state of configuration
#
# CONFIGURED     = Module is set properly without any required changes on
#                  force.
# NOT_CONFIGURED = Module is not set properly for VDSM and need to be
#                  configured.
# NOT_SURE       = VDSM configured module already but on force configure vdsm
#                  will set configurations to defaults parameters.
#
CONFIGURED, NOT_CONFIGURED, NOT_SURE = range(3)


class _ModuleConfigure(object):

    def __init__(self):
        pass

    def getName(self):
        return None

    def getServices(self):
        return []

    def validate(self):
        return True

    def configure(self):
        pass

    def isconfigured(self):
        return NOT_CONFIGURED


class LibvirtModuleConfigure(_ModuleConfigure):
    def __init__(self):
        super(LibvirtModuleConfigure, self).__init__()

    def getName(self):
        return 'libvirt'

    def getServices(self):
        return ["supervdsmd", "vdsmd", "libvirtd"]

    def _exec_libvirt_configure(self, action):
        """
        Invoke libvirt_configure.sh script
        """
        if os.getuid() != 0:
            raise UserWarning("Must run as root")

        rc, out, err = utils.execCmd(
            (
                os.path.join(
                    P_VDSM_EXEC,
                    'libvirt_configure.sh'
                ),
                action,
            ),
            raw=True,
        )
        sys.stdout.write(out)
        sys.stderr.write(err)
        if rc != 0:
            raise RuntimeError("Failed to perform libvirt action.")

    def configure(self):
        self._exec_libvirt_configure("reconfigure")

    def validate(self):
        """
        Validate conflict in configured files
        """
        try:
            self._exec_libvirt_configure("test_conflict_configurations")
            return True
        except RuntimeError:
            return False

    def isconfigured(self):
        """
        Check if libvirt is already configured for vdsm
        """
        try:
            self._exec_libvirt_configure("check_if_configured")
            return NOT_SURE
        except RuntimeError:
            return NOT_CONFIGURED


class SanlockModuleConfigure(_ModuleConfigure):

    SANLOCK_GROUPS = (QEMU_PROCESS_GROUP, VDSM_GROUP)

    def __init__(self):
        super(SanlockModuleConfigure, self).__init__()

    def getName(self):
        return 'sanlock'

    def getServices(self):
        return ['sanlock']

    def configure(self):
        """
        Configure sanlock process groups
        """
        if os.getuid() != 0:
            raise UserWarning("Must run as root")

        rc, out, err = utils.execCmd(
            (
                '/usr/sbin/usermod',
                '-a',
                '-G',
                ','.join(self.SANLOCK_GROUPS),
                'sanlock'
            ),
            raw=True,
        )
        sys.stdout.write(out)
        sys.stderr.write(err)
        if rc != 0:
            raise RuntimeError("Failed to perform sanlock config.")

    def isconfigured(self):
        """
        True if sanlock service is configured, False if sanlock service
        requires a restart to reload the relevant supplementary groups.
        """
        configured = NOT_CONFIGURED
        groups = [g.gr_name for g in grp.getgrall()
                  if 'sanlock' in g.gr_mem]
        gid = pwd.getpwnam('sanlock').pw_gid
        groups.append(grp.getgrgid(gid).gr_name)
        if all(group in groups for group in self.SANLOCK_GROUPS):
            configured = NOT_SURE

        if configured == NOT_SURE:
            try:
                with open("/var/run/sanlock/sanlock.pid", "r") as f:
                    sanlock_pid = f.readline().strip()
                with open(os.path.join('/proc', sanlock_pid, 'status'),
                          "r") as sanlock_status:
                    proc_status_group_prefix = "Groups:\t"
                    for status_line in sanlock_status:
                        if status_line.startswith(proc_status_group_prefix):
                            groups = [int(x) for x in status_line[
                                len(proc_status_group_prefix):]
                                .strip().split(" ")]
                            break
                    else:
                        raise RuntimeError(
                            "Unable to find sanlock service groups"
                        )

                is_sanlock_groups_set = True
                for g in self.SANLOCK_GROUPS:
                    if grp.getgrnam(g)[2] not in groups:
                        is_sanlock_groups_set = False
                if is_sanlock_groups_set:
                    configured = CONFIGURED

            except IOError as e:
                if e.errno == os.errno.ENOENT:
                    configured = CONFIGURED
                else:
                    raise

        return configured


__configurers = (
    LibvirtModuleConfigure(),
    SanlockModuleConfigure(),
)


@expose("configure")
def configure(*args):
    """
    Configure external services for vdsm
    """
    args = _parse_args("configure")
    configurer_to_trigger = []

    sys.stdout.write("\nChecking configuration status...\n\n")
    for c in __configurers:
        if c.getName() in args.modules:
            override = args.force and (c.isconfigured != CONFIGURED)
            if not override and not c.validate():
                raise RuntimeError(
                    "Configuration of %s is invalid" % c.getName()
                )
            if override:
                configurer_to_trigger.append(c)

    services = []
    for c in configurer_to_trigger:
        for s in c.getServices():
            if service.service_status(s, False) == 0:
                if not args.force:
                    raise RuntimeError(
                        "\n\nCannot configure while service '%s' is "
                        "running.\n Stop the service manually or use the "
                        "--force flag.\n" % s
                    )
                services.append(s)

    for s in services:
        service.service_stop(s)

    sys.stdout.write("\nRunning configure...\n")
    for c in configurer_to_trigger:
        c.configure()

    for s in reversed(services):
        service.service_start(s)
    sys.stdout.write("\nDone configuring modules to VDSM.\n")


@expose("is-configured")
def isconfigured(*args):
    """
    Determine if module is configured
    """
    ret = True
    args = _parse_args('is-configured')

    m = [
        c.getName() for c in __configurers
        if c.getName() in args.modules and c.isconfigured() == NOT_CONFIGURED
    ]

    if m:
        sys.stdout.write(
            "Modules %s are not configured\n " % ','.join(m),
        )
        ret = False

    if not ret:
        msg = \
            """

One of the modules is not configured to work with VDSM.
To configure the module use the following:
'vdsm-tool configure [module_name]'.

If all modules are not configured try to use:
'vdsm-tool configure --force'
(The force flag will stop the module's service and start it
afterwards automatically to load the new configuration.)
"""
        raise RuntimeError(msg)


@expose("validate-config")
def validate_config(*args):
    """
    Determine if configuration is valid
    """
    ret = True
    args = _parse_args('validate-config')

    m = [
        c.getName() for c in __configurers
        if c.getName() in args.modules and not c.validate()
    ]

    if m:
        sys.stdout.write(
            "Modules %s contains invalid configuration\n " % ','.join(m),
        )
        ret = False

    if not ret:
        raise RuntimeError("Config is not valid. Check conf files")


def _parse_args(action):
    parser = argparse.ArgumentParser('vdsm-tool %s' % (action))
    allModules = [n.getName() for n in __configurers]
    parser.add_argument(
        '--module',
        dest='modules',
        choices=allModules,
        default=[],
        metavar='STRING',
        action='append',
        help=(
            'Specify the module to run the action on '
            '(e.g %(choices)s).\n'
            'If non is specified, operation will run for '
            'all related modules.'
        ),
    )
    if action == "configure":
        parser.add_argument(
            '--force',
            dest='force',
            default=False,
            action='store_true',
            help='Force configuration, trigger services restart',
        )
    args = parser.parse_args(sys.argv[2:])
    if not args.modules:
        args.modules = allModules
    return args
