/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.services;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URISyntaxException;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.util.EntityUtils;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.api.metamodel.runtime.xml.XmlWriter;
import org.ovirt.engine.sdk4.Error;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.DiskBuilder;
import org.ovirt.engine.sdk4.builders.StorageDomainBuilder;
import org.ovirt.engine.sdk4.internal.HttpConnection;
import org.ovirt.engine.sdk4.internal.xml.XmlDiskReader;
import org.ovirt.engine.sdk4.internal.xml.XmlDiskWriter;
import org.ovirt.engine.sdk4.internal.xml.XmlStorageDomainWriter;
import org.ovirt.engine.sdk4.services.AssignedPermissionsService;
import org.ovirt.engine.sdk4.services.DiskService;
import org.ovirt.engine.sdk4.services.StatisticsService;
import org.ovirt.engine.sdk4.types.Action;
import org.ovirt.engine.sdk4.types.Disk;
import org.ovirt.engine.sdk4.types.StorageDomain;

public class DiskServiceImpl extends ServiceImpl implements DiskService {
    
    public DiskServiceImpl(HttpConnection connection, String path) {
        super(connection, path);
    }
    
    private class CopyRequestImpl implements CopyRequest {
        private Boolean async;
        public CopyRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        private Disk disk;
        public CopyRequest disk(Disk disk) {
            this.disk = disk;
            return this;
        }
        
        public CopyRequest disk(DiskBuilder disk) {
            this.disk = disk.build();
            return this;
        }
        
        private Boolean filter;
        public CopyRequest filter(Boolean filter) {
            this.filter = filter;
            return this;
        }
        
        private StorageDomain storageDomain;
        public CopyRequest storageDomain(StorageDomain storageDomain) {
            this.storageDomain = storageDomain;
            return this;
        }
        
        public CopyRequest storageDomain(StorageDomainBuilder storageDomain) {
            this.storageDomain = storageDomain.build();
            return this;
        }
        
        public CopyResponse send() {
            HttpPost request = new HttpPost(getConnection().getUrl() + getPath() + "/copy");
            try (
              ByteArrayOutputStream output = new ByteArrayOutputStream();
              XmlWriter xmlWriter = new XmlWriter(output, true)
            ) {
                xmlWriter.writeStartElement("action");
                if (disk != null) {
                    XmlDiskWriter.writeOne(disk, xmlWriter);
                }
                if (storageDomain != null) {
                    XmlStorageDomainWriter.writeOne(storageDomain, xmlWriter);
                }
                if (async != null) {
                    xmlWriter.writeBoolean("async", async);
                }
                if (filter != null) {
                    xmlWriter.writeBoolean("filter", filter);
                }
                xmlWriter.writeEndElement();
                xmlWriter.flush();
                request.setEntity(new ByteArrayEntity(output.toByteArray()));
            }
            catch (IOException ex) {
                throw new Error("Failed to write request", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (response.getStatusLine().getStatusCode() == 200) {
                Action action = checkAction(response);
                EntityUtils.consumeQuietly(response.getEntity());
                return new CopyResponseImpl();
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class CopyResponseImpl implements CopyResponse {
    }
    
    public CopyRequest copy() {
        return new CopyRequestImpl();
    }
    
    private class ExportRequestImpl implements ExportRequest {
        private Boolean async;
        public ExportRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        private Boolean filter;
        public ExportRequest filter(Boolean filter) {
            this.filter = filter;
            return this;
        }
        
        private StorageDomain storageDomain;
        public ExportRequest storageDomain(StorageDomain storageDomain) {
            this.storageDomain = storageDomain;
            return this;
        }
        
        public ExportRequest storageDomain(StorageDomainBuilder storageDomain) {
            this.storageDomain = storageDomain.build();
            return this;
        }
        
        public ExportResponse send() {
            HttpPost request = new HttpPost(getConnection().getUrl() + getPath() + "/export");
            try (
              ByteArrayOutputStream output = new ByteArrayOutputStream();
              XmlWriter xmlWriter = new XmlWriter(output, true)
            ) {
                xmlWriter.writeStartElement("action");
                if (storageDomain != null) {
                    XmlStorageDomainWriter.writeOne(storageDomain, xmlWriter);
                }
                if (async != null) {
                    xmlWriter.writeBoolean("async", async);
                }
                if (filter != null) {
                    xmlWriter.writeBoolean("filter", filter);
                }
                xmlWriter.writeEndElement();
                xmlWriter.flush();
                request.setEntity(new ByteArrayEntity(output.toByteArray()));
            }
            catch (IOException ex) {
                throw new Error("Failed to write request", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (response.getStatusLine().getStatusCode() == 200) {
                Action action = checkAction(response);
                EntityUtils.consumeQuietly(response.getEntity());
                return new ExportResponseImpl();
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class ExportResponseImpl implements ExportResponse {
    }
    
    public ExportRequest export() {
        return new ExportRequestImpl();
    }
    
    private class GetRequestImpl implements GetRequest {
        public GetResponse send() {
            HttpGet request = null;
            try {
                URIBuilder uriBuilder = new URIBuilder(getConnection().getUrl() + getPath());
                request = new HttpGet(uriBuilder.build());
            }
            catch (URISyntaxException ex) {
                throw new Error("Failed to build URL", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (
              response.getStatusLine().getStatusCode() == 200
            ) {
                try (
                  XmlReader reader = new XmlReader(response.getEntity().getContent())
                ) {
                    return new GetResponseImpl(XmlDiskReader.readOne(reader));
                }
                catch (IOException ex) {
                    throw new Error("Failed to read response", ex);
                }
                finally {
                    EntityUtils.consumeQuietly(response.getEntity());
                }
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class GetResponseImpl implements GetResponse {
        private Disk disk;
        
        public GetResponseImpl(Disk disk) {
            this.disk = disk;
        }
        
        public Disk disk() {
            return disk;
        }
        
    }
    
    public GetRequest get() {
        return new GetRequestImpl();
    }
    
    private class MoveRequestImpl implements MoveRequest {
        private Boolean async;
        public MoveRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        private Boolean filter;
        public MoveRequest filter(Boolean filter) {
            this.filter = filter;
            return this;
        }
        
        private StorageDomain storageDomain;
        public MoveRequest storageDomain(StorageDomain storageDomain) {
            this.storageDomain = storageDomain;
            return this;
        }
        
        public MoveRequest storageDomain(StorageDomainBuilder storageDomain) {
            this.storageDomain = storageDomain.build();
            return this;
        }
        
        public MoveResponse send() {
            HttpPost request = new HttpPost(getConnection().getUrl() + getPath() + "/move");
            try (
              ByteArrayOutputStream output = new ByteArrayOutputStream();
              XmlWriter xmlWriter = new XmlWriter(output, true)
            ) {
                xmlWriter.writeStartElement("action");
                if (storageDomain != null) {
                    XmlStorageDomainWriter.writeOne(storageDomain, xmlWriter);
                }
                if (async != null) {
                    xmlWriter.writeBoolean("async", async);
                }
                if (filter != null) {
                    xmlWriter.writeBoolean("filter", filter);
                }
                xmlWriter.writeEndElement();
                xmlWriter.flush();
                request.setEntity(new ByteArrayEntity(output.toByteArray()));
            }
            catch (IOException ex) {
                throw new Error("Failed to write request", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (response.getStatusLine().getStatusCode() == 200) {
                Action action = checkAction(response);
                EntityUtils.consumeQuietly(response.getEntity());
                return new MoveResponseImpl();
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class MoveResponseImpl implements MoveResponse {
    }
    
    public MoveRequest move() {
        return new MoveRequestImpl();
    }
    
    private class RemoveRequestImpl implements RemoveRequest {
        private Boolean async;
        public RemoveRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        public RemoveResponse send() {
            HttpDelete request = null;
            try {
                URIBuilder uriBuilder = new URIBuilder(getConnection().getUrl() + getPath());
                if (async != null) {
                    uriBuilder.addParameter("async", XmlWriter.renderBoolean(async));
                }
                request = new HttpDelete(uriBuilder.build());
            }
            catch (URISyntaxException ex) {
                throw new Error("Failed to build URL", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (
              response.getStatusLine().getStatusCode() == 200
            ) {
                EntityUtils.consumeQuietly(response.getEntity());
                return new RemoveResponseImpl();
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class RemoveResponseImpl implements RemoveResponse {
    }
    
    public RemoveRequest remove() {
        return new RemoveRequestImpl();
    }
    
    public AssignedPermissionsService permissionsService() {
        return new AssignedPermissionsServiceImpl(getConnection(), getPath() + "/permissions");
    }
    
    public StatisticsService statisticsService() {
        return new StatisticsServiceImpl(getConnection(), getPath() + "/statistics");
    }
    
    public Service service(String path) {
        if (path == null) {
            return this;
        }
        
        if (path.equals("permissions")) {
            return permissionsService();
        }
        if (path.startsWith("permissions/")) {
            return permissionsService().service(path.substring(12));
        }
        if (path.equals("statistics")) {
            return statisticsService();
        }
        if (path.startsWith("statistics/")) {
            return statisticsService().service(path.substring(11));
        }
        throw new Error("The path " + path + " doesn't correspond to any service");
    }
    
    public String toString() {
        return "DiskService:" + getPath();
    }
    
}

