/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.DiskAttachmentBuilder;
import org.ovirt.engine.sdk4.types.DiskAttachment;

/**
 * This service manages the attachment of a disk to a virtual machine.
 */
public interface DiskAttachmentService extends Service {
    /**
     * Returns the details of the attachment, including the bootable flag and link to the disk.
     */
    public interface GetRequest extends Request<GetRequest, GetResponse> {
    }
    
    /**
     * Returns the details of the attachment, including the bootable flag and link to the disk.
     */
    public interface GetResponse extends Response {
        DiskAttachment attachment();
    }
    
    /**
     * Returns the details of the attachment, including the bootable flag and link to the disk.
     */
    GetRequest get();
    
    /**
     * Removes the disk attachment. This will only detach the disk from the virtual machine, but won't remove it from
     * the system, unless the `detach_only` parameter is `false`.
     */
    public interface RemoveRequest extends Request<RemoveRequest, RemoveResponse> {
        /**
         * Indicates if the disk should only be detached from the virtual machine, but not removed from the system.
         * The default value is `true`, which won't remove the disk from the system.
         */
        RemoveRequest detachOnly(Boolean detachOnly);
    }
    
    /**
     * Removes the disk attachment. This will only detach the disk from the virtual machine, but won't remove it from
     * the system, unless the `detach_only` parameter is `false`.
     */
    public interface RemoveResponse extends Response {
    }
    
    /**
     * Removes the disk attachment. This will only detach the disk from the virtual machine, but won't remove it from
     * the system, unless the `detach_only` parameter is `false`.
     */
    RemoveRequest remove();
    
    /**
     * Update the disk attachment and the disk properties within it.
     * 
     * [source]
     * ----
 * PUT /vms/{vm:id}/disksattachments/{attachment:id}
 * <disk_attachment>
 *   <bootable>true</bootable>
 *   <interface>ide</interface>
 *   <disk>
 *     <name>mydisk</name>
 *     <provisioned_size>1024</provisioned_size>
 *     ...
 *   </disk>
 * </disk_attachment>
 * ----
 */
public interface UpdateRequest extends Request<UpdateRequest, UpdateResponse> {
    UpdateRequest diskAttachment(DiskAttachment diskAttachment);
    UpdateRequest diskAttachment(DiskAttachmentBuilder diskAttachment);
}

/**
 * Update the disk attachment and the disk properties within it.
 * 
 * [source]
 * ----
 * PUT /vms/{vm:id}/disksattachments/{attachment:id}
 * <disk_attachment>
 *   <bootable>true</bootable>
 *   <interface>ide</interface>
 *   <disk>
 *     <name>mydisk</name>
 *     <provisioned_size>1024</provisioned_size>
 *     ...
 *   </disk>
 * </disk_attachment>
 * ----
 */
public interface UpdateResponse extends Response {
    DiskAttachment diskAttachment();
}

/**
 * Update the disk attachment and the disk properties within it.
 * 
 * [source]
 * ----
 * PUT /vms/{vm:id}/disksattachments/{attachment:id}
 * <disk_attachment>
 *   <bootable>true</bootable>
 *   <interface>ide</interface>
 *   <disk>
 *     <name>mydisk</name>
 *     <provisioned_size>1024</provisioned_size>
 *     ...
 *   </disk>
 * </disk_attachment>
 * ----
 */
UpdateRequest update();

/**
 * Service locator method, returns individual service on which the URI is dispatched.
 */
Service service(String path);
}

