/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.lang.String;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.ClusterBuilder;
import org.ovirt.engine.sdk4.builders.HostBuilder;
import org.ovirt.engine.sdk4.builders.HostNicBuilder;
import org.ovirt.engine.sdk4.builders.IscsiDetailsBuilder;
import org.ovirt.engine.sdk4.builders.NetworkAttachmentBuilder;
import org.ovirt.engine.sdk4.builders.NetworkLabelBuilder;
import org.ovirt.engine.sdk4.builders.SshBuilder;
import org.ovirt.engine.sdk4.types.Cluster;
import org.ovirt.engine.sdk4.types.Host;
import org.ovirt.engine.sdk4.types.HostNic;
import org.ovirt.engine.sdk4.types.IscsiDetails;
import org.ovirt.engine.sdk4.types.NetworkAttachment;
import org.ovirt.engine.sdk4.types.NetworkLabel;
import org.ovirt.engine.sdk4.types.PowerManagement;
import org.ovirt.engine.sdk4.types.Ssh;
import org.ovirt.engine.sdk4.types.StorageDomain;

public interface HostService extends Service {
    public interface ActivateRequest extends Request<ActivateRequest, ActivateResponse> {
        /**
         * Indicates if the activation should be performed asynchronously.
         */
        ActivateRequest async(Boolean async);
    }
    
    public interface ActivateResponse extends Response {
    }
    
    ActivateRequest activate();
    
    public interface ApproveRequest extends Request<ApproveRequest, ApproveResponse> {
        /**
         * Indicates if the approval should be performed asynchronously.
         */
        ApproveRequest async(Boolean async);
        ApproveRequest cluster(Cluster cluster);
        ApproveRequest cluster(ClusterBuilder cluster);
    }
    
    public interface ApproveResponse extends Response {
    }
    
    ApproveRequest approve();
    
    /**
     * Marks the network configuration as good and persists it inside the host.
     */
    public interface CommitNetConfigRequest extends Request<CommitNetConfigRequest, CommitNetConfigResponse> {
        /**
         * Indicates if the action should be performed asynchronously.
         */
        CommitNetConfigRequest async(Boolean async);
    }
    
    /**
     * Marks the network configuration as good and persists it inside the host.
     */
    public interface CommitNetConfigResponse extends Response {
    }
    
    /**
     * Marks the network configuration as good and persists it inside the host.
     */
    CommitNetConfigRequest commitNetConfig();
    
    public interface DeactivateRequest extends Request<DeactivateRequest, DeactivateResponse> {
        /**
         * Indicates if the deactivation should be performed asynchronously.
         */
        DeactivateRequest async(Boolean async);
        DeactivateRequest reason(String reason);
        /**
         * Indicates if the gluster service should be stopped as part of deactivating the host. It can be used while
         * performing maintenance operations on the gluster host. Default value for this variable is `false`.
         */
        DeactivateRequest stopGlusterService(Boolean stopGlusterService);
    }
    
    public interface DeactivateResponse extends Response {
    }
    
    DeactivateRequest deactivate();
    
    public interface EnrollCertificateRequest extends Request<EnrollCertificateRequest, EnrollCertificateResponse> {
        /**
         * Indicates if the enrollment should be performed asynchronously.
         */
        EnrollCertificateRequest async(Boolean async);
    }
    
    public interface EnrollCertificateResponse extends Response {
    }
    
    EnrollCertificateRequest enrollCertificate();
    
    public interface FenceRequest extends Request<FenceRequest, FenceResponse> {
        /**
         * Indicates if the fencing should be performed asynchronously.
         */
        FenceRequest async(Boolean async);
        FenceRequest fenceType(String fenceType);
    }
    
    public interface FenceResponse extends Response {
        PowerManagement powerManagement();
    }
    
    FenceRequest fence();
    
    public interface ForceSelectSpmRequest extends Request<ForceSelectSpmRequest, ForceSelectSpmResponse> {
        /**
         * Indicates if the action should be performed asynchronously.
         */
        ForceSelectSpmRequest async(Boolean async);
    }
    
    public interface ForceSelectSpmResponse extends Response {
    }
    
    ForceSelectSpmRequest forceSelectSpm();
    
    public interface GetRequest extends Request<GetRequest, GetResponse> {
        /**
         * Indicates if the results should be filtered according to the permissions of the user.
         */
        GetRequest filter(Boolean filter);
    }
    
    public interface GetResponse extends Response {
        Host host();
    }
    
    GetRequest get();
    
    public interface InstallRequest extends Request<InstallRequest, InstallResponse> {
        /**
         * Indicates if the installation should be performed asynchronously.
         */
        InstallRequest async(Boolean async);
        /**
         * When set to `true` it means this host should deploy also hosted
         * engine components. Missing value is treated as `true` i.e deploy.
         * Omitting this parameter means `false` and will perform no operation
         * in hosted engine area.
         */
        InstallRequest deployHostedEngine(Boolean deployHostedEngine);
        InstallRequest host(Host host);
        /**
         * This `override_iptables` property is used to indicate if the firewall configuration should be
         * replaced by the default one.
         */
        InstallRequest host(HostBuilder host);
        /**
         * When installing an oVirt node a image ISO file is needed.
         */
        InstallRequest image(String image);
        /**
         * The password of of the `root` user, used to connect to the host via SSH.
         */
        InstallRequest rootPassword(String rootPassword);
        InstallRequest ssh(Ssh ssh);
        /**
         * The SSH details used to connect to the host.
         */
        InstallRequest ssh(SshBuilder ssh);
        /**
         * When set to `true` it means this host should un-deploy hosted engine
         * components and this host will not function as part of the High
         * Availability cluster. Missing value is treated as `true` i.e un-deploy
         * Omitting this parameter means `false` and will perform no operation
         * in hosted engine area.
         */
        InstallRequest undeployHostedEngine(Boolean undeployHostedEngine);
    }
    
    public interface InstallResponse extends Response {
    }
    
    InstallRequest install();
    
    public interface IscsiDiscoverRequest extends Request<IscsiDiscoverRequest, IscsiDiscoverResponse> {
        /**
         * Indicates if the discovery should be performed asynchronously.
         */
        IscsiDiscoverRequest async(Boolean async);
        IscsiDiscoverRequest iscsi(IscsiDetails iscsi);
        IscsiDiscoverRequest iscsi(IscsiDetailsBuilder iscsi);
    }
    
    public interface IscsiDiscoverResponse extends Response {
        List<String> iscsiTargets();
    }
    
    IscsiDiscoverRequest iscsiDiscover();
    
    public interface IscsiLoginRequest extends Request<IscsiLoginRequest, IscsiLoginResponse> {
        /**
         * Indicates if the login should be performed asynchronously.
         */
        IscsiLoginRequest async(Boolean async);
        IscsiLoginRequest iscsi(IscsiDetails iscsi);
        IscsiLoginRequest iscsi(IscsiDetailsBuilder iscsi);
    }
    
    public interface IscsiLoginResponse extends Response {
    }
    
    IscsiLoginRequest iscsiLogin();
    
    public interface RefreshRequest extends Request<RefreshRequest, RefreshResponse> {
        /**
         * Indicates if the refresh should be performed asynchronously.
         */
        RefreshRequest async(Boolean async);
    }
    
    public interface RefreshResponse extends Response {
    }
    
    RefreshRequest refresh();
    
    public interface RemoveRequest extends Request<RemoveRequest, RemoveResponse> {
        /**
         * Indicates if the remove should be performed asynchronously.
         */
        RemoveRequest async(Boolean async);
    }
    
    public interface RemoveResponse extends Response {
    }
    
    RemoveRequest remove();
    
    /**
     * This method is used to change the configuration of the network interfaces of a host.
     * 
     * For example, lets assume that you have a host with three network interfaces `eth0`, `eth1` and `eth2` and that
     * you want to configure a new bond using `eth0` and `eth1`, and put a VLAN on top of it. Using a simple shell
     * script and the `curl` command line HTTP client that can be done as follows:
     * 
     * [source]
     * ----
     * #!/bin/sh -ex
     * 
     * url="https://engine.example.com/ovirt-engine/api"
     * user="admin@internal"
     * password="..."
     * 
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --user "${user}:${password}" \
     * --request POST \
     * --header "Version: 4" \
     * --header "Content-Type: application/xml" \
     * --header "Accept: application/xml" \
     * --data '
     * <action>
     *   <modified_bonds>
     *     <host_nic>
     *       <name>bond0</name>
     *       <bonding>
     *         <options>
     *           <option>
     *             <name>mode</name>
     *             <value>4</value>
     *           </option>
     *           <option>
     *             <name>miimon</name>
     *             <value>100</value>
     *           </option>
     *         </options>
     *         <slaves>
     *           <host_nic>
     *             <name>eth1</name>
     *           </host_nic>
     *           <host_nic>
     *             <name>eth2</name>
     *           </host_nic>
     *         </slaves>
     *       </bonding>
     *     </host_nic>
     *   </modified_bonds>
     *   <modified_network_attachments>
     *     <network_attachment>
     *       <network>
     *         <name>myvlan</name>
     *       </network>
     *       <host_nic>
     *         <name>bond0</name>
     *       </host_nic>
     *       <ip_address_assignments>
     *         <assignment_method>static</assignment_method>
     *         <ip_address_assignment>
     *           <ip>
     *             <address>192.168.122.10</address>
     *             <netmask>255.255.255.0</netmask>
     *           </ip>
     *         </ip_address_assignment>
     *       </ip_address_assignments>
     *     </network_attachment>
     *   </modified_network_attachments>
     *  </action>
     * ' \
     * "${url}/hosts/1ff7a191-2f3b-4eff-812b-9f91a30c3acc/setupnetworks"
     * ----
     * 
     * Note that this is valid for version 4 of the API. In previous versions some elements were represented as XML
     * attributes instead of XML elements. In particular the `options` and `ip` elements were represented as follows:
     * 
     * [source,xml]
     * ----
     * <options name="mode" value="4"/>
     * <options name="miimon" value="100"/>
     * <ip address="192.168.122.10" netmask="255.255.255.0"/>
     * ----
     * 
     * Using the Python SDK the same can be done with the following code:
     * 
     * [source,python]
     * ----
     * host.setupnetworks(
     *   params.Action(
     *     modified_bonds=params.HostNics(
     *       host_nic=[
     *         params.HostNIC(
     *           name="bond0",
     *           bonding=params.Bonding(
     *             options=params.Options(
     *               option=[
     *                 params.Option(name="mode", value="4"),
     *                 params.Option(name="miimon", value="100"),
     *               ],
     *             ),
     *             slaves=params.Slaves(
     *               host_nic=[
     *                 params.HostNIC(name="eth1"),
     *                 params.HostNIC(name="eth2"),
     *               ],
     *             ),
     *           ),
     *         ),
     *       ],
     *     ),
     *     modified_network_attachments=params.NetworkAttachments(
     *       network_attachment=[
     *         params.NetworkAttachment(
     *           network=params.Network(name="myvlan"),
     *           host_nic=params.HostNIC(name="bond0"),
     *           ip_address_assignments=params.IpAddressAssignments(
     *             ip_address_assignment=[
     *               params.IpAddressAssignment(
     *                 assignment_method="static",
     *                 ip=params.IP(
     *                   address="192.168.122.10",
     *                   netmask="255.255.255.0",
     *                 ),
     *               ),
     *             ],
     *           ),
     *         ),
     *       ],
     *     ),
     *   ),
     * )
     * ----
     * 
     * IMPORTANT: To make sure that the network configuration has been saved in the host, and that it will be applied
     * when the host is rebooted, remember to call <<services/host/methods/commitnetconfig, commitnetconfig>>.
     */
    public interface SetupNetworksRequest extends Request<SetupNetworksRequest, SetupNetworksResponse> {
        /**
         * Indicates if the action should be performed asynchronously.
         */
        SetupNetworksRequest async(Boolean async);
        SetupNetworksRequest checkConnectivity(Boolean checkConnectivity);
        SetupNetworksRequest connectivityTimeout(Integer connectivityTimeout);
        SetupNetworksRequest connectivityTimeout(Long connectivityTimeout);
        SetupNetworksRequest connectivityTimeout(BigInteger connectivityTimeout);
        SetupNetworksRequest modifiedBonds(List<HostNic> modifiedBonds);
        SetupNetworksRequest modifiedBonds(HostNic... modifiedBonds);
        SetupNetworksRequest modifiedBonds(HostNicBuilder... modifiedBonds);
        SetupNetworksRequest modifiedLabels(List<NetworkLabel> modifiedLabels);
        SetupNetworksRequest modifiedLabels(NetworkLabel... modifiedLabels);
        SetupNetworksRequest modifiedLabels(NetworkLabelBuilder... modifiedLabels);
        SetupNetworksRequest modifiedNetworkAttachments(List<NetworkAttachment> modifiedNetworkAttachments);
        SetupNetworksRequest modifiedNetworkAttachments(NetworkAttachment... modifiedNetworkAttachments);
        SetupNetworksRequest modifiedNetworkAttachments(NetworkAttachmentBuilder... modifiedNetworkAttachments);
        SetupNetworksRequest removedBonds(List<HostNic> removedBonds);
        SetupNetworksRequest removedBonds(HostNic... removedBonds);
        SetupNetworksRequest removedBonds(HostNicBuilder... removedBonds);
        SetupNetworksRequest removedLabels(List<NetworkLabel> removedLabels);
        SetupNetworksRequest removedLabels(NetworkLabel... removedLabels);
        SetupNetworksRequest removedLabels(NetworkLabelBuilder... removedLabels);
        SetupNetworksRequest removedNetworkAttachments(List<NetworkAttachment> removedNetworkAttachments);
        SetupNetworksRequest removedNetworkAttachments(NetworkAttachment... removedNetworkAttachments);
        SetupNetworksRequest removedNetworkAttachments(NetworkAttachmentBuilder... removedNetworkAttachments);
        SetupNetworksRequest synchronizedNetworkAttachments(List<NetworkAttachment> synchronizedNetworkAttachments);
        SetupNetworksRequest synchronizedNetworkAttachments(NetworkAttachment... synchronizedNetworkAttachments);
        SetupNetworksRequest synchronizedNetworkAttachments(NetworkAttachmentBuilder... synchronizedNetworkAttachments);
    }
    
    /**
     * This method is used to change the configuration of the network interfaces of a host.
     * 
     * For example, lets assume that you have a host with three network interfaces `eth0`, `eth1` and `eth2` and that
     * you want to configure a new bond using `eth0` and `eth1`, and put a VLAN on top of it. Using a simple shell
     * script and the `curl` command line HTTP client that can be done as follows:
     * 
     * [source]
     * ----
     * #!/bin/sh -ex
     * 
     * url="https://engine.example.com/ovirt-engine/api"
     * user="admin@internal"
     * password="..."
     * 
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --user "${user}:${password}" \
     * --request POST \
     * --header "Version: 4" \
     * --header "Content-Type: application/xml" \
     * --header "Accept: application/xml" \
     * --data '
     * <action>
     *   <modified_bonds>
     *     <host_nic>
     *       <name>bond0</name>
     *       <bonding>
     *         <options>
     *           <option>
     *             <name>mode</name>
     *             <value>4</value>
     *           </option>
     *           <option>
     *             <name>miimon</name>
     *             <value>100</value>
     *           </option>
     *         </options>
     *         <slaves>
     *           <host_nic>
     *             <name>eth1</name>
     *           </host_nic>
     *           <host_nic>
     *             <name>eth2</name>
     *           </host_nic>
     *         </slaves>
     *       </bonding>
     *     </host_nic>
     *   </modified_bonds>
     *   <modified_network_attachments>
     *     <network_attachment>
     *       <network>
     *         <name>myvlan</name>
     *       </network>
     *       <host_nic>
     *         <name>bond0</name>
     *       </host_nic>
     *       <ip_address_assignments>
     *         <assignment_method>static</assignment_method>
     *         <ip_address_assignment>
     *           <ip>
     *             <address>192.168.122.10</address>
     *             <netmask>255.255.255.0</netmask>
     *           </ip>
     *         </ip_address_assignment>
     *       </ip_address_assignments>
     *     </network_attachment>
     *   </modified_network_attachments>
     *  </action>
     * ' \
     * "${url}/hosts/1ff7a191-2f3b-4eff-812b-9f91a30c3acc/setupnetworks"
     * ----
     * 
     * Note that this is valid for version 4 of the API. In previous versions some elements were represented as XML
     * attributes instead of XML elements. In particular the `options` and `ip` elements were represented as follows:
     * 
     * [source,xml]
     * ----
     * <options name="mode" value="4"/>
     * <options name="miimon" value="100"/>
     * <ip address="192.168.122.10" netmask="255.255.255.0"/>
     * ----
     * 
     * Using the Python SDK the same can be done with the following code:
     * 
     * [source,python]
     * ----
     * host.setupnetworks(
     *   params.Action(
     *     modified_bonds=params.HostNics(
     *       host_nic=[
     *         params.HostNIC(
     *           name="bond0",
     *           bonding=params.Bonding(
     *             options=params.Options(
     *               option=[
     *                 params.Option(name="mode", value="4"),
     *                 params.Option(name="miimon", value="100"),
     *               ],
     *             ),
     *             slaves=params.Slaves(
     *               host_nic=[
     *                 params.HostNIC(name="eth1"),
     *                 params.HostNIC(name="eth2"),
     *               ],
     *             ),
     *           ),
     *         ),
     *       ],
     *     ),
     *     modified_network_attachments=params.NetworkAttachments(
     *       network_attachment=[
     *         params.NetworkAttachment(
     *           network=params.Network(name="myvlan"),
     *           host_nic=params.HostNIC(name="bond0"),
     *           ip_address_assignments=params.IpAddressAssignments(
     *             ip_address_assignment=[
     *               params.IpAddressAssignment(
     *                 assignment_method="static",
     *                 ip=params.IP(
     *                   address="192.168.122.10",
     *                   netmask="255.255.255.0",
     *                 ),
     *               ),
     *             ],
     *           ),
     *         ),
     *       ],
     *     ),
     *   ),
     * )
     * ----
     * 
     * IMPORTANT: To make sure that the network configuration has been saved in the host, and that it will be applied
     * when the host is rebooted, remember to call <<services/host/methods/commitnetconfig, commitnetconfig>>.
     */
    public interface SetupNetworksResponse extends Response {
    }
    
    /**
     * This method is used to change the configuration of the network interfaces of a host.
     * 
     * For example, lets assume that you have a host with three network interfaces `eth0`, `eth1` and `eth2` and that
     * you want to configure a new bond using `eth0` and `eth1`, and put a VLAN on top of it. Using a simple shell
     * script and the `curl` command line HTTP client that can be done as follows:
     * 
     * [source]
     * ----
     * #!/bin/sh -ex
     * 
     * url="https://engine.example.com/ovirt-engine/api"
     * user="admin@internal"
     * password="..."
     * 
     * curl \
     * --verbose \
     * --cacert /etc/pki/ovirt-engine/ca.pem \
     * --user "${user}:${password}" \
     * --request POST \
     * --header "Version: 4" \
     * --header "Content-Type: application/xml" \
     * --header "Accept: application/xml" \
     * --data '
     * <action>
     *   <modified_bonds>
     *     <host_nic>
     *       <name>bond0</name>
     *       <bonding>
     *         <options>
     *           <option>
     *             <name>mode</name>
     *             <value>4</value>
     *           </option>
     *           <option>
     *             <name>miimon</name>
     *             <value>100</value>
     *           </option>
     *         </options>
     *         <slaves>
     *           <host_nic>
     *             <name>eth1</name>
     *           </host_nic>
     *           <host_nic>
     *             <name>eth2</name>
     *           </host_nic>
     *         </slaves>
     *       </bonding>
     *     </host_nic>
     *   </modified_bonds>
     *   <modified_network_attachments>
     *     <network_attachment>
     *       <network>
     *         <name>myvlan</name>
     *       </network>
     *       <host_nic>
     *         <name>bond0</name>
     *       </host_nic>
     *       <ip_address_assignments>
     *         <assignment_method>static</assignment_method>
     *         <ip_address_assignment>
     *           <ip>
     *             <address>192.168.122.10</address>
     *             <netmask>255.255.255.0</netmask>
     *           </ip>
     *         </ip_address_assignment>
     *       </ip_address_assignments>
     *     </network_attachment>
     *   </modified_network_attachments>
     *  </action>
     * ' \
     * "${url}/hosts/1ff7a191-2f3b-4eff-812b-9f91a30c3acc/setupnetworks"
     * ----
     * 
     * Note that this is valid for version 4 of the API. In previous versions some elements were represented as XML
     * attributes instead of XML elements. In particular the `options` and `ip` elements were represented as follows:
     * 
     * [source,xml]
     * ----
     * <options name="mode" value="4"/>
     * <options name="miimon" value="100"/>
     * <ip address="192.168.122.10" netmask="255.255.255.0"/>
     * ----
     * 
     * Using the Python SDK the same can be done with the following code:
     * 
     * [source,python]
     * ----
     * host.setupnetworks(
     *   params.Action(
     *     modified_bonds=params.HostNics(
     *       host_nic=[
     *         params.HostNIC(
     *           name="bond0",
     *           bonding=params.Bonding(
     *             options=params.Options(
     *               option=[
     *                 params.Option(name="mode", value="4"),
     *                 params.Option(name="miimon", value="100"),
     *               ],
     *             ),
     *             slaves=params.Slaves(
     *               host_nic=[
     *                 params.HostNIC(name="eth1"),
     *                 params.HostNIC(name="eth2"),
     *               ],
     *             ),
     *           ),
     *         ),
     *       ],
     *     ),
     *     modified_network_attachments=params.NetworkAttachments(
     *       network_attachment=[
     *         params.NetworkAttachment(
     *           network=params.Network(name="myvlan"),
     *           host_nic=params.HostNIC(name="bond0"),
     *           ip_address_assignments=params.IpAddressAssignments(
     *             ip_address_assignment=[
     *               params.IpAddressAssignment(
     *                 assignment_method="static",
     *                 ip=params.IP(
     *                   address="192.168.122.10",
     *                   netmask="255.255.255.0",
     *                 ),
     *               ),
     *             ],
     *           ),
     *         ),
     *       ],
     *     ),
     *   ),
     * )
     * ----
     * 
     * IMPORTANT: To make sure that the network configuration has been saved in the host, and that it will be applied
     * when the host is rebooted, remember to call <<services/host/methods/commitnetconfig, commitnetconfig>>.
     */
    SetupNetworksRequest setupNetworks();
    
    public interface UnregisteredStorageDomainsDiscoverRequest extends Request<UnregisteredStorageDomainsDiscoverRequest, UnregisteredStorageDomainsDiscoverResponse> {
        /**
         * Indicates if the discovery should be performed asynchronously.
         */
        UnregisteredStorageDomainsDiscoverRequest async(Boolean async);
        UnregisteredStorageDomainsDiscoverRequest iscsi(IscsiDetails iscsi);
        UnregisteredStorageDomainsDiscoverRequest iscsi(IscsiDetailsBuilder iscsi);
    }
    
    public interface UnregisteredStorageDomainsDiscoverResponse extends Response {
        List<StorageDomain> storageDomains();
    }
    
    UnregisteredStorageDomainsDiscoverRequest unregisteredStorageDomainsDiscover();
    
    public interface UpdateRequest extends Request<UpdateRequest, UpdateResponse> {
        /**
         * Indicates if the update should be performed asynchronously.
         */
        UpdateRequest async(Boolean async);
        UpdateRequest host(Host host);
        UpdateRequest host(HostBuilder host);
    }
    
    public interface UpdateResponse extends Response {
        Host host();
    }
    
    UpdateRequest update();
    
    public interface UpgradeRequest extends Request<UpgradeRequest, UpgradeResponse> {
        /**
         * Indicates if the upgrade should be performed asynchronously.
         */
        UpgradeRequest async(Boolean async);
    }
    
    public interface UpgradeResponse extends Response {
    }
    
    UpgradeRequest upgrade();
    
    /**
     * List of scheduling labels assigned to this host.
     */
    AssignedAffinityLabelsService affinityLabelsService();
    HostDevicesService devicesService();
    FenceAgentsService fenceAgentsService();
    HostHooksService hooksService();
    KatelloErrataService katelloErrataService();
    NetworkAttachmentsService networkAttachmentsService();
    HostNicsService nicsService();
    HostNumaNodesService numaNodesService();
    AssignedPermissionsService permissionsService();
    HostStorageService storageService();
    StorageServerConnectionExtensionsService storageConnectionExtensionsService();
    AssignedTagsService tagsService();
    UnmanagedNetworksService unmanagedNetworksService();
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

