/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.services;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URISyntaxException;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.util.EntityUtils;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.api.metamodel.runtime.xml.XmlWriter;
import org.ovirt.engine.sdk4.Error;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.ActionBuilder;
import org.ovirt.engine.sdk4.internal.HttpConnection;
import org.ovirt.engine.sdk4.internal.xml.XmlActionWriter;
import org.ovirt.engine.sdk4.internal.xml.XmlJobReader;
import org.ovirt.engine.sdk4.services.JobService;
import org.ovirt.engine.sdk4.services.StepsService;
import org.ovirt.engine.sdk4.types.Action;
import org.ovirt.engine.sdk4.types.Job;

public class JobServiceImpl extends ServiceImpl implements JobService {
    
    public JobServiceImpl(HttpConnection connection, String path) {
        super(connection, path);
    }
    
    private class ClearRequestImpl implements ClearRequest {
        private Boolean async;
        public ClearRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        public ClearResponse send() {
            HttpPost request = new HttpPost(getConnection().getUrl() + getPath() + "/clear");
            try (
              ByteArrayOutputStream output = new ByteArrayOutputStream();
              XmlWriter xmlWriter = new XmlWriter(output, true)
            ) {
                ActionBuilder action = new ActionBuilder();
                action.async(async);
                XmlActionWriter.writeOne(action.build(), xmlWriter);
                xmlWriter.flush();
                request.setEntity(new ByteArrayEntity(output.toByteArray()));
            }
            catch (IOException ex) {
                throw new Error("Failed to write request", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (response.getStatusLine().getStatusCode() == 200) {
                Action action = checkAction(response);
                EntityUtils.consumeQuietly(response.getEntity());
                return new ClearResponseImpl();
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class ClearResponseImpl implements ClearResponse {
    }
    
    public ClearRequest clear() {
        return new ClearRequestImpl();
    }
    
    private class EndRequestImpl implements EndRequest {
        private Boolean async;
        public EndRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        private Boolean force;
        public EndRequest force(Boolean force) {
            this.force = force;
            return this;
        }
        
        public EndResponse send() {
            HttpPost request = new HttpPost(getConnection().getUrl() + getPath() + "/end");
            try (
              ByteArrayOutputStream output = new ByteArrayOutputStream();
              XmlWriter xmlWriter = new XmlWriter(output, true)
            ) {
                ActionBuilder action = new ActionBuilder();
                action.async(async);
                action.force(force);
                XmlActionWriter.writeOne(action.build(), xmlWriter);
                xmlWriter.flush();
                request.setEntity(new ByteArrayEntity(output.toByteArray()));
            }
            catch (IOException ex) {
                throw new Error("Failed to write request", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (response.getStatusLine().getStatusCode() == 200) {
                Action action = checkAction(response);
                EntityUtils.consumeQuietly(response.getEntity());
                return new EndResponseImpl();
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class EndResponseImpl implements EndResponse {
    }
    
    public EndRequest end() {
        return new EndRequestImpl();
    }
    
    private class GetRequestImpl implements GetRequest {
        public GetResponse send() {
            HttpGet request = null;
            try {
                URIBuilder uriBuilder = new URIBuilder(getConnection().getUrl() + getPath());
                request = new HttpGet(uriBuilder.build());
            }
            catch (URISyntaxException ex) {
                throw new Error("Failed to build URL", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (
              response.getStatusLine().getStatusCode() == 200
            ) {
                try (
                  XmlReader reader = new XmlReader(response.getEntity().getContent())
                ) {
                    return new GetResponseImpl(XmlJobReader.readOne(reader));
                }
                catch (IOException ex) {
                    throw new Error("Failed to read response", ex);
                }
                finally {
                    EntityUtils.consumeQuietly(response.getEntity());
                }
            }
            else {
                checkFault(response);
                return new GetResponseImpl(null);
            }
        }
        
    }
    
    private class GetResponseImpl implements GetResponse {
        private Job job;
        
        public GetResponseImpl(Job job) {
            this.job = job;
        }
        
        public Job job() {
            return job;
        }
        
    }
    
    public GetRequest get() {
        return new GetRequestImpl();
    }
    
    public StepsService stepsService() {
        return new StepsServiceImpl(getConnection(), getPath() + "/steps");
    }
    
    public Service service(String path) {
        if (path == null) {
            return this;
        }
        
        if (path.equals("steps")) {
            return stepsService();
        }
        if (path.startsWith("steps/")) {
            return stepsService().service(path.substring(6));
        }
        throw new Error("The path " + path + " doesn't correspond to any service");
    }
    
    public String toString() {
        return "JobService:" + getPath();
    }
    
}

