/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.DiskBuilder;
import org.ovirt.engine.sdk4.types.Disk;

public interface DisksService extends Service {
    /**
     * Adds a new floating disk.
     * When creating a new floating <<types/disk,Disk>>, the API requires the `storage_domain`, `provisioned_size` and
     * `format` attributes.
     * 
     * To create a new floating disk with specified `provisioned_size`, `format` and `name` on a storage domain with an
     * id `e9fedf39-5edc-4e0a-8628-253f1b9c5693`, send a request as follows:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/disks
     * ----
     * 
     * With a request body as follows:
     * 
     * [source,xml]
     * ----
     * <disk>
     *   <storage_domains>
     *     <storage_domain id="e9fedf39-5edc-4e0a-8628-253f1b9c5693"/>
     *   </storage_domains>
     *   <name>disk1</name>
     *   <provisioned_size>1048576</provisioned_size>
     *   <format>cow</format>
     * </disk>
     * ----
     */
    public interface AddRequest extends Request<AddRequest, AddResponse> {
        AddRequest disk(Disk disk);
        AddRequest disk(DiskBuilder disk);
    }
    
    /**
     * Adds a new floating disk.
     * When creating a new floating <<types/disk,Disk>>, the API requires the `storage_domain`, `provisioned_size` and
     * `format` attributes.
     * 
     * To create a new floating disk with specified `provisioned_size`, `format` and `name` on a storage domain with an
     * id `e9fedf39-5edc-4e0a-8628-253f1b9c5693`, send a request as follows:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/disks
     * ----
     * 
     * With a request body as follows:
     * 
     * [source,xml]
     * ----
     * <disk>
     *   <storage_domains>
     *     <storage_domain id="e9fedf39-5edc-4e0a-8628-253f1b9c5693"/>
     *   </storage_domains>
     *   <name>disk1</name>
     *   <provisioned_size>1048576</provisioned_size>
     *   <format>cow</format>
     * </disk>
     * ----
     */
    public interface AddResponse extends Response {
        Disk disk();
    }
    
    /**
     * Adds a new floating disk.
     * When creating a new floating <<types/disk,Disk>>, the API requires the `storage_domain`, `provisioned_size` and
     * `format` attributes.
     * 
     * To create a new floating disk with specified `provisioned_size`, `format` and `name` on a storage domain with an
     * id `e9fedf39-5edc-4e0a-8628-253f1b9c5693`, send a request as follows:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/disks
     * ----
     * 
     * With a request body as follows:
     * 
     * [source,xml]
     * ----
     * <disk>
     *   <storage_domains>
     *     <storage_domain id="e9fedf39-5edc-4e0a-8628-253f1b9c5693"/>
     *   </storage_domains>
     *   <name>disk1</name>
     *   <provisioned_size>1048576</provisioned_size>
     *   <format>cow</format>
     * </disk>
     * ----
     */
    AddRequest add();
    
    public interface ListRequest extends Request<ListRequest, ListResponse> {
        /**
         * Indicates if the search performed using the `search` parameter should be performed taking case into
         * account. The default value is `true`, which means that case is taken into account. If you want to search
         * ignoring case set it to `false`.
         */
        ListRequest caseSensitive(Boolean caseSensitive);
        /**
         * Sets the maximum number of disks to return. If not specified all the disks are returned.
         */
        ListRequest max(Integer max);
        /**
         * Sets the maximum number of disks to return. If not specified all the disks are returned.
         */
        ListRequest max(Long max);
        /**
         * Sets the maximum number of disks to return. If not specified all the disks are returned.
         */
        ListRequest max(BigInteger max);
        /**
         * A query string used to restrict the returned disks.
         */
        ListRequest search(String search);
    }
    
    public interface ListResponse extends Response {
        List<Disk> disks();
    }
    
    ListRequest list();
    
    DiskService diskService(String id);
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

