package org.ovirt.engine.ui.common.widget.uicommon.disks;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.ovirt.engine.core.common.businessentities.storage.Disk;
import org.ovirt.engine.core.common.businessentities.storage.DiskImage;
import org.ovirt.engine.core.common.businessentities.storage.DiskInterface;
import org.ovirt.engine.core.common.businessentities.storage.DiskStorageType;
import org.ovirt.engine.core.common.businessentities.storage.ImageStatus;
import org.ovirt.engine.core.common.businessentities.storage.LunDisk;
import org.ovirt.engine.core.common.businessentities.storage.StorageType;
import org.ovirt.engine.core.common.businessentities.storage.VolumeType;
import org.ovirt.engine.core.common.utils.SizeConverter;
import org.ovirt.engine.core.compat.Guid;
import org.ovirt.engine.core.compat.StringFormat;
import org.ovirt.engine.core.compat.StringHelper;
import org.ovirt.engine.ui.common.CommonApplicationConstants;
import org.ovirt.engine.ui.common.CommonApplicationMessages;
import org.ovirt.engine.ui.common.CommonApplicationResources;
import org.ovirt.engine.ui.common.gin.AssetProvider;
import org.ovirt.engine.ui.common.widget.table.cell.Cell;
import org.ovirt.engine.ui.common.widget.table.cell.StatusCompositeCell;
import org.ovirt.engine.ui.common.widget.table.column.AbstractColumn;
import org.ovirt.engine.ui.common.widget.table.column.AbstractDiskSizeColumn;
import org.ovirt.engine.ui.common.widget.table.column.AbstractEnumColumn;
import org.ovirt.engine.ui.common.widget.table.column.AbstractFullDateTimeColumn;
import org.ovirt.engine.ui.common.widget.table.column.AbstractImageResourceColumn;
import org.ovirt.engine.ui.common.widget.table.column.AbstractTextColumn;
import org.ovirt.engine.ui.common.widget.table.column.DiskContainersColumn;
import org.ovirt.engine.ui.common.widget.table.column.DiskStatusColumn;
import org.ovirt.engine.ui.common.widget.table.column.DiskUploadImageProgressColumn;
import org.ovirt.engine.ui.common.widget.table.column.StorageDomainsColumn;
import org.ovirt.engine.ui.uicompat.EnumTranslator;

import com.google.gwt.cell.client.HasCell;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.resources.client.ImageResource;
import com.google.gwt.safehtml.shared.SafeHtml;
import com.google.gwt.safehtml.shared.SafeHtmlUtils;

public class DisksViewColumns {

    private static final CommonApplicationResources resources = AssetProvider.getResources();
    private static final CommonApplicationConstants constants = AssetProvider.getConstants();
    private static final CommonApplicationMessages messages = AssetProvider.getMessages();

    public static AbstractTextColumn<Disk> getAliasColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractTextColumn<Disk>() {
            @Override
            public String getValue(Disk object) {
                return object.getDiskAlias();
            }
        };

        return makeSortable(column, sortBy);
    }

    public static AbstractTextColumn<Disk> getIdColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractTextColumn<Disk>() {
            @Override
            public String getValue(Disk object) {
                return object.getId().toString();
            }
        };

        return makeSortable(column, sortBy);
    }

    public static AbstractTextColumn<Disk> getQoutaColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractTextColumn<Disk>() {
            @Override
            public String getValue(Disk object) {
                String value = "";
                if (object.getDiskStorageType() == DiskStorageType.IMAGE) {
                    DiskImage diskImage = (DiskImage) object;
                    ArrayList<String> quotaNamesArr = diskImage.getQuotaNames();
                    if (quotaNamesArr != null) {
                        value = StringHelper.join(", ", quotaNamesArr.toArray());//$NON-NLS-1$
                    }
                }
                return value;
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractImageResourceColumn<Disk> bootableDiskColumn = new AbstractImageResourceColumn<Disk>() {
        {
            setContextMenuTitle(constants.bootableDisk());
        }

        @Override
        public ImageResource getValue(Disk object) {
            if (object.getDiskVmElements().size() == 1) {
                if (object.getDiskVmElements().iterator().next().isBoot()) {
                    return getDefaultImage();
                }
            }
            return null;
        }

        @Override
        public ImageResource getDefaultImage() {
            return resources.bootableDiskIcon();
        }

        @Override
        public SafeHtml getTooltip(Disk object) {
            if (object.getDiskVmElements().size() == 1) {
                if (object.getDiskVmElements().iterator().next().isBoot()) {
                    return SafeHtmlUtils.fromSafeConstant(constants.bootableDisk());
                }
            }
            return null;
        }
    };

    public static final AbstractImageResourceColumn<Disk> shareableDiskColumn = new AbstractImageResourceColumn<Disk>() {
        {
            setContextMenuTitle(constants.shareable());
        }

        @Override
        public ImageResource getValue(Disk object) {
            return object.isShareable() ? getDefaultImage() : null;
        }

        @Override
        public ImageResource getDefaultImage() {
            return resources.shareableDiskIcon();
        }

        @Override
        public SafeHtml getTooltip(Disk object) {
            if (object.isShareable()) {
                return SafeHtmlUtils.fromSafeConstant(constants.shareable());
            }
            return null;
        }
    };

    public static final AbstractImageResourceColumn<Disk> readOnlyDiskColumn = new AbstractImageResourceColumn<Disk>() {
        {
            setContextMenuTitle(constants.readOnly());
        }

        @Override
        public ImageResource getValue(Disk object) {
            return object.getReadOnly() ? getDefaultImage() : null;
        }

        @Override
        public ImageResource getDefaultImage() {
            return resources.readOnlyDiskIcon();
        }

        @Override
        public SafeHtml getTooltip(Disk object) {
            if (object.getReadOnly()) {
                return SafeHtmlUtils.fromSafeConstant(constants.readOnly());
            }
            return null;
        }
    };

    public static final AbstractImageResourceColumn<Disk> diskContainersIconColumn = new AbstractImageResourceColumn<Disk>() {
        {
            setContextMenuTitle(constants.containersIconDisk());
        }

        @Override
        public ImageResource getValue(Disk object) {
            if (object.getVmEntityType() == null) {
                return null;
            }
            return object.getVmEntityType().isVmType() ? resources.vmsImage() :
                    object.getVmEntityType().isTemplateType() ? resources.templatesImage() : null;
        }

        @Override
        public SafeHtml getTooltip(Disk object) {
            if (object.getVmEntityType() == null) {
                return SafeHtmlUtils.fromSafeConstant(constants.unattachedDisk());
            }
            else {
                String status = EnumTranslator.getInstance().translate(object.getVmEntityType());
                return SafeHtmlUtils.fromString(status);
            }
        }
    };

    public static final DiskStatusColumn diskStatusColumn = new DiskStatusColumn() {
        {
            setContextMenuTitle(constants.statusDisk());
        }
    };

    public static final DiskContainersColumn diskContainersColumn = new DiskContainersColumn();

    public static final AbstractTextColumn<Disk> diskAlignmentColumn = new AbstractTextColumn<Disk>() {
        @Override
        public String getValue(Disk object) {
            return object.getAlignment().toString();
        }

        @Override
        public SafeHtml getTooltip(Disk object) {
            if (object.getLastAlignmentScan() != null) {
                String lastScanDate = DateTimeFormat
                        .getFormat("yyyy-MM-dd, HH:mm").format(object.getLastAlignmentScan()); //$NON-NLS-1$
                return SafeHtmlUtils.fromString(messages.lastDiskAlignment(lastScanDate));
            }
            return null;
        }
    };

    public static final StorageDomainsColumn storageDomainsColumn = new StorageDomainsColumn();

    public static final AbstractTextColumn<Disk> storageTypeColumn = new AbstractEnumColumn<Disk, StorageType>() {
        @Override
        protected StorageType getRawValue(Disk object) {
            if (object.getDiskStorageType() != DiskStorageType.IMAGE &&
                    object.getDiskStorageType() != DiskStorageType.CINDER) {
                return null;
            }
            DiskImage disk = (DiskImage) object;

            return disk.getStorageTypes().isEmpty() ? null : disk.getStorageTypes().get(0);
        }
    };

    public static final AbstractDiskSizeColumn<Disk> getSizeColumn(String sortBy) {
        AbstractDiskSizeColumn<Disk> column = new AbstractDiskSizeColumn<Disk>() {
            @Override
            protected Long getRawValue(Disk object) {
                switch (object.getDiskStorageType()) {
                    case LUN:
                        return (long) (((LunDisk) object).getLun().getDeviceSize() * Math.pow(1024, 3));
                    default:
                        return object.getSize();
                }
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractDiskSizeColumn<Disk> getActualSizeColumn(String sortBy) {
        AbstractDiskSizeColumn<Disk> column = new AbstractDiskSizeColumn<Disk>(SizeConverter.SizeUnit.GiB) {
            @Override
            protected Long getRawValue(Disk object) {
                return object.getDiskStorageType() == DiskStorageType.IMAGE ||
                        object.getDiskStorageType() == DiskStorageType.CINDER ?
                        Math.round(((DiskImage) object).getActualDiskWithSnapshotsSize())
                        : (long) ((LunDisk) object).getLun().getDeviceSize();
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractTextColumn<Disk> getAllocationColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractEnumColumn<Disk, VolumeType>() {
            @Override
            protected VolumeType getRawValue(Disk object) {
                return object.getDiskStorageType() == DiskStorageType.IMAGE ||
                        object.getDiskStorageType() == DiskStorageType.CINDER ?
                        ((DiskImage) object).getVolumeType() : null;
            }

            @Override
            public SafeHtml getTooltip(Disk object) {
                if (object.getDiskStorageType() != DiskStorageType.IMAGE &&
                        object.getDiskStorageType() != DiskStorageType.CINDER) {
                    return null;
                }

                VolumeType originalVolumeType = null;
                for (DiskImage snapshot : ((DiskImage) object).getSnapshots()) {
                    if (snapshot.getParentId() == null || snapshot.getParentId().equals(Guid.Empty)) {
                        originalVolumeType = snapshot.getVolumeType();
                        break;
                    }
                }

                if (originalVolumeType == null) {
                    return null;
                }

                return SafeHtmlUtils.fromString(
                        StringFormat.format("%s: %s",  //$NON-NLS-1$
                                AssetProvider.getConstants().originalAllocationDisk(),
                                EnumTranslator.getInstance().translate(originalVolumeType)));
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractTextColumn<Disk> getInterfaceColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractEnumColumn<Disk, DiskInterface>() {
            @Override
            protected DiskInterface getRawValue(Disk object) {
                if (object.getDiskVmElements().size() == 1) {
                    return object.getDiskVmElements().iterator().next().getDiskInterface();
                }
                return null;
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractTextColumn<Disk> getDateCreatedColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractFullDateTimeColumn<Disk>() {
            @Override
            protected Date getRawValue(Disk object) {
                return object.getDiskStorageType() == DiskStorageType.IMAGE ||
                        object.getDiskStorageType() == DiskStorageType.CINDER ?
                        ((DiskImage) object).getCreationDate() : null;
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractColumn<Disk, Disk> getStatusColumn(String sortBy) {
        AbstractTextColumn<Disk> statusColumn = new AbstractEnumColumn<Disk, ImageStatus>() {
            @Override
            protected ImageStatus getRawValue(Disk object) {
                return object.getDiskStorageType() == DiskStorageType.IMAGE ||
                        object.getDiskStorageType() == DiskStorageType.CINDER ?
                        ((DiskImage) object).getImageStatus() : null;
            }

            @Override
            public String getValue(Disk object) {
                if (object.getImageTransferPhase() != null) {
                    // will be rendered by progress column
                    return null;
                }
                return super.getValue(object);
            }
        };

        DiskUploadImageProgressColumn uploadImageProgressColumn = new DiskUploadImageProgressColumn();

        List<HasCell<Disk, ?>> list = new ArrayList<>();
        list.add(statusColumn);
        list.add(uploadImageProgressColumn);

        Cell<Disk> compositeCell = new StatusCompositeCell<>(list);

        AbstractColumn<Disk, Disk> column = new AbstractColumn<Disk, Disk>(compositeCell) {
            @Override
            public Disk getValue(Disk object) {
                return object;
            }
        };

        if (sortBy != null) {
            column.makeSortable(sortBy);
        }
        return column;
    }

    public static final AbstractTextColumn<Disk> getDescriptionColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractTextColumn<Disk>() {
            @Override
            public String getValue(Disk object) {
                return object.getDiskDescription();
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractTextColumn<Disk> getLunIdColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractTextColumn<Disk>() {
            @Override
            public String getValue(Disk object) {
                return object.getDiskStorageType() == DiskStorageType.LUN ?
                        ((LunDisk) object).getLun().getLUNId() : null;
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractTextColumn<Disk> getLunVendorIdColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractTextColumn<Disk>() {
            @Override
            public String getValue(Disk object) {
                return object.getDiskStorageType() == DiskStorageType.LUN ?
                        ((LunDisk) object).getLun().getVendorId() : null;
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractTextColumn<Disk> getLunProductIdColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractTextColumn<Disk>() {
            @Override
            public String getValue(Disk object) {
                return object.getDiskStorageType() == DiskStorageType.LUN ?
                        ((LunDisk) object).getLun().getProductId() : null;
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractTextColumn<Disk> getLunSerialColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractTextColumn<Disk>() {
            @Override
            public String getValue(Disk object) {
                return object.getDiskStorageType() == DiskStorageType.LUN ?
                        ((LunDisk) object).getLun().getSerial() : null;
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractDiskSizeColumn<Disk> getSnapshotSizeColumn(String sortBy) {
        AbstractDiskSizeColumn<Disk> column = new AbstractDiskSizeColumn<Disk>() {
            @Override
            protected Long getRawValue(Disk object) {
                return ((DiskImage) object).getActualSizeInBytes();
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractTextColumn<Disk> getSnapshotDescriptionColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractTextColumn<Disk>() {
            @Override
            public String getValue(Disk object) {
                return ((DiskImage) object).getVmSnapshotDescription();
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractTextColumn<Disk> getDiskSnapshotIDColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractTextColumn<Disk>() {
            @Override
            public String getValue(Disk disk) {
                return disk.getDiskStorageType() == DiskStorageType.IMAGE ||
                        disk.getDiskStorageType() == DiskStorageType.CINDER
                                ? ((DiskImage) disk).getImageId().toString() : null;
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractTextColumn<Disk> getDiskStorageTypeColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractEnumColumn<Disk, DiskStorageType>() {
            @Override
            protected DiskStorageType getRawValue(Disk object) {
                return object.getDiskStorageType();
            }
        };

        return makeSortable(column, sortBy);
    }

    public static final AbstractTextColumn<Disk> getCinderVolumeTypeColumn(String sortBy) {
        AbstractTextColumn<Disk> column = new AbstractTextColumn<Disk>() {
            @Override
            public String getValue(Disk object) {
                return object.getCinderVolumeType();
            }
        };

        return makeSortable(column, sortBy);
    }

    public static <C extends AbstractTextColumn<T>, T> C makeSortable(C column, String sortBy) {
        if (sortBy == null ) {
            // Client sorting
            column.makeSortable();
        }
        else if (!sortBy.equals(constants.empty())) {
            // Server sorting
            column.makeSortable(sortBy);
        }

        return column;
    }
}
