#
# Copyright 2015 Hat, Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
#
# Refer to the README and COPYING files for full details of the license
from __future__ import absolute_import
from __future__ import division

from collections import OrderedDict
import logging

# TODO remove dependency on build code
from vdsm.common import dsaversion
from vdsm.common import fileutils

DNS_CONF_FILE = '/etc/resolv.conf'
CONFFILE_HEADER_SIGNATURE = '# Generated by VDSM version'
DNS_NAMESERVER = 'nameserver'
MAX_DNS_PER_HOST = 3


def get_host_nameservers():
    """Returns a list of nameservers listed in /etc/resolv.conf"""
    try:
        with open(DNS_CONF_FILE, 'r') as file_object:
            file_text = file_object.read()
    except IOError as e:
        logging.warning('Failed to read {}: {}'.format(DNS_CONF_FILE, e))
        return []
    return _parse_nameservers(file_text)


def add_host_nameservers(servers):
    nameservers = servers + get_host_nameservers()
    nameservers = list(OrderedDict.fromkeys(nameservers).keys())
    _set_host_nameservers(nameservers[:MAX_DNS_PER_HOST])


def _set_host_nameservers(servers):
    with fileutils.atomic_file_write(DNS_CONF_FILE, 'w') as file_object:
        file_object.write(
            CONFFILE_HEADER_SIGNATURE
            + ' '
            + dsaversion.raw_version_revision
            + '\n'
        )
        nameservers = '\n'.join(
            ['{} {}'.format(DNS_NAMESERVER, server) for server in servers]
        )
        file_object.write(nameservers)


def _parse_nameservers(file_text):
    nameservers = []
    for line in file_text.splitlines():
        words = line.strip().split()
        if len(words) < 2:
            continue
        if words[0] == 'nameserver':
            nameservers.append(words[1])
    return nameservers
